<?php

namespace Onyx;

class Api
{

    private $methods_file = null;
    private $token;
    private $methods;
    private $returnJson = false;

    private $default_base_url = 'https://h030mt6fa5.execute-api.us-east-2.amazonaws.com/prod';
    private $base_url;
    private $_lastResponseHeaders = null;

    public function __construct($token, $base_url = null, $methods_file = null)
    {
        if ($base_url) {
            $this->base_url = $base_url;
        }
        if ($methods_file === null) {
            $methods_file = dirname(realpath(__FILE__)) . '/methods.json';
        }

        if (!file_exists($methods_file)) {
            exit("Onyx methods file '{$methods_file}' does not exist!");
        }
        $this->methods_file = $methods_file;

        $this->token = $token;
    }

    public function __call($method, $args)
    {
        try {
            $validArguments = RequestValidator::validateParams(@$args[0], $this->getMethods($method));
            if (isset($validArguments['error'])) {
                throw new \Exception('Invalid params for method "' . $method . '": ' . implode(', ', $validArguments['error']));
            }

            return call_user_func_array(array($this, 'request'), array(
                array(
                    'method' => $method,
                    'args' => array(
                        'data' => @$validArguments['data'],
                        'params' => @$args[0]['params'],
                    )
                )));

        } catch (\Exception $e) {
            $invalidData = [
                'last_response' => $e->getMessage(),
                'message' => $e->getMessage(),
                'http_code' => $e->getCode(),
                'trace' => $e->getTraceAsString(),
            ];
            return $invalidData;
        }
    }

    public function getMethods($method_name)
    {
        if ($method_name) {
            if (empty($this->methods)) {
                $this->methods = json_decode(file_get_contents($this->methods_file), true);
            }
            if (isset($this->methods[$method_name]))
                return $this->methods[$method_name];
        }
        throw new \Exception('Method "' . $method_name . '" not exists');
    }

    private function request($arguments)
    {
        $method = $this->getMethods($arguments['method']);
        $args = $arguments['args'];
        $params = $this->prepareParameters($args['params']);
        $data = @$this->prepareData($args['data']);
        $uri = preg_replace_callback('@:(.+?)(\/|$)@', function ($matches) use ($args, &$params) {
            unset($params[$matches[1]]);
            return $args["params"][$matches[1]] . $matches[2];
        }, $method['uri']);

        if (!$this->base_url) {
            if (isset($method['base_url']) && $method['base_url'] != '') {
                $this->base_url = $method['base_url'];
            } else {
                $this->base_url = $this->default_base_url;
            }
        }

        if (!empty($params)) {
            $uri .= "?" . http_build_query($params);
        }
        $this->setRequestUri($uri);
        $this->setHttpMethod($method['http_method']);


        return $this->validateResponse($args, $this->requestCall($uri, [], $data, $method['http_method']));
    }

    private function prepareParameters($params)
    {

        $query_pairs = array();
        if ($params) {
            foreach ($params as $key => $value) {
                $query_pairs[$key] = $value;
            }
        }
        return $query_pairs;
    }

    private function prepareData($data)
    {
        $result = array();
        foreach ($data as $key => $value) {
            $type = gettype($value);
            if ($type !== 'boolean') {
                $result[$key] = $value;
                continue;
            }

            $result[$key] = $value ? 1 : 0;
        }

        return $result;
    }

    private function prepareFields($fields)
    {

        return implode(',', $fields);
    }

    public function setRequestUri($uri)
    {
        $this->requestUri = $uri;
    }

    public function setHttpMethod($method)
    {
        $this->httpMethod = $method;
    }

    protected function validateResponse($request_args, $response)
    {
        return $response;
    }

    public function requestCall($uri, $headers, $params = array(), $method = 'GET')
    {

        $url = trim($this->base_url, '/') . "{$uri}";

        $headers = $this->getHeaders($headers);

        $ch = curl_init();

        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_ENCODING, '');
        curl_setopt($ch, CURLOPT_HEADER, 1);

        if (is_array($params)) {
            $params = json_encode($params);
        }

        if (strtolower($method) === 'post') {
            curl_setopt($ch, CURLOPT_POST, 1);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
        } elseif (strtolower($method) === 'put') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $params);
        } elseif (strtolower($method) === 'delete') {
            curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        }

        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

//        echo '<pre>';
//        print_r($url);
//        print_r($headers);
//        print_r($params);

        $curlResponse = curl_exec($ch);
        $curlError = curl_error($ch);

        $header_size = curl_getinfo($ch, CURLINFO_HEADER_SIZE);

        $response = $this->curlParseHeaders($curlResponse, $header_size);

        if ($response === false || $curlError) {
            curl_close($ch);

            return ['success' => false, 'message' => $curlError];
        } else {
            $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);

            curl_close($ch);

            if ($httpCode >= 200 && $httpCode <= 300) {
                return ['success' => true, 'data' => json_decode($response, true)];
            } else {
                return ['success' => false, 'message' => json_decode($response, true)];
            }
        }
    }

    private function getHeaders($headers = [])
    {

        $defaultHeader = [
            "Authorization : Bearer " . $this->token
        ];

        $headers = array_merge($defaultHeader, $headers);

        return $headers;
    }

    /**
     * @param $server_output
     * @param $header_size
     * @return bool|string
     */
    private function curlParseHeaders($server_output, $header_size)
    {
        $header = substr($server_output, 0, $header_size);
        $response = substr($server_output, $header_size);
        $this->_lastResponseHeaders = $header;

        return $response;
    }

    public function getLastResponseHeaders()
    {
        return $this->_lastResponseHeaders;
    }
}