<?php

namespace Amazon\Sdk\Product;

//@TODO add logging with PSR logger obj

/**
 * Class Category
 * @package Amazon\Sdk\Product
 * @method getSubAttributes
 * @method getAttributes
 */
abstract class Category extends \Amazon\Sdk\Base
{
    const KEY_DELIMITER = '_';
    const SEQUENCE_DELIMITER = "\\";
    const MULTIVALUE_DELIMITER = '||';

    const ERRORS = [
        0 => 'optional',
        1 => 'required',
    ];

    const BASE_ATTRIBUTES = [
        //@TODO: Add validation for SKU, space not allowed.
        'SKU' => [
            'sequence' => '10',
            'name' => 'SKU',
            'dataType' => "StringNotNull",
            'minOccurs' => '1',
        ],
        //@TODO: Add multiple identifiers.
        'StandardProductID_Type' => [
            'sequence' => '20\10',
            'name' => 'Barcode Type',
            'dataType' => "StringNotNull",
            'minOccurs' => '0',
            'depends' => [
                'StandardProductID_Value'
            ],
            "restriction" => [
                    "optionValues" => [
                            0 => "ISBN",
                            1 => "UPC",
                            2 => "EAN",
                            3 => "ASIN",
                            4 => "GTIN",
                            5 => "GCID",
                            6 => "PZN",
                        ],
                ],
            'default' => 'EAN'
        ],
        'StandardProductID_Value' => [
            'sequence' => '20\20',
            'name' => 'Barcode',
            'dataType' => "Barcode",
            'minOccurs' => '1',
            "length" => "8:16"
        ],
        'RelatedProductID_Type' => [
            'sequence' => '21\10',
            'name' => 'Related Product Id Type',
            'dataType' => "StringNotNull",
            'minOccurs' => '0',
            "restriction" => [
                    "optionValues" => [
                            0 => "UPC",
                            1 => "EAN",
                            2 => "GTIN",
                        ],
                ],
        ],
        'RelatedProductID_Value' => [
            'sequence' => '21\20',
            'name' => 'Related Product Id Type',
            'dataType' => "Barcode",
            'minOccurs' => '0',
            "length" => "8:16"
        ],
        'ProductTaxCode' => [
            'sequence' => '22',
            'name' => 'Product Tax Code',
            'dataType' => "StringNotNull",
            'minOccurs' => '0',
            "length" => "1:50"
        ],
        'LaunchDate' => [
            'sequence' => '23',
            'name' => 'LaunchDate',
            'dataType' => "dateTime",
            'minOccurs' => '0',
        ],
        'DiscontinueDate' => [
            'sequence' => '24',
            'name' => 'DiscontinueDate',
            'dataType' => "dateTime",
            'minOccurs' => '0',
        ],
        'ReleaseDate' => [
            'sequence' => '25',
            'name' => 'ReleaseDate',
            'dataType' => "dateTime",
            'minOccurs' => '0',
        ],
        'Condition_ConditionType' => [
            'sequence' => '26\10',
            'name' => 'Condition',
            'dataType' => "StringNotNull",
            'minOccurs' => '0',
            "restriction" => [
                "optionValues" => [
                        "New",
                        "UsedLikeNew",
                        "UsedVeryGood",
                        "UsedGood",
                        "UsedAcceptable",
                        "CollectibleLikeNew",
                        "CollectibleVeryGood",
                        "CollectibleGood",
                        "CollectibleAcceptable",
                        "Refurbished", // India Only
                        "Club",
                    ],
            ],
        ],
        'Condition_ConditionNote' => [
            'sequence' => '26\20',
            'name' => 'ConditionNote',
            'dataType' => "TwoThousandString",
            'minOccurs' => '0',
        ],
        'ItemPackageQuantity' => [
            'sequence' => '27',
            'name' => 'ItemPackageQuantity',
            'dataType' => "positiveInteger",
            'minOccurs' => '0',
        ],
        'NumberOfItems' => [
            'sequence' => '28',
            'name' => 'NumberOfItems',
            'dataType' => "positiveInteger",
            'minOccurs' => '0',
        ],
        'DescriptionData_Title' => [
            'sequence' => '30\10',
            'name' => 'Title',
            'dataType' => "StringNotNull",
            'minOccurs' => '1',
        ],
        'DescriptionData_Brand' => [
            'sequence' => '30\20',
            'name' => 'Brand',
            'dataType' => "StringNotNull",
            'minOccurs' => '1',
        ],
        'DescriptionData_Description' => [
            'sequence' => '30\30',
            'name' => 'Description',
            'dataType' => "richText",
            'length' => '1:2000',
            'minOccurs' => '1',
        ],
        'DescriptionData_BulletPoint' => [
            'sequence' => '30\40',
            'name' => 'BulletPoint',
            'dataType' => "LongStringNotNull",
            'minOccurs' => '0',
            'maxOccurs' => '5',
        ],
        'DescriptionData_ItemDimensions_Length' => [
            'sequence' => '30\50\10',
            'name' => 'Item Length',
            'dataType' => 'Dimension',
            'minOccurs' => '0',
            'attribute' => 'DescriptionData_ItemDimensions_Length_unitOfMeasure_LengthUnitOfMeasure'
        ],
        'DescriptionData_ItemDimensions_Width' => [
            'sequence' => '30\50\20',
            'name' => 'Item Width',
            'dataType' => 'Dimension',
            'minOccurs' => '0',
            'attribute' => 'DescriptionData_ItemDimensions_Width_unitOfMeasure_LengthUnitOfMeasure'
        ],
        'DescriptionData_ItemDimensions_Height' => [
            'sequence' => '30\50\30',
            'name' => 'Item Height',
            'dataType' => 'Dimension',
            'minOccurs' => '0',
            'attribute' => 'DescriptionData_ItemDimensions_Height_unitOfMeasure_LengthUnitOfMeasure'
        ],
        'DescriptionData_PackageDimensions_Length' => [
            'sequence' => '30\60\10',
            'name' => 'Package Length',
            'dataType' => 'Dimension',
            'minOccurs' => '0',
            'attribute' => 'DescriptionData_PackageDimensions_Length_unitOfMeasure_LengthUnitOfMeasure'
        ],
        'DescriptionData_PackageDimensions_Width' => [
            'sequence' => '30\60\20',
            'name' => 'Package Width',
            'dataType' => 'Dimension',
            'minOccurs' => '0',
            'attribute' => 'DescriptionData_PackageDimensions_Width_unitOfMeasure_LengthUnitOfMeasure'
        ],
        'DescriptionData_PackageDimensions_Height' => [
            'sequence' => '30\60\30',
            'name' => 'Package Height',
            'dataType' => 'Dimension',
            'minOccurs' => '0',
            'attribute' => 'DescriptionData_PackageDimensions_Height_unitOfMeasure_LengthUnitOfMeasure'
        ],
        'DescriptionData_PackageWeight' => [
            'sequence' => '30\65',
            'name' => 'PackageWeight',
            'dataType' => 'Dimension',
            'minOccurs' => '0',
            'attribute' => 'DescriptionData_PackageWeight_unitOfMeasure_WeightUnitOfMeasure'
        ],
        'DescriptionData_ShippingWeight' => [
            'sequence' => '30\70',
            'name' => 'ShippingWeight',
            'dataType' => 'PositiveDimension',
            'minOccurs' => '0',
            'attribute' => 'DescriptionData_ShippingWeight_unitOfMeasure_WeightUnitOfMeasure'
        ],
        'DescriptionData_MerchantCatalogNumber' => [
            'sequence' => '30\75',
            'name' => 'MerchantCatalogNumber',
            'dataType' => 'FortyStringNotNull',
            'minOccurs' => '0',
        ],
        'DescriptionData_MSRP' => [
            'sequence' => '30\76',
            'name' => 'MSRP',
            'dataType' => 'PositiveDimensionTwoDecimal',
            'minOccurs' => '0',
            'attribute' => 'MSRP_BaseCurrencyCode'
        ],
        'DescriptionData_MSRPWithTax' => [
            'sequence' => '30\77',
            'name' => 'MSRPWithTax',
            'dataType' => 'PositiveDimensionTwoDecimal',
            'minOccurs' => '0',
            'attribute' => 'MSRPWithTax_BaseCurrencyCode'
        ],
        'DescriptionData_MaxOrderQuantity' => [
            'sequence' => '30\80',
            'name' => 'MaxOrderQuantity',
            'dataType' => 'positiveInteger',
            'minOccurs' => '0',
        ],
        'DescriptionData_SerialNumberRequired' => [
            'sequence' => '30\85',
            'name' => 'SerialNumberRequired',
            'dataType' => 'boolean',
            'minOccurs' => '0',
        ],
        'DescriptionData_Prop65' => [
            'sequence' => '30\90',
            'name' => 'Prop65',
            'dataType' => 'boolean',
            'minOccurs' => '0',
        ],
        'DescriptionData_CPSIAWarning' => [
            'sequence' => '30\95',
            'name' => 'CPSIAWarning',
            'dataType' => 'StringNotNull',
            'minOccurs' => '0',
            'maxOccurs' => '4',
            "restriction" => [
                "optionValues" => [
                    "choking_hazard_balloon",
                    "choking_hazard_contains_a_marble",
                    "choking_hazard_contains_small_ball",
                    "choking_hazard_is_a_marble",
                    "choking_hazard_is_a_small_ball",
                    "choking_hazard_small_parts",
                    "no_warning_applicable",
                ],
            ],
        ],
        'DescriptionData_CPSIAWarningDescription' => [
            'sequence' => '30\105',
            'name' => 'CPSIAWarningDescription',
            'dataType' => 'TwoFiftyStringNotNull',
            'minOccurs' => '0',
        ],
        'DescriptionData_LegalDisclaimer' => [
            'sequence' => '30\110',
            'name' => 'LegalDisclaimer',
            'dataType' => 'normalizedString',
            'minOccurs' => '0',
            "restriction" => [
                "maxLength" => "2500"
            ]
        ],
        'DescriptionData_Manufacturer' => [
            'sequence' => '30\150',
            'name' => 'Manufacturer',
            'dataType' => 'FortyStringNotNull',
            'minOccurs' => '0',
        ],
        'DescriptionData_MfrPartNumber' => [
            'sequence' => '30\160',
            'name' => 'MfrPartNumber',
            'dataType' => 'FortyStringNotNull',
            'minOccurs' => '0',
        ],
        'DescriptionData_SearchTerms' => [
            'sequence' => '30\161',
            'name' => 'SearchTerms',
            'dataType' => 'StringNotNull',
            'minOccurs' => '0',
            'maxOccurs' => '5',
        ],
        'DescriptionData_PlatinumKeywords' => [
            'sequence' => '30\200',
            'name' => 'PlatinumKeywords',
            'dataType' => 'StringNotNull',
            'minOccurs' => '0',
            'maxOccurs' => '20',
        ],
        'DescriptionData_ItemType' => [
            'sequence' => '30\300',
            'name' => 'ItemType',
            'dataType' => 'LongStringNotNull',
            'minOccurs' => '0',
        ],
        'DescriptionData_OtherItemAttributes' => [
            'sequence' => '30\400',
            'name' => 'OtherItemAttributes',
            'dataType' => 'LongStringNotNull',
            'minOccurs' => '0',
            'maxOccurs' => '5'
        ],
        'DescriptionData_TargetAudience' => [
            'sequence' => '30\500',
            'name' => 'TargetAudience',
            'dataType' => 'StringNotNull',
            'minOccurs' => '0',
            'maxOccurs' => '4'
        ],
        'DescriptionData_SubjectContent' => [
            'sequence' => '30\600',
            'name' => 'SubjectContent',
            'dataType' => 'StringNotNull',
            'minOccurs' => '0',
            'maxOccurs' => '5'
        ],
        'DescriptionData_IsGiftWrapAvailable' => [
            'sequence' => '30\700',
            'name' => 'IsGiftWrapAvailable',
            'dataType' => 'boolean',
            'minOccurs' => '0',
        ],
        'DescriptionData_IsGiftMessageAvailable' => [
            'sequence' => '30\800',
            'name' => 'IsGiftMessageAvailable',
            'dataType' => 'boolean',
            'minOccurs' => '0',
        ],
        'DescriptionData_PromotionKeywords' => [
            'sequence' => '30\900',
            'name' => 'PromotionKeywords',
            'dataType' => 'StringNotNull',
            'minOccurs' => '0',
            'maxOccurs' => '10'
        ],
        'DescriptionData_IsDiscontinuedByManufacturer' => [
            'sequence' => '30\1000',
            'name' => 'IsDiscontinuedByManufacturer',
            'dataType' => 'boolean',
            'minOccurs' => '0',
        ],
        'DescriptionData_DeliveryScheduleGroupID' => [
            'sequence' => '30\1100',
            'name' => 'DeliveryScheduleGroupID',
            'dataType' => 'StringNotNull',
            'minOccurs' => '0',
        ],
        // TODO: verify: maxOccurs with optionValues
        'DescriptionData_DeliveryChannel' => [
            'sequence' => '30\1200',
            'name' => 'DeliveryChannel',
            'dataType' => 'string',
            'minOccurs' => '0',
            'maxOccurs' => '2',
            "restriction" => [
                "optionValues" => [
                    "in_store",
                    "direct_ship",
                ],
            ],
        ],
        'DescriptionData_ExternalProductInformation' => [
            'sequence' => '30\1300',
            'name' => 'ExternalProductInformation',
            'dataType' => 'StringNotNull',
            'minOccurs' => '0',
        ],
        'DescriptionData_MaxAggregateShipQuantity' => [
            'sequence' => '30\1400',
            'name' => 'MaxAggregateShipQuantity',
            'dataType' => 'positiveInteger',
            'minOccurs' => '0',
        ],
        'DescriptionData_RecommendedBrowseNode' => [
            'sequence' => '30\1500',
            'name' => 'RecommendedBrowseNode',
            'dataType' => 'StringNotNull',
            'minOccurs' => '0',
            'maxOccurs' => '2',
        ],
        'DescriptionData_MerchantShippingGroupName' => [
            'sequence' => '30\1600',
            'name' => 'MerchantShippingGroupName',
            'dataType' => 'LongString',
            'minOccurs' => '0',
        ],
        "DescriptionData_Battery_AreBatteriesIncluded" => [
            "sequence" => "30\\1700\\10",
            "name" => "AreBatteriesIncluded",
            "dataType" => "boolean",
            "minOccurs" => "0",
        ],
        "DescriptionData_Battery_AreBatteriesRequired" => [
            "sequence" => "30\\1700\\20",
            "name" => "AreBatteriesRequired",
            "dataType" => "boolean",
            "minOccurs" => "0",
        ],
        "DescriptionData_Battery_BatterySubgroup_BatteryType" => [
            "sequence" => "30\\1700\\30\\10",
            "name" => "BatteryType",
            "dataType" => "StringNotNull",
            "minOccurs" => "0",
            "restriction" => [
                    "optionValues" => [
                            0 => "battery_type_2/3A",
                            1 => "battery_type_4/3A",
                            2 => "battery_type_4/5A",
                            3 => "battery_type_9v",
                            4 => "battery_type_12v",
                            5 => "battery_type_a",
                            6 => "battery_type_a76",
                            7 => "battery_type_aa",
                            8 => "battery_type_aaa",
                            9 => "battery_type_aaaa",
                            10 => "battery_type_c",
                            11 => "battery_type_cr123a",
                            12 => "battery_type_cr2",
                            13 => "battery_type_cr5",
                            14 => "battery_type_d",
                            15 => "battery_type_lithium_ion",
                            16 => "battery_type_lithium_metal",
                            17 => "battery_type_L-SC",
                            18 => "battery_type_p76",
                            19 => "battery_type_product_specific",
                            20 => "battery_type_SC",
                        ],
                ],
        ],
        "DescriptionData_Battery_BatterySubgroup_NumberOfBatteries" => [
            "sequence" => "30\\1700\\30\\20",
            "name" => "NumberOfBatteries",
            "dataType" => "positiveInteger",
            "minOccurs" => "0",
        ],
        'DescriptionData_BatteryCellType' => [
            'sequence' => '30\1800',
            'name' => 'BatteryCellType',
            'dataType' => 'StringNotNull',
            'minOccurs' => '0',
            "restriction" => [
                "optionValues" => [
                    "NiCAD",
                    "NiMh",
                    "alkaline",
                    "aluminum_oxygen",
                    "lead_acid",
                    "lead_calcium",
                    "lithium",
                    "lithium_ion",
                    "lithium_manganese_dioxide",
                    "lithium_metal",
                    "lithium_polymer",
                    "manganese",
                    "polymer",
                    "silver_oxide",
                    "zinc",
                ],
            ],
        ],
        'DescriptionData_BatteryWeight' => [
            'sequence' => '30\1900',
            'name' => 'BatteryWeight',
            'dataType' => 'Dimension',
            'minOccurs' => '0',
            'attribute' => 'DescriptionData_BatteryWeight_Weight_unitOfMeasure_WeightUnitOfMeasure',
        ],
        'DescriptionData_NumberOfLithiumMetalCells' => [
            'sequence' => '30\2000',
            'name' => 'NumberOfLithiumMetalCells',
            'dataType' => 'positiveInteger',
            'minOccurs' => '0',
        ],
        'DescriptionData_NumberOfLithiumIonCells' => [
            'sequence' => '30\2100',
            'name' => 'NumberOfLithiumIonCells',
            'dataType' => 'positiveInteger',
            'minOccurs' => '0',
        ],
        'DescriptionData_LithiumBatteryPackaging' => [
            'sequence' => '30\2200',
            'name' => 'LithiumBatteryPackaging',
            'dataType' => 'StringNotNull',
            'minOccurs' => '0',
            "restriction" => [
                "optionValues" => [
                    "batteries_contained_in_equipment",
                    "batteries_only",
                    "batteries_packed_with_equipment",
                ],
            ],
        ],
        'DescriptionData_LithiumBatteryEnergyContent' => [
            'sequence' => '30\2300',
            'name' => 'LithiumBatteryEnergyContent',
            'dataType' => 'Dimension',
            'minOccurs' => '0',
            'attribute' => 'DescriptionData_LithiumBatteryEnergyContent_EnergyConsumption_unitOfMeasure_EnergyConsumptionUnitOfMeasure',
        ],
        'DescriptionData_LithiumBatteryWeight' => [
            'sequence' => '30\2400',
            'name' => 'LithiumBatteryWeight',
            'dataType' => 'Dimension',
            'minOccurs' => '0',
            'attribute' => 'DescriptionData_LithiumBatteryWeight_Weight_unitOfMeasure_WeightUnitOfMeasure',
        ],
        'DescriptionData_ItemWeight' => [
            'sequence' => '30\2500',
            'name' => 'ItemWeight',
            'dataType' => 'Dimension',
            'minOccurs' => '0',
            'attribute' => 'DescriptionData_ItemWeight_Weight_unitOfMeasure_WeightUnitOfMeasure',
        ],
        'DescriptionData_ItemVolume' => [
            'sequence' => '30\2600',
            'name' => 'ItemVolume',
            'dataType' => 'Dimension',
            'minOccurs' => '0',
            'attribute' => 'DescriptionData_ItemVolume_Volume_unitOfMeasure_LengthUnitOfMeasure',
        ],
        'DescriptionData_FlashPoint' => [
            'sequence' => '30\2700',
            'name' => 'FlashPoint',
            'dataType' => 'StringNotNull',
            'minOccurs' => '0',
        ],
        'DescriptionData_GHSClassificationClass' => [
            'sequence' => '30\2800',
            'name' => 'GHSClassificationClass',
            'dataType' => 'StringNotNull',
            'minOccurs' => '0',
            "restriction" => [
                "optionValues" => [
                    "explosive",
                    "oxidizing",
                    "toxic",
                    "corrosive",
                    "amzn_specific_no_label_with_warning",
                    "flammable",
                    "irritant",
                    "health_hazard",
                    "environmentally_damaging",
                    "compressed_gas",
                ]
            ]
        ],
        'DescriptionData_SupplierDeclaredDGHZRegulation' => [
            'sequence' => '30\2900',
            'name' => 'SupplierDeclaredDGHZRegulation',
            'dataType' => 'StringNotNull',
            'minOccurs' => '0',
            "restriction" => [
                "optionValues" => [
                    "ghs",
                    "storage",
                    "waste",
                    "not_applicable",
                    "transportation",
                    "other",
                    "unknown",
                ]
            ]
        ],
        'DescriptionData_HazmatUnitedNationsRegulatoryID' => [
            'sequence' => '30\3000',
            'name' => 'HazmatUnitedNationsRegulatoryID',
            'dataType' => 'string',
            'minOccurs' => '0',
        ],
        'DescriptionData_SafetyDataSheetURL' => [
            'sequence' => '30\3100',
            'name' => 'SafetyDataSheetURL',
            'dataType' => 'anyURI',
            'minOccurs' => '0',
        ],
        'DiscoveryData_Priority' => [
            'sequence' => '40\10',
            'name' => 'Priority',
            'dataType' => 'positiveInteger',
            'minOccurs' => '0',
            "restriction" => [
                'minInclusive' => '1',
                'maxInclusive' => '10',
            ]
        ],
        'DiscoveryData_BrowseExclusion' => [
            'sequence' => '40\20',
            'name' => 'BrowseExclusion',
            'dataType' => 'boolean',
            'minOccurs' => '0',
        ],
        'DiscoveryData_RecommendationExclusion' => [
            'sequence' => '40\30',
            'name' => 'RecommendationExclusion',
            'dataType' => 'boolean',
            'minOccurs' => '0',
        ],
    ];

    const SUB_ATTRIBUTES = [
        'DescriptionData_ItemDimensions_Length_unitOfMeasure_LengthUnitOfMeasure' => [
            'name' => 'unitOfMeasure',
            'minOccurs' => '1',
            'restriction' => [
                    'optionValues' => [
                            "MM",
                            "CM",
                            "M",
                            "IN",
                            "FT",
                            "inches",
                            "feet",
                            "meters",
                            "decimeters",
                            "centimeters",
                            "millimeters",
                            "micrometers",
                            "nanometers",
                            "picometers",
                        ],
                ],
            'default' => 'CM'
        ],
        'DescriptionData_ItemDimensions_Width_unitOfMeasure_LengthUnitOfMeasure' => [
            'name' => 'unitOfMeasure',
            'minOccurs' => '1',
            'restriction' => [
                    'optionValues' => [
                            "MM",
                            "CM",
                            "M",
                            "IN",
                            "FT",
                            "inches",
                            "feet",
                            "meters",
                            "decimeters",
                            "centimeters",
                            "millimeters",
                            "micrometers",
                            "nanometers",
                            "picometers",
                        ],
                ],
            'default' => 'CM'
        ],
        'DescriptionData_ItemDimensions_Height_unitOfMeasure_LengthUnitOfMeasure' => [
            'name' => 'unitOfMeasure',
            'minOccurs' => '1',
            'restriction' => [
                    'optionValues' => [
                            "MM",
                            "CM",
                            "M",
                            "IN",
                            "FT",
                            "inches",
                            "feet",
                            "meters",
                            "decimeters",
                            "centimeters",
                            "millimeters",
                            "micrometers",
                            "nanometers",
                            "picometers",
                        ],
                ],
            'default' => 'CM'
        ],
        'DescriptionData_PackageDimensions_Length_unitOfMeasure_LengthUnitOfMeasure' => [
            'name' => 'unitOfMeasure',
            'minOccurs' => '1',
            'restriction' => [
                    'optionValues' => [
                            "MM",
                            "CM",
                            "M",
                            "IN",
                            "FT",
                            "inches",
                            "feet",
                            "meters",
                            "decimeters",
                            "centimeters",
                            "millimeters",
                            "micrometers",
                            "nanometers",
                            "picometers",
                        ],
                ],
            'default' => 'CM'
        ],
        'DescriptionData_PackageDimensions_Width_unitOfMeasure_LengthUnitOfMeasure' => [
            'name' => 'unitOfMeasure',
            'minOccurs' => '1',
            'restriction' => [
                    'optionValues' => [
                            "MM",
                            "CM",
                            "M",
                            "IN",
                            "FT",
                            "inches",
                            "feet",
                            "meters",
                            "decimeters",
                            "centimeters",
                            "millimeters",
                            "micrometers",
                            "nanometers",
                            "picometers",
                        ],
                ],
            'default' => 'CM'
        ],
        'DescriptionData_PackageDimensions_Height_unitOfMeasure_LengthUnitOfMeasure' => [
            'name' => 'unitOfMeasure',
            'minOccurs' => '1',
            'restriction' => [
                    'optionValues' => [
                            "MM",
                            "CM",
                            "M",
                            "IN",
                            "FT",
                            "inches",
                            "feet",
                            "meters",
                            "decimeters",
                            "centimeters",
                            "millimeters",
                            "micrometers",
                            "nanometers",
                            "picometers",
                        ],
                ],
            'default' => 'CM'
        ],
        'DescriptionData_PackageWeight_unitOfMeasure_WeightUnitOfMeasure' => [
            'name' => 'unitOfMeasure',
            'minOccurs' => '1',
            'restriction' => [
                'optionValues' => [
                        "GR",
                        "KG",
                        "OZ",
                        "LB",
                        "MG",
                    ],
            ],
            'default' => 'KG'
        ],
        'DescriptionData_ShippingWeight_unitOfMeasure_WeightUnitOfMeasure' => [
            'name' => 'unitOfMeasure',
            'minOccurs' => '1',
            'restriction' => [
                'optionValues' => [
                        "GR",
                        "KG",
                        "OZ",
                        "LB",
                        "MG",
                    ],
            ],
            'default' => 'KG'
        ],
        'DescriptionData_ItemWeight_Weight_unitOfMeasure_WeightUnitOfMeasure' => [
            'name' => 'unitOfMeasure',
            'minOccurs' => '1',
            'restriction' => [
                'optionValues' => [
                        "GR",
                        "KG",
                        "OZ",
                        "LB",
                        "MG",
                    ],
            ],
            'default' => 'KG'
        ],
        'DescriptionData_BatteryWeight_Weight_unitOfMeasure_WeightUnitOfMeasure' => [
            'name' => 'unitOfMeasure',
            'minOccurs' => '1',
            'restriction' => [
                'optionValues' => [
                        "GR",
                        "KG",
                        "OZ",
                        "LB",
                        "MG",
                    ],
            ],
            'default' => 'KG'
        ],
        'DescriptionData_LithiumBatteryWeight_Weight_unitOfMeasure_WeightUnitOfMeasure' => [
            'name' => 'unitOfMeasure',
            'minOccurs' => '1',
            'restriction' => [
                'optionValues' => [
                        "GR",
                        "KG",
                        "OZ",
                        "LB",
                        "MG",
                    ],
            ],
            'default' => 'KG'
        ],
        'DescriptionData_ItemVolume_Volume_unitOfMeasure_LengthUnitOfMeasure' => [
            'name' => 'unitOfMeasure',
            'minOccurs' => '1',
            'restriction' => [
                'optionValues' => [
                        "cubic-cm",
                        "cubic-ft",
                        "cubic-in",
                        "cubic-m",
                        "cubic-yd",
                        "cup",
                        "fluid-oz",
                        "gallon",
                        "liter",
                        "milliliter",
                        "ounce",
                        "pint",
                        "quart",
                        "liters",
                        "deciliters",
                        "centiliters",
                        "milliliters",
                        "microliters",
                        "nanoliters",
                        "picoliters",
                    ],
            ],
            'default' => 'cubic-cm'
        ],
        'DescriptionData_LithiumBatteryEnergyContent_EnergyConsumption_unitOfMeasure_EnergyConsumptionUnitOfMeasure' => [
            'name' => 'unitOfMeasure',
            'minOccurs' => '1',
            'restriction' => [
                'optionValues' => [
                        "kilowatt_hours",
                        "btu",
                        "kilowatts",
                        "watt_hours",
                    ],
            ],
            'default' => 'watt_hours'
        ],
        'MSRP_BaseCurrencyCode' => [
            'name' => 'currency',
            'minOccurs' => '1',
            'restriction' => [
                'optionValues' => [
                        "USD",
                        "GBP",
                        "EUR",
                        "JPY",
                        "CAD",
                        "CNY",
                        "INR",
                        "MXN",
                        "BRL",
                        "AUD",
                        "TRY",
                    ],
            ],
            'default' => 'USD'
        ],
        'MSRPWithTax_BaseCurrencyCode' => [
            'name' => 'currency',
            'minOccurs' => '1',
            'restriction' => [
                'optionValues' => [
                        "USD",
                        "GBP",
                        "EUR",
                        "JPY",
                        "CAD",
                        "CNY",
                        "INR",
                        "MXN",
                        "BRL",
                        "AUD",
                        "TRY",
                    ],
            ],
            'default' => 'USD'
        ],
    ];

    const ALLOWED_MARKETPLACE = \Amazon\Sdk\Marketplace::MARKETPLACE_IDS;

    /**
     * Product unique id
     * @var $id
     */
    public $id = 0;

    /**
     * Amazon Category
     * @var null
     */
    public $category = null;

    /**
     * Amazon Sub Category
     * @var null|string $category
     */
    public $subCategory = null;

    /**
     * Amazon Attributes and Values
     * @var array $data
     */
    public $data = [];

    /**
     * Amazon Category Attributes
     * @var array
     */
    public $attributes = [];

    /**
     * Amazon Attribute Value Errors
     * @var array
     */
    public $errors = [];

    /**
     * Attribute Filter Params
     * @var array $params
     */
    public $params;

    /**
     * Sanitized Value to be set on attribute
     * @var string $value
     */
    public $value;

    /** @var \Ced\Validator\Barcode|null $barcodeValidator */
    public $barcodeValidator;

    public $barcodeExemption = false;

    /**
     * Category constructor.
     * @param $subCategory
     */
    public function __construct($subCategory)
    {
        $this->subCategory = $subCategory;
    }

    /**
     * Get a attribute value for a product
     * @param $key
     * @return mixed
     * @throws \Exception
     */
    public function __get($key)
    {
        if (!array_key_exists($key, $this->data)) {
            throw new \Exception(
                "Property '{$key}' does not exist for the product."
            );
        }

        return $this->data[$key];
    }

    /**
     * Get a attribute value for a product
     * @param $key
     * @return string|null
     */
    public function get($key)
    {
        if (!array_key_exists($key, $this->data)) {
            return null;
        }

        return $this->data[$key];
    }

    /**
     * Set a attribute value for a product
     * @param $key
     * @param $value
     * @return $this|boolean
     */
    public function __set($key, $value)
    {
        $this->value = $value;

        if (!array_key_exists($key, $this->getAttributes()) &&
            !array_key_exists($key, $this->getSubAttributes())
        ) {
            return $this;
        }

        if ($value == null || !$this->isValidValue($key, $value)) {
            return false;
        }

        $this->data[$key] = $this->value;
        return $this;
    }

    public function __unset($key)
    {
        if (isset($this->data[$key])) {
            unset($this->data[$key]);
        }
    }

    /**
     * Set Barcode Exemption
     * @param boolean $value
     */
    public function setBarcodeExemption($value)
    {
        $this->barcodeExemption = $value;
    }

    /**
     * Get Barcode Exemption
     * @return bool
     */
    public function getBarcodeExemption()
    {
        return $this->barcodeExemption;
    }

    public function getMessageType()
    {
        return \Amazon\Sdk\Base::MESSAGE_TYPE_PRODUCT;
    }

    /**
     * Check validity of a attribute value before setting.
     * @param $key
     * @param $value
     * @return bool
     */
    public function isValidValue($key, $value)
    {
        $valid = false;
        $attribute = !empty($this->getAttribute($key)) ? $this->getAttribute($key) : $this->getSubAttribute($key);
        if (!empty($attribute) && $value !== null) {
            $valid = true;
            $level = isset($attribute['minOccurs']) ? $attribute['minOccurs'] : 0;
            $name = $attribute['name'];

            // Checking dataType
            if (isset($attribute['dataType']) === true
                && $this->typeCheck($attribute['dataType'], $value, $attribute) === false
            ) {
                $valid = false;
                $this->setError(
                    $key,
                    "Invalid value data type - '{$value}'. '{$name}' must be a '{$attribute['dataType']}'",
                    $level
                );
            }

            // Checking optionValues
            if (isset($attribute['restriction']['optionValues']) === true
                and in_array($value, $attribute['restriction']['optionValues']) === false
            ) {
                $valid = false;
                $options = implode("|", $attribute['restriction']['optionValues']);
                $this->setError(
                    $key,
                    "Invalid option value - '{$value}'. '{$name}' must contains values from '{$options}'",
                    $level
                );
            }
        }
        return $valid;
    }

    /**
     * @TODO: Add all cases
     * Check type of attribute value
     * @param $type
     * @param $value
     * @param array $attribute
     * @return bool
     */
    public function typeCheck($type, $value, $attribute = [])
    {
        $this->value = $value;

        $typeChecks = [
            'string' => function ($value) {
                return is_string($value) || is_integer($value) || is_bool($value);
            },
            'String' => function ($value) {
                return is_string($value) || is_integer($value) || is_bool($value);
            },
            'StringNotNull' => function ($value) {
                return is_string($value);
            },
            'LongStringNotNull' => function ($value) {
                return is_string($value);
            },
            'SuperLongStringNotNull' => function ($value) {
                return is_string($value);
            },
            'FortyStringNotNull' => function ($value) {
                return is_string($value) && (strlen($value) < 41);
            },
            'FourDigitYear' => function ($value) {
                return is_string($value) && (strlen($value) < 5);
            },
            'integer' => function ($value) {
                return is_numeric($value);
            },
            'positiveInteger' => function ($value) {
                return is_numeric($value) && ($value > 0);
            },
            'boolean' => function ($value) {
                return is_bool($value) || ($value == 'true' || $value = 'false');
            },
            'float' => function ($value) {
                return is_numeric($value);
            },
            'decimal' => function ($value) {
                return is_numeric($value);
            },
            'PositiveDimensionTwoDecimal' => function ($value, $self, $attribute) {
                if (isset($attribute['attribute']) && !is_numeric($value)) {
                    $unitAttribute = $self->getSubAttribute($attribute['attribute']);
                    $unitResult = $this->extractUnitWithValue($unitAttribute, [], $value);

                    if (isset($unitResult['value'])) {
                        $value = $unitResult['value'];
                    }
                }
                return is_numeric($value) && (strlen($value) < 12) && ($value > 0);
            },
            'PositiveDimensionFourDecimal' => function ($value, $self, $attribute) {
                if (isset($attribute['attribute']) && !is_numeric($value)) {
                    $unitAttribute = $self->getSubAttribute($attribute['attribute']);
                    $unitResult = $this->extractUnitWithValue($unitAttribute, [], $value);

                    if (isset($unitResult['value'])) {
                        $value = $unitResult['value'];
                    }
                }
                return is_numeric($value) && (strlen($value) < 12) && ($value > 0);
            },
            'PositiveDimension' => function ($value, $self, $attribute) {
                if (isset($attribute['attribute']) && !is_numeric($value)) {
                    $unitAttribute = $self->getSubAttribute($attribute['attribute']);
                    $unitResult = $this->extractUnitWithValue($unitAttribute, [], $value);

                    if (isset($unitResult['value'])) {
                        $value = $unitResult['value'];
                    }
                }
                return is_numeric($value) && (strlen($value) < 12) && ($value > 0);
            },
            'PositiveIntegerDimension' => function ($value, $self, $attribute) {
                if (isset($attribute['attribute']) && !is_numeric($value)) {
                    $unitAttribute = $self->getSubAttribute($attribute['attribute']);
                    $unitResult = $this->extractUnitWithValue($unitAttribute, [], $value);

                    if (isset($unitResult['value'])) {
                        $value = $unitResult['value'];
                    }
                }
                return is_numeric($value) && (strlen($value) < 12) && ($value > 0);
            },
            'Dimension' => function ($value, $self, $attribute) {
                if (isset($attribute['attribute']) && !is_numeric($value)) {
                    $unitAttribute = $self->getSubAttribute($attribute['attribute']);
                    $unitResult = $this->extractUnitWithValue($unitAttribute, [], $value);

                    if (isset($unitResult['value'])) {
                        $value = $unitResult['value'];
                    }
                }

                return is_numeric($value) && (strlen($value) < 13);
            },
            'PositiveNonZeroDimension' => function ($value, $self, $attribute) {
                if (isset($attribute['attribute']) && !is_numeric($value)) {
                    $unitAttribute = $self->getSubAttribute($attribute['attribute']);
                    $unitResult = $this->extractUnitWithValue($unitAttribute, [], $value);

                    if (isset($unitResult['value'])) {
                        $value = $unitResult['value'];
                    }
                }

                return is_numeric($value) && (strlen($value) < 13);
            },
            'anyURI' => function ($value) {
                return filter_var($value, FILTER_VALIDATE_URL);
            },
            'Barcode' => function ($value, $self) {
                /** @var \Amazon\Sdk\Product\Category $self */
                $result = false;
                if (is_string($value) && strlen($value) < 17) {
                    /** @var \Ced\Validator\Barcode $validator */
                    $validator = $self->getBarcodeValidator();
                    $validator->setBarcode($value);
                    $result = $validator->isValid();
                    if ($result) {
                        $this->data["StandardProductID_Type"] = $validator->getType();
                    }
                }
                return $result;
            },
            'normalizedString' => function ($value, $self) {
                if (is_string($value)) {
                    $self->value = strip_tags($value);
                    return true;
                }

                return false;
            },
            'richText' => function ($value, $self) {
                // "<br><p><b>" These tags are allowed and ignored in validation.
                // Please use Zend Filter to remove tag attributes.
                // https://framework.zend.com/manual/2.1/en/modules/zend.filter.strip-tags.html
                if (is_string($value)) {
                    $self->value = strip_tags($value, "<br><p><b>");
                    return true;
                }

                return false;
            },
            'dateTime' => function ($value, $self) {
                //YYYY-MM-DD HH:MM:SS Ex: '2008-09-01 12:35:45'
                return preg_match('/\d{4}-\d{2}-\d{2} \d{1,2}:\d{2}:\d{2}/', (string)$value);
            },
        ];

        if (isset($typeChecks[$type])) {
            $result = $typeChecks[$type]($value, $this, $attribute);
        } else {
            $result = is_string($value);
        }

        return $result;
    }

    public function getBarcodeValidator()
    {
        if (!isset($this->barcodeValidator)) {
            $this->barcodeValidator = new \Ced\Validator\Barcode();
        }

        return $this->barcodeValidator;
    }

    public function isValid()
    {
        $this->validateData();
        if (isset($this->getError()[self::ERRORS[1]]) and
            !empty($this->getError()[self::ERRORS[1]])
        ) {
            return false;
        }
        return true;
    }

    public function validateData()
    {
        $attributes = $this->getAttributes();
        foreach ($attributes as $attributeId => $attribute) {
            if (isset($attribute['minOccurs']) and
                $attribute['minOccurs'] == '1' and
                !isset($this->data[$attributeId])
            ) {
                $name = $attribute['name'];
                $this->setError(
                    $attributeId,
                    "Required attribute '{$name}' value missing or invalid.",
                    $attribute['minOccurs']
                );
            } elseif (isset($this->data[$attributeId], $attribute['depends'])) {
                $this->dependsCheck($attributeId, $attribute);
            }
        }
    }

    public function setError($key, $value, $level = 0)
    {
        if ($level == 1) {
            $this->errors[$this->getId()][self::ERRORS[$level]][$key] = $value;
        } else {
            $this->errors[$this->getId()][self::ERRORS[0]][$key] = $value;
        }
        return true;
    }

    public function getId()
    {
        return $this->id;
    }

    public function setId($id = 0, $unique = false)
    {
        $this->id = $id;
    }

    public function dependsCheck($attributeId, $attribute)
    {
        $name = $attribute['name'];
        $attributes = implode(",", array_values($attribute['depends']));
        foreach ($attribute['depends'] as $id) {
            if (!isset($this->data[$id])) {
                $this->setError(
                    $attributeId,
                    "'{$name}' attribute depends on following attributes - '{$attributes}'.",
                    0
                );
                return false;
            }
        }
        return true;
    }

    public function getError($level = null)
    {
        $errors = [];
        $id = $this->getId();
        if (isset($level, $this->errors[$id][$level])) {
            return isset($this->errors[$id][$level]) ? $this->errors[$id][$level] : $errors;
        }
        return isset($this->errors[$id]) ? $this->errors[$id] : $errors;
    }

    /**
     * Sanitize string, remove multiple space to single space. convert  to lower case.
     * @param $input
     * @return string
     */
    private function sanitize($input)
    {
        $output = strtolower($input);
        $output = preg_replace('/\s+/', ' ', $output);
        return $output;
    }

    public function uasort($a, $b)
    {
        $result = 0;

        if (isset($a['sequence'], $b['sequence'])) {
            $a = explode("\\", $a['sequence']);
            $b = explode("\\", $b['sequence']);
            foreach ($a as $index => $sequence) {
                if (isset($b[$index])) {
                    if ($sequence < $b[$index]) {
                        $result = -1;
                        break;
                    } else {
                        $result = 1;
                    }
                }
            }
        }

        return $result;
    }

    public function getData($key = null)
    {
        if (isset($key) && isset($this->data[$key])) {
            return $this->data[$key];
        }

        $data = [];
        $attributes = $this->getAttributes();
//        uasort($attributes, [$this, 'uasort']);
        foreach ($attributes as $attributeId => $attribute) {
            // Default values are to be send in Product Upload only. Skipped in Delete operation
            if (isset($attribute['default']) &&
                (!isset($this->data[$attributeId]) || empty($this->data[$attributeId])) &&
                $this->getOperationType() === \Amazon\Sdk\Base::OPERATION_TYPE_UPDATE
            ) {
                $this->data[$attributeId] = $attribute['default'];
            }

            if (isset($this->data[$attributeId])) {
                $value = $this->data[$attributeId];

                // Checking for dependent attribute value
                if (isset($attribute['depends']) && !$this->dependsCheck($attributeId, $attribute)) {
                    continue;
                }

                // Checking for sub attribute value, for example units
                $unitValues = [];
                if (isset($attribute['attribute'])) {
                    $sa = $this->getSubAttribute($attribute['attribute']);
                    if (isset($sa, $this->data[$attribute['attribute']])) {
                        $unitValues[$sa['name']] = $this->data[$attribute['attribute']];
                    } elseif (isset($sa)) {
                        $unitResult = $this->extractUnitWithValue($sa, $unitValues, $value);
                        if (isset($unitResult['unitValues']) && is_array($unitResult['unitValues'])) {
                            $unitValues = array_merge($unitValues, $unitResult['unitValues']);
                        }

                        if (isset($unitResult['value'])) {
                            $value = $unitResult['value'];
                        }
                    }
                }

                // Checking for multi value attribute, generate array for string value
                if (isset($attribute['maxOccurs']) and $attribute['maxOccurs'] > 1) {
                    $value = array_slice(
                        explode(self::MULTIVALUE_DELIMITER, $this->data[$attributeId]),
                        0,
                        $attribute['maxOccurs']
                    );
                }

                // Checking for value length
                if (isset($attribute['length']) and is_string($value)) {
                    $length = explode(':', $attribute['length']);
                    $value = mb_substr($value, 0, $length[1]);
                }

                // Fixing dimensions decimal points
                if (!empty($value) &&
                    in_array($attribute['dataType'], ['Dimension', 'PositiveDimension', 'PositiveNonZeroDimension'])) {
                    $value = number_format((float)$value, 2, '.', '');
                }

                // Removing decimal points
                if (!empty($value) &&
                    in_array($attribute['dataType'], ['PositiveIntegerDimension', 'positiveInteger'])) {
                    if (is_array($value)) {
                        foreach ($value as &$v) {
                            $v = number_format((float)$v, 0, '.', '');
                        }
                    } else {
                        $value = number_format((float)$value, 0, '.', '');
                    }
                }

                // Fixing dimensions two decimal points
                if (!empty($value) &&
                    in_array($attribute['dataType'], ['PositiveDimensionTwoDecimal'])) {
                    $value = number_format((float)$value, 2, '.', '');
                }

                // Fixing dimensions four decimal points
                if (!empty($value) &&
                    in_array($attribute['dataType'], ['PositiveDimensionFourDecimal'])) {
                    $value = number_format((float)$value, 4, '.', '');
                }

                $sequence = $attribute['sequence'];
                $value = $this->generateArray($attributeId, $value, $unitValues, $sequence);
                $data = $this->arrayMerge([$data, $value]);
            }
        }

        return $data;
    }

    /**
     * Extract Unit from Value
     * @param $unitAttribute
     * @param $unitValues
     * @param $value
     * @return array
     */
    public function extractUnitWithValue($unitAttribute, $unitValues, $value)
    {
        // Split the value to value and unit. Ex: "1.6Ghz" to ['1.6', 'GHz']
        $valueWithUnit = preg_split('/(?<=[0-9])(?=[a-z]+)/i', str_replace(" ", "", (string)$value));
        $unitId = $unitAttribute['name'];

        if (isset($valueWithUnit[1]) && !empty($valueWithUnit[1])) {
            foreach ($unitAttribute['restriction']['optionValues'] as $unit) {
                if ($this->sanitize($unit) == $this->sanitize($valueWithUnit[1])) {
                    $unitValues[$unitId] = $unit;
                    break;
                }
            }

            if (!empty($unitValues[$unitId])) {
                $unitValues[$unitId] = $unitAttribute['default'];
                if (isset($valueWithUnit[0]) && !empty($valueWithUnit[0])) {
                    $value = $valueWithUnit[0];
                }
            }
        } else {
            $unitValues[$unitId] = $unitAttribute['default'];
        }

        return ["value" => $value, "unitValues" => $unitValues];
    }

    public function setData($key, $value)
    {
        return $this->__set($key, $value);
    }

    public function getSubAttribute($key)
    {
        $subAttributes = $this->getSubAttributes();
        if (isset($subAttributes[$key])) {
            return $subAttributes[$key];
        }
        return null;
    }

    /**
     * Generate Array
     * @param null $key
     * @param null $value
     * @param array $attribute
     * @param $sequence
     * @return array
     */
    public function generateArray($key = null, $value = null, $attribute = [], $sequence)
    {
        try {
            $data = [];
            if ($key != null) {
                $attributeArray = explode(self::KEY_DELIMITER, $key);
                $sequence = explode(self::SEQUENCE_DELIMITER, $sequence);
                for ($i = count($attributeArray) - 1; $i >= 0; $i--) {
                    if (count($attributeArray) - 1 == $i) {
                        $data['_attribute'] = [];
                        if (is_array($value)) {
                            foreach ($value as $index => $item) {
                                $data['_value'][$sequence[$i] + $index][$attributeArray[$i]] = [
                                    '_attribute' => $attribute,
                                    '_value' => (string)$item
                                ];
                            }
                        } else {
                            $data['_value'][$sequence[$i]][$attributeArray[$i]] = [
                                '_attribute' => $attribute,
                                '_value' => (string)$value
                            ];
                        }
                    } else {
                        $data = [
                            '_attribute' => [],
                            '_value' => [
                                $sequence[$i] => [
                                    $attributeArray[$i] => $data
                                ]
                            ]
                        ];
                    }
                }
            }
        } catch (\Exception $e) {
            throw new \InvalidArgumentException("Invalid data for property '{$key}'");
        }

        return $data;
    }

    /**
     * Array Merge with sequence maintained.
     * @param $arrays
     * @return array
     */
    public function arrayMerge($arrays)
    {
        $result = [];
        foreach ($arrays as $array) {
            foreach ($array as $key => $value) {
                if (isset($result[$key]) && is_array($result[$key]) && is_array($value)) {
                    // Recurse when both values are arrays. Array indexes are maintained.
                    $result[$key] = $this->arrayMerge([$result[$key], $value]);
                } else {
                    // Otherwise, use the latter value, overriding any previous value.
                    $result[$key] = $value;
                }
            }
        }
        return $result;
    }

    public function clear()
    {
        $this->data = [];
        $this->errors = [];
    }

    public function getAttribute($key)
    {
        $attributes = $this->getAttributes();
        if (isset($attributes[$key])) {
            return $attributes[$key];
        }
        return false;
    }

    public function filterAttributes($value)
    {
        $status = false;
        if (isset($this->params["minOccurs"], $value["minOccurs"]) &&
            $value["minOccurs"] == $this->params["minOccurs"]
        ) {
            $status = true;
        }

        if (isset($value['visible']) && $value['visible'] == false) {
            $status = false;
        }

        return $status;
    }

    public function filterParentageAttribute($value)
    {
        $status = false;
        if (isset($value) && strpos($value, "VariationData_Parentage") !== false) {
            $status = true;
        }
        if (!$status && isset($value) && strpos($value, "Parentage") !== false) {
            $status = true;
        }

        return $status;
    }

    public function getParentageAttribute()
    {
        $attribute = array_filter(array_keys($this->getAttributes()), [$this, 'filterParentageAttribute']);
        $key = array_pop($attribute);
        return $key;
    }

    public function filterVariationThemeAttribute($value)
    {
        $status = false;
        if (isset($value) && strpos($value, "VariationData_VariationTheme") !== false) {
            $status = true;
        }

        return $status;
    }

    public function getVariationThemeAttribute()
    {
        $attribute = array_filter(array_keys($this->getAttributes()), [$this, 'filterVariationThemeAttribute']);
        $key = array_pop($attribute);
        return $key;
    }
}
