<?php

namespace Amazon\Sdk;

/**
 * @TODO add xml preparation and validatation
 * Class Envelope
 * @package Amazon\Sdk\Product
 */
class Envelope
{
    /**
     * Message Types List Mapping with Feed Types
     * @var array
     */
    const MESSAGE_TYPES = [
        \Amazon\Sdk\Api\Feed::PRODUCT => \Amazon\Sdk\Base::MESSAGE_TYPE_PRODUCT,
        \Amazon\Sdk\Api\Feed::PRODUCT_PRICING => \Amazon\Sdk\Base::MESSAGE_TYPE_PRICE,
        \Amazon\Sdk\Api\Feed::PRODUCT_INVENTORY => \Amazon\Sdk\Base::MESSAGE_TYPE_INVENTORY,
        \Amazon\Sdk\Api\Feed::PRODUCT_IMAGE => \Amazon\Sdk\Base::MESSAGE_TYPE_IMAGE,
        \Amazon\Sdk\Api\Feed::PRODUCT_RELATIONSHIP => \Amazon\Sdk\Base::MESSAGE_TYPE_RELATIONSHIP,
        \Amazon\Sdk\Api\Feed::ORDER_ACKNOWLEDGEMENT => \Amazon\Sdk\Base::MESSAGE_TYPE_ORDER_ACKNOWLEDGEMENT,
        \Amazon\Sdk\Api\Feed::ORDER_FULFILLMENT => \Amazon\Sdk\Base::MESSAGE_TYPE_ORDER_FULFILLMENT,
    ];

    /**
     * Message Types List
     * @var array
     */
    public $messageTypes = [
        \Amazon\Sdk\Base::MESSAGE_TYPE_FULFILLMENT_CENTER,
        \Amazon\Sdk\Base::MESSAGE_TYPE_INVENTORY,
        \Amazon\Sdk\Base::MESSAGE_TYPE_LISTINGS,
        \Amazon\Sdk\Base::MESSAGE_TYPE_ORDER_ACKNOWLEDGEMENT,
        \Amazon\Sdk\Base::MESSAGE_TYPE_ORDER_ADJUSTMENT,
        \Amazon\Sdk\Base::MESSAGE_TYPE_ORDER_FULFILLMENT,
        \Amazon\Sdk\Base::MESSAGE_TYPE_OVERRIDE,
        \Amazon\Sdk\Base::MESSAGE_TYPE_PRICE,
        \Amazon\Sdk\Base::MESSAGE_TYPE_PROCESSING_REPORT,
        \Amazon\Sdk\Base::MESSAGE_TYPE_PRODUCT,
        \Amazon\Sdk\Base::MESSAGE_TYPE_IMAGE,
        \Amazon\Sdk\Base::MESSAGE_TYPE_RELATIONSHIP,
        \Amazon\Sdk\Base::MESSAGE_TYPE_SETTLEMENT_REPORT,
    ];

    public $envelope = [
        'AmazonEnvelope' => [
            '_attribute' => [
                'xmlns:xsi' => 'http://www.w3.org/2001/XMLSchema-instance',
                'xsi:noNamespaceSchemaLocation' => 'amzn-envelope.xsd'
            ],
            '_value' => [
                0 => [
                    'Header' => [
                        'DocumentVersion' => '1.01',
                        'MerchantIdentifier' => ''
                    ],
                    'MessageType' => \Amazon\Sdk\Base::MESSAGE_TYPE_PRODUCT,
                    'PurgeAndReplace' => 'false',
                ]

            ],
        ]
    ];

    /**
     * Counter Index, Incremented "after" every message is added.
     * @var int
     */
    public $index = 1;

    /**
     * Index to Message Id Mapping
     * @var array
     */
    public $indexList = [];

    /**
     * All Message data stored.
     * @var array
     */
    public $data = [];

    public $path;

    /**
     * Envelope constructor.
     * @param $merchantIdentifier
     * @param string $messageType
     * @param string $purgeAndReplace
     * CAUTION: An empty PurgeAndReplace feed, by definition, will cancel all of your listings. Since it
     * does not contain any product information, all of your listings will no longer be available on Amazon
     * and they will also be gone from your seller account.
     * The Purge-and-Replace option only shows effect when it is inserted into a product feed. For other
     * feed types such as image, relation or stock/price feeds, it will not have any effect.
     */
    public function __construct(
        $merchantIdentifier,
        $messageType = \Amazon\Sdk\Base::MESSAGE_TYPE_PRODUCT,
        $purgeAndReplace = 'false'
    ) {
        $this->envelope['AmazonEnvelope']['_value'][0]['Header']['MerchantIdentifier'] = $merchantIdentifier;
        if (in_array($messageType, $this->messageTypes)) {
            $this->envelope['AmazonEnvelope']['_value'][0]['MessageType'] = $messageType;
        } else {
            $messageTypes = implode('|', $this->messageTypes);
            throw new \InvalidArgumentException(
                "Invalid value for 'messageType'. Allowed values are: [{$messageTypes}]"
            );
        }
        $this->envelope['AmazonEnvelope']['_value'][0]['PurgeAndReplace'] = $purgeAndReplace;
    }

    /**
     * Get MessageType from FeedType
     * @param string $feedType
     * @return mixed
     */
    public static function getMessageType($feedType = \Amazon\Sdk\Api\Feed::PRODUCT)
    {
        $messageType = self::MESSAGE_TYPES[\Amazon\Sdk\Api\Feed::PRODUCT];
        if (isset(self::MESSAGE_TYPES[$feedType])) {
            $messageType = self::MESSAGE_TYPES[$feedType];
        }
        return $messageType;
    }

    /**
     * Add a Product Message
     * @param \Amazon\Sdk\Product\CategoryInterface $product
     * @return int
     */
    public function addProduct(\Amazon\Sdk\Product\CategoryInterface $product)
    {
        $id = (string)$product->getId();
        if (isset($this->indexList[$id])) {
            $index = $this->indexList[$id];
        } else {
            $index = $this->index++;
        }

        $message = [
            'Message' => [
                '_attribute' => [],
                '_value' => [
                    'MessageID' => $id,
                    'OperationType' => (string)$product->getOperationType(),
                    $product->getMessageType() => $product->getData(),
                ]
            ]
        ];

        $this->envelope['AmazonEnvelope']['_value'][$index] = $message;
        $this->indexList[$id] = $index;

        return $index;
    }

    /**
     * Remove a Product Message
     * @param $index
     * @return bool
     */
    public function removeProduct($index)
    {
        if (isset($this->envelope['AmazonEnvelope']['_value'][$index]) && $index > 0) {
            unset($this->envelope['AmazonEnvelope']['_value'][$index]);
            $indexes = array_flip($this->indexList);
            if (isset($indexes[$index])) {
                $id = $indexes[$index];
                if (isset($this->indexList[$id])) {
                    unset($this->indexList[$id]);
                }
            }

            return true;
        }

        return false;
    }

    /**
     * Add a Inventory Message
     * @param \Amazon\Sdk\Product\Inventory $inventory
     * @return int
     */
    public function addInventory(\Amazon\Sdk\Product\Inventory $inventory)
    {
        $id = (string)$inventory->getId();
        if (isset($this->indexList[$id])) {
            $index = $this->indexList[$id];
        } else {
            $index = $this->index++;
        }

        $message = [
            'Message' => [
                '_attribute' => [],
                '_value' => [
                    'MessageID' => $id,
                    'OperationType' => (string)$inventory->getOperationType(),
                    $inventory->getMessageType() => $inventory->getData(),
                ]
            ]
        ];

        $this->envelope['AmazonEnvelope']['_value'][$index] = $message;
        $this->indexList[$id] = $index;

        return $this->index;
    }

    /**
     * Add a Product Image Message
     * @param \Amazon\Sdk\Product\Image $image
     * @return int
     */
    public function addImage(\Amazon\Sdk\Product\Image $image)
    {
        $message = [
            'Message' => [
                '_attribute' => [],
                '_value' => [
                    'MessageID' => (string)$image->getId(),
                    'OperationType' => (string)$image->getOperationType(),
                    $image->getMessageType() => $image->getData(),
                ]
            ]
        ];

        $this->envelope['AmazonEnvelope']['_value'][$this->index++] = $message;

        return $this->index;
    }

    /**
     * Add a Product Price Message
     * @param \Amazon\Sdk\Product\Price $price
     * @return int
     */
    public function addPrice(\Amazon\Sdk\Product\Price $price)
    {
        $id = (string)$price->getId();
        if (isset($this->indexList[$id])) {
            $index = $this->indexList[$id];
        } else {
            $index = $this->index++;
        }

        $message = [
            'Message' => [
                '_attribute' => [],
                '_value' => [
                    'MessageID' => $id,
                    'OperationType' => (string)$price->getOperationType(),
                    $price->getMessageType() => [
                        '_attribute' => [],
                        '_value' => $price->getData()
                    ],
                ]
            ]
        ];

        $this->envelope['AmazonEnvelope']['_value'][$index] = $message;
        $this->indexList[$id] = $index;

        return $this->index;
    }

    /**
     * Add a Product Relationship Message
     * @param \Amazon\Sdk\Product\Relationship $relationship
     * @return int
     */
    public function addRelationship(\Amazon\Sdk\Product\Relationship $relationship)
    {
        $message = [
            'Message' => [
                '_attribute' => [],
                '_value' => [
                    'MessageID' => (string)$relationship->getId(),
                    'OperationType' => (string)$relationship->getOperationType(),
                    $relationship->getMessageType() => [
                        '_attribute' => [],
                        '_value' => $relationship->getData()
                    ],
                ]
            ]
        ];

        $this->envelope['AmazonEnvelope']['_value'][$this->index++] = $message;

        return $this->index;
    }

    /**
     * Add a Order Acknowledgement Message
     * @param \Amazon\Sdk\Order\Acknowledgement $acknowledgement
     * @return int
     */
    public function addAcknowledgement(\Amazon\Sdk\Order\Acknowledgement $acknowledgement)
    {
        $message = [
            'Message' => [
                '_attribute' => [],
                '_value' => [
                    'MessageID' => (string)$acknowledgement->getId(),
                    'OperationType' => (string)$acknowledgement->getOperationType(),
                    $acknowledgement->getMessageType() => [
                        '_attribute' => [],
                        '_value' => $acknowledgement->getData()
                    ],
                ]
            ]
        ];

        $this->envelope['AmazonEnvelope']['_value'][$this->index++] = $message;

        return $this->index;
    }

    /**
     * Add a Order Fulfillment Message
     * @param \Amazon\Sdk\Order\Fulfillment $fulfillment
     * @return int
     */
    public function addFulfillment(\Amazon\Sdk\Order\Fulfillment $fulfillment)
    {
        $message = [
            'Message' => [
                '_attribute' => [],
                '_value' => [
                    'MessageID' => (string)$fulfillment->getId(),
                    'OperationType' => (string)$fulfillment->getOperationType(),
                    $fulfillment->getMessageType() => [
                        '_attribute' => [],
                        '_value' => $fulfillment->getData()
                    ],
                ]
            ]
        ];

        $this->envelope['AmazonEnvelope']['_value'][$this->index++] = $message;

        return $this->index;
    }

    /**
     * Add a Order Adjustment Message
     * @param \Amazon\Sdk\Order\Adjustment $adjustment
     * @return int
     */
    public function addAdjustment(\Amazon\Sdk\Order\Adjustment $adjustment)
    {
        $message = [
            'Message' => [
                '_attribute' => [],
                '_value' => [
                    'MessageID' => (string)$adjustment->getId(),
                    'OperationType' => (string)$adjustment->getOperationType(),
                    $adjustment->getMessageType() => [
                        '_attribute' => [],
                        '_value' => $adjustment->getData()
                    ],
                ]
            ]
        ];

        $this->envelope['AmazonEnvelope']['_value'][$this->index++] = $message;

        return $this->index;
    }

    /**
     * Get Envelope data as Array or Xml
     * @param string $type , xml || null
     * @param null $path
     * @return array|string
     * @throws \DOMException
     */
    public function getData($type = 'xml', $path = null)
    {
        if (isset($this->envelope['AmazonEnvelope']['_value'][1])) {
            if ($type == 'xml') {
                // Setting xml string in data.
                $generator = new \Amazon\Sdk\Generator();
                $this->data = $generator->arrayToXml($this->envelope)->__toString();

                if (isset($path) && file_exists($path)) {
                    // Saving file if path is valid.
                    $this->path = $path;
                    $generator->save($this->path);
                }
            } else {
                // Setting the associative array in data.
                $this->data = $this->envelope;
            }
        }

        return $this->data;
    }
}
