<?php

/**
 * CedCommerce
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the End User License Agreement (EULA)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://cedcommerce.com/license-agreement.txt
 *
 * @category    Ced
 * @package     Ced_Amazon
 * @author      CedCommerce Core Team <connect@cedcommerce.com>
 * @copyright   Copyright CedCommerce (http://cedcommerce.com/)
 * @license     http://cedcommerce.com/license-agreement.txt
 */

namespace Amazon\Sdk;

/**
 * Base object class
 * Class Base
 * @package Amazon\Sdk
 */
abstract class Base
{
    /**
     * Operation Types:
     *
     * Update: All specified information overwrites any existing information.
     * Any unspecified information is erased.
     *
     * Delete: All information is removed.
     *
     * PartialUpdate: All specified information overwrites any existing information,
     * but unspecified information is unaffected.
     */
    const OPERATION_TYPE_UPDATE = 'Update';

    const OPERATION_TYPE_DELETE = 'Delete';

    const OPERATION_TYPE_PARTIAL_UPDATE = 'PartialUpdate';

    const OPERATION_TYPE_REQUEST = 'Request'; // SDK specific for report request

    const OPERATION_TYPE_GET = 'Get'; // SDK specific for report request

    /**
     * Message Types:
     */
    const MESSAGE_TYPE_PRODUCT = 'Product';

    const MESSAGE_TYPE_INVENTORY = 'Inventory';

    const MESSAGE_TYPE_PRICE = 'Price';

    const MESSAGE_TYPE_IMAGE = 'ProductImage';

    const MESSAGE_TYPE_RELATIONSHIP = 'Relationship';

    const MESSAGE_TYPE_OVERRIDE = 'Override';

    const MESSAGE_TYPE_LISTINGS = 'Listings';

    const MESSAGE_TYPE_FULFILLMENT_CENTER = 'FulfillmentCenter';

    const MESSAGE_TYPE_ORDER_ACKNOWLEDGEMENT = 'OrderAcknowledgement';

    const MESSAGE_TYPE_ORDER_FULFILLMENT = 'OrderFulfillment';

    const MESSAGE_TYPE_ORDER_ADJUSTMENT = 'OrderAdjustment';

    const MESSAGE_TYPE_SETTLEMENT_REPORT = 'SettlementReport';

    const MESSAGE_TYPE_PROCESSING_REPORT = 'ProcessingReport';

    /**
     * Message Id
     * @var int
     */
    public $id = 0;

    /**
     * Object data
     * @var array $data
     */
    public $data;

    /**
     * Operation Type
     * @var $operationType
     */
    public $operationType = self::OPERATION_TYPE_UPDATE;

    /** @var string  */
    public $messageType = self::MESSAGE_TYPE_PRODUCT;

    /**
     * Allowed Operation Types
     * @var array $operationTypes
     */
    public $operationTypes = [
        self::OPERATION_TYPE_UPDATE,
        self::OPERATION_TYPE_DELETE,
        self::OPERATION_TYPE_PARTIAL_UPDATE
    ];

    /**
     * Getting Operation Type
     * @return string
     */
    public function getOperationType()
    {
        return $this->operationType;
    }

    /**
     * Getting Message Type
     * @return string
     */
    public function getMessageType()
    {
        return $this->messageType;
    }

    /**
     * Setting Operation Type
     * @param $operationType
     */
    public function setOperationType($operationType)
    {
        if (in_array($operationType, $this->operationTypes)) {
            $this->operationType = $operationType;
        } else {
            $operationTypes = implode('|', $this->operationTypes);
            throw new \InvalidArgumentException(
                "Invalid value for 'operationType'. Allowed values are: [{$operationTypes}]"
            );
        }
    }

    /**
     * Get Data from the object
     * @return array
     */
    public function getData()
    {
        return $this->data;
    }

    /**
     * Get Message Id
     * @return int
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set Message Id
     * @param integer $id
     * @param bool $unique
     * @throws \Exception
     */
    public function setId($id, $unique = false)
    {
        if ($unique || strlen($id) > 20) {
            $date = new \DateTime();
            $timestamp = $date->getTimestamp();
            $id = $timestamp . rand(1000000000, 9999999999);
        }

        $this->id = $id;
    }

    public function validateDate($date, $format = 'Y-m-d H:i:s P')
    {
        $d = \DateTime::createFromFormat($format, $date);
        return $d && $d->format($format) == $date;
    }

    public function generateDate($date, $outputFormat = 'c', $inputFormat = 'Y-m-d H:i:s P')
    {
        $d = \DateTime::createFromFormat($inputFormat, $date);
        if ($d) {
            return $d->format($outputFormat);
        }
        return false;
    }

    public function getDate($date)
    {
        $now = new \DateTime($date);
        $d = $now->format('c');
        return $d;
    }
}
