<?php
/**
 * Copyright 2013 CPI Group, LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 *
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

namespace Amazon\Sdk\Api\Product;

/**
 * Core class for Amazon Products API.
 *
 * This is the core class for all objects in the Amazon Products section.
 * It contains a few methods that all Amazon Products Core objects use.
 */
abstract class ProductsCore extends \Amazon\Sdk\Api\Core
{
    protected $productList;
    protected $index = 0;

    /**
     * AmazonProductsCore constructor sets up key information used in all Amazon Products Core requests
     *
     * This constructor is called when initializing all objects in the Amazon Products Core.
     * The parameters are passed by the child objects' constructors, which are
     * in turn passed to the AmazonCore constructor. See it for more information
     * on these parameters and common methods.
     * @param \Amazon\Sdk\Api\ConfigInterface|null $config ,
     * @param \Psr\Log\LoggerInterface|null $logger ,
     * @param boolean $mockMode [optional] <p>This is a flag for enabling Mock Mode.
     * This defaults to <b>FALSE</b>.</p>
     * @param array|string $mockFiles [optional] <p>The files (or file) to use in Mock Mode.</p>
     */
    public function __construct(
        \Amazon\Sdk\Api\ConfigInterface $config = null,
        \Psr\Log\LoggerInterface $logger = null,
        $mockMode = false,
        $mockFiles = null
    ) {
        parent::__construct($config, $logger, $mockMode, $mockFiles);

        $this->urlbranch = 'Products/' . \Amazon\Sdk\Api\Environment::AMAZON_VERSION_PRODUCTS;
        $this->options['Version'] = \Amazon\Sdk\Api\Environment::AMAZON_VERSION_PRODUCTS;

        /** @var array|string $marketplaceIds */
        $marketplaceIds = $this->config->getMarketplaceId();
        $this->setMarketplaceIds($marketplaceIds);

        $this->throttleLimit = \Amazon\Sdk\Api\Environment::THROTTLE_LIMIT_PRODUCT;
    }

    public function setMarketplaceIds($marketplaceIds)
    {
        // NOTE: Product data can be obtained for single marketplace only.
        if (isset($marketplaceIds) && !empty($marketplaceIds)) {
            if ($marketplaceIds && is_string($marketplaceIds)) {
                $this->resetMarketplaceIds();
                $this->options["MarketplaceId"] = $marketplaceIds;
            } elseif ($marketplaceIds && is_array($marketplaceIds)) {
                $this->resetMarketplaceIds();
                $marketplaceId = reset($marketplaceIds);
                $this->options["MarketplaceId"] = $marketplaceId;
            }
        } else {
            $this->log("Marketplace Id is missing", 'ERROR');
        }
    }

    /**
     * Returns product specified or array of products.
     *
     * See the <i>AmazonProduct</i> class for more information on the returned objects.
     * @param int $num [optional] <p>List index to retrieve the value from. Defaults to 0.</p>
     * @return \Amazon\Sdk\Api\Product|array|boolean Product (or list of Products)
     */
    public function getProduct($num = null)
    {
        if (!isset($this->productList)) {
            return false;
        }
        if (is_numeric($num)) {
            return $this->productList[$num];
        } else {
            return $this->productList;
        }
    }

    /**
     * Parses XML response into array.
     *
     * This is what reads the response XML and converts it into an array.
     * @param SimpleXMLObject $xml <p>The XML response from Amazon.</p>
     * @return boolean <b>FALSE</b> if no XML data is found
     */
    protected function parseXML($xml)
    {
        if (!$xml) {
            return false;
        }

        foreach ($xml->children() as $x) {
            if ($x->getName() == 'ResponseMetadata') {
                continue;
            }
            $temp = (array)$x->attributes();
            if (isset($temp['@attributes']['status']) && $temp['@attributes']['status'] != 'Success') {
                $this->log("Product return was not successful", 'DEBUG');
            }
            if (isset($x->Products)) {
                foreach ($x->Products->children() as $z) {
                    $this->productList[$this->index] =
                        new \Amazon\Sdk\Api\Product(
                            $z,
                            $this->config,
                            $this->logger,
                            $this->mockMode,
                            $this->mockFiles
                        );
                    $this->index++;
                }
            } elseif ($x->getName() == 'GetProductCategoriesForSKUResult' and
                $x->getName() == 'GetProductCategoriesForASINResult'
            ) {
                $this->productList[$this->index] =
                    new \Amazon\Sdk\Api\Product(
                        $x,
                        $this->config,
                        $this->logger,
                        $this->mockMode,
                        $this->mockFiles
                    );
                $this->index++;
            } else {
                foreach ($x->children() as $z) {
                    if ($z->getName() == 'Error') {
                        $error = (string)$z->Message;
                        $this->productList['Error'] = $error;
                        $this->log("Product Error: $error", 'DEBUG');
                    } elseif ($z->getName() != 'Product') {
                        $this->productList[$z->getName()] = (string)$z;
                        $this->log("Special case: " . $z->getName(), 'WARNING');
                    } else {
                        $this->productList[$this->index] =
                            new \Amazon\Sdk\Api\Product(
                                $z,
                                $this->config,
                                $this->logger,
                                $this->mockMode,
                                $this->mockFiles
                            );
                        $this->index++;
                    }
                }
            }
        }
        return true;
    }
}
