<?php
/**
 * CedCommerce
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the End User License Agreement (EULA)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://cedcommerce.com/license-agreement.txt
 *
 * @category    Ced
 * @package     Ced_Amazon_sdk
 * @author      CedCommerce Core Team <connect@cedcommerce.com>
 * @copyright   Copyright © 2018 CedCommerce. All rights reserved.
 * @license     EULA http://cedcommerce.com/license-agreement.txt
 */

namespace Amazon\Sdk\Api\Merchant\Fulfillment;

/**
 * Submits a request to get shipping label from Shipment Id.
 *
 * This Amazon object can submit a request to Amazon to
 * get a shipping label. In order to get
 * shipping label, Shipment Id is needed.
 */
class GetShipment extends \Amazon\Sdk\Api\Core
{
    private $shippingLabelResponse = [];
    private $shipmentId;
    private $index = 0;
    private $trackingId;
    private $status;

    /**
     * AmazonMerchantFulfillment GetShipment gets shipping label. You need a shipment Id.
     *
     * The parameters are passed to the parent constructor, which are
     * in turn passed to the AmazonCore constructor. See it for more information
     * on these parameters and common methods.
     * @param \Amazon\Sdk\Api\ConfigInterface $config = null,
     * @param \Psr\Log\LoggerInterface $logger = null,
     * @param boolean $mockMode [optional] <p>This is a flag for enabling Mock Mode.
     * This defaults to <b>FALSE</b>.</p>
     * @param array|string $mockFiles [optional] <p>The files (or file) to use in Mock Mode.</p>
     * @throws \Amazon\Sdk\Api\Exception\InvalidConfigValue
     */
    public function __construct(
        \Amazon\Sdk\Api\ConfigInterface $config = null,
        \Psr\Log\LoggerInterface $logger = null,
        $mockMode = false,
        $mockFiles = null
    )
    {
        parent::__construct($config, $logger, $mockMode, $mockFiles);

        $this->options['Action'] = 'GetShipment';
    }

    /**
     * Sets the Shipment ID. (Required)
     *
     * This method sets the Shipment ID to be sent in the next request.
     * This parameter is required for getting shipping label.
     * Shipment ID can be generated using the <i>CreateShipment</i> object.
     * @param string $s <p>Maximum 50 characters.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setShipmentId($s)
    {
        if (is_string($s)) {
            $this->options['ShipmentId'] = $s;
            return true;
        }

        return false;
    }

    /**
     * Fetches Shipping Label.
     *
     * @return boolean <b>FALSE</b> if error is returned
     * @throws \Amazon\Sdk\Api\Exception\InvalidConfigValue
     */
    public function fetchShippingLabel()
    {
        if (!array_key_exists('ShipmentId', $this->options)) {
            $this->log("ShipmentId must be set in order to get shipping label", 'WARNING');
            return false;
        }

        $url = $this->urlbase . $this->urlbranch;

        $query = $this->genQuery();
        $path = $this->options['Action'] . 'Result';
        if ($this->mockMode) {
            $response = $this->fetchMockFile(self::MOCK_FILE_MERCHANT_FULFILLMENT_GET_SHIPMENT);
            if ($response) {
                $xml = $response->$path;
            } else {
                return false;
            }
        } else {
            $response = $this->sendRequest($url, ['Post' => $query]);
            if (!$this->checkResponse($response)) {
                return false;
            }
            $xml = simplexml_load_string($response['body'])->$path;
        }
        $this->parseXML($xml);

        return true;
    }

    /**
     * Parses XML response into array.
     *
     * This is what reads the response XML and converts it into an array.
     * @param \SimpleXMLElement $xml <p>The XML response from Amazon.</p>
     * @return boolean <b>FALSE</b> if no XML data is found
     * @throws \Amazon\Sdk\Api\Exception\InvalidConfigValue
     */
    protected function parseXML($xml)
    {
        if (!$xml) {
            return false;
        }

        if (isset($xml->Shipment)) {
            $data = $xml->Shipment;
            $this->shippingLabelResponse['Insurance'] = [
                'CurrencyCode' => (string)$data->Insurance->CurrencyCode,
                'Amount' => (string)$data->Insurance->Amount
            ];
            $this->shippingLabelResponse['ShipToAddress'] = [
                'City' => (string)$data->ShipToAddress->City,
                'Phone' => (string)$data->ShipToAddress->Phone,
                'CountryCode' => (string)$data->ShipToAddress->CountryCode,
                'PostalCode' => (string)$data->ShipToAddress->PostalCode,
                'Name' => (string)$data->ShipToAddress->Name,
                'AddressLine1' => (string)$data->ShipToAddress->AddressLine1,
                'StateOrProvinceCode' => (string)$data->ShipToAddress->StateOrProvinceCode,
                'Email' => (string)$data->ShipToAddress->Email,
            ];
            $this->shippingLabelResponse['AmazonOrderId'] = (string)$data->AmazonOrderId;
            $this->shippingLabelResponse['Weight'] = [
                'Value' => (string)$data->Weight->Value,
                'Unit' => (string)$data->Weight->Unit
            ];
            $this->shippingLabelResponse['Label'] = [
                'CustomTextForLabel' => (string)$data->Label->CustomTextForLabel,
                'LabelFormat' => (string)$data->Label->LabelFormat,
                'FileContents' => [
                    'Checksum' => (string)$data->Label->FileContents->Checksum,
                    'Contents' => (string)$data->Label->FileContents->Contents,
                    'FileType' => (string)$data->Label->FileContents->FileType,
                    'LabelFormat' => (string)$data->Label->FileContents->LabelFormat
                ],
                'Dimensions' => [
                    'Width' => (string)$data->Label->Dimensions->Width,
                    'Length' => (string)$data->Label->Dimensions->Length,
                    'Unit' => (string)$data->Label->Dimensions->Unit,
                ],
                'StandardIdForLabel' => (string)$data->Label->StandardIdForLabel
            ];
            $this->shippingLabelResponse['ShippingService'] = [
                'CarrierName' => (string)$data->ShippingService->CarrierName,
                'ShippingServiceOptions' => [
                    'CarrierWillPickUp' => (string)$data->ShippingService->ShippingServiceOptions->CarrierWillPickUp,
                    'DeclaredValue' => [
                        'CurrencyCode' => (string)$data->ShippingService->ShippingServiceOptions->DeclaredValue->CurrencyCode,
                        'Amount' => (string)$data->ShippingService->ShippingServiceOptions->DeclaredValue->Amount
                    ],
                    'DeliveryExperience' => (string)$data->ShippingService->ShippingServiceOptions->DeliveryExperience
                ],
                'ShippingServiceId' => (string)$data->ShippingService->ShippingServiceId,
                'Rate' => [
                    'CurrencyCode' => (string)$data->ShippingService->Rate->CurrencyCode,
                    'Amount' => (string)$data->ShippingService->Rate->Amount
                ],
                'LatestEstimatedDeliveryDate' => (string)$data->ShippingService->LatestEstimatedDeliveryDate,
                'EarliestEstimatedDeliveryDate' => (string)$data->ShippingService->EarliestEstimatedDeliveryDate,
                'ShippingServiceOfferId' => (string)$data->ShippingService->ShippingServiceOfferId,
                'ShipDate' => (string)$data->ShippingService->ShipDate,
                'ShippingServiceName' => (string)$data->ShippingService->ShippingServiceName
            ];
            $this->shippingLabelResponse['PackageDimensions'] = [
                'Width' => (string)$data->PackageDimensions->Width,
                'Length' => (string)$data->PackageDimensions->Length,
                'Unit' => (string)$data->PackageDimensions->Unit,
                'Height' => (string)$data->PackageDimensions->Height,
            ];
            $this->shippingLabelResponse['CreatedDate'] = (string)$data->CreatedDate;
            $this->shippingLabelResponse['ShipFromAddress'] = [
                'City' => (string)$data->ShipFromAddress->City,
                'Phone' => (string)$data->ShipFromAddress->Phone,
                'CountryCode' => (string)$data->ShipFromAddress->CountryCode,
                'PostalCode' => (string)$data->ShipFromAddress->PostalCode,
                'Name' => (string)$data->ShipFromAddress->Name,
                'AddressLine1' => (string)$data->ShipFromAddress->AddressLine1,
                'StateOrProvinceCode' => (string)$data->ShipFromAddress->StateOrProvinceCode
            ];
            $this->shippingLabelResponse['ShipmentId'] = (string)$data->ShipmentId;
            foreach ($data->ItemList->children() as $key => $item) {
                $this->shippingLabelResponse['ItemList'][$this->index]['Item'] = [
                    'OrderItemId' => (string)$item->Item->OrderItemId,
                    'Quantity' => (string)$item->Item->Quantity,
                    'ItemWeight' => (string)$item->Item->ItemWeight,
                    'ItemDescription' => (string)$item->Item->ItemDescription,
                ];
                $this->index++;
            }
            $this->shippingLabelResponse['Status'] = (string)$data->Status;
            $this->shippingLabelResponse['TrackingId'] = (string)$data->TrackingId;
        }
        return true;
    }

    /**
     * Returns the list of shipping Label Response.
     * @return array|boolean shipping label response, or <b>FALSE</b> if list not filled yet
     */
    public function getShippingLabelResponse()
    {
        if (isset($this->shippingLabelResponse)) {
            return $this->shippingLabelResponse;
        }
        return false;
    }

    public function getTrackingId()
    {
        if (isset($this->shippingLabelResponse['TrackingId']) && $this->shippingLabelResponse['TrackingId']) {
            $this->trackingId = $this->shippingLabelResponse['TrackingId'];
            return $this->trackingId;
        } else {
            return false;
        }
    }

    public function getStatus()
    {
        if (isset($this->shippingLabelResponse['Status']) && $this->shippingLabelResponse['Status']) {
            $this->status = $this->shippingLabelResponse['Status'];
            return $this->status;
        } else {
            return false;
        }
    }

    /**
     * Saves Shipping Label in provided path.
     * @param string $gzipFile <p>File path for gzip file </p>
     * @param string $pdfFile <p>File path for pdf file </p>
     * @return boolean
     */
    public function getShippingLabel($gzipFile, $pdfFile)
    {
        if ($this->mockMode) {
            $path = realpath(__DIR__ . '/../../test-cases/mock/'.self::MOCK_SHIPPING_LABEL);
            if (file_exists($path)) {
                copy($path, $pdfFile);
            }
        } else {
            if (isset($this->shippingLabelResponse['Label']['FileContents']['Contents'])) {
                $contents = $this->shippingLabelResponse['Label']['FileContents']['Contents'];
                $decodedString = base64_decode($contents);

                //copying data in gz file
                $gzipFileOpened = fopen($gzipFile, 'w');
                fwrite($gzipFileOpened, $decodedString);
                fclose($gzipFileOpened);

                //extracting shipping label
                $buffer_size = 4096;
                $pdfFileOpen = fopen($pdfFile, 'wb');
                $openGzFile = gzopen($gzipFile, 'rb');

                // Keep repeating until the end of the input file
                while (!gzeof($openGzFile)) {
                    // Read buffer-size bytes
                    // Both fwrite and gzread and binary-safe
                    fwrite($pdfFileOpen, gzread($openGzFile, $buffer_size));
                }

                // Files are done, close files
                fclose($pdfFileOpen);
                gzclose($openGzFile);
                return true;
            }
            return false;
        }
    }

    public function getPackingSlip($pdfFile) {
        if ($this->mockMode) {
            $path = realpath(__DIR__ . '/../../test-cases/mock/' . self::MOCK_PACKING_SLIP);
            if (file_exists($path)) {
                copy($path, $pdfFile);
            }
        } else {
            return false;
        }
    }
}
