<?php
/**
 * CedCommerce
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the End User License Agreement (EULA)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://cedcommerce.com/license-agreement.txt
 *
 * @category    Ced
 * @package     Ced_Amazon_sdk
 * @author      CedCommerce Core Team <connect@cedcommerce.com>
 * @copyright   Copyright © 2018 CedCommerce. All rights reserved.
 * @license     EULA http://cedcommerce.com/license-agreement.txt
 */

namespace Amazon\Sdk\Api\Merchant\Fulfillment;

/**
 * Submits a request to get a shipment Id and shipping label.
 *
 * This Amazon object can submit a request to Amazon to
 * get a shipment Id and shipping label. In order to get
 * shipping label, Amazon Order Id is needed.
 */
class CreateShipment extends \Amazon\Sdk\Api\Core
{
    private $shippingLabelResponse = [];
    private $shipmentId;
    private $index = 0;
    private $trackingId;
    private $status;

    /**
     * AmazonMerchantFulfillment CreateShipment gets a shipment Id and shipping label. You need a amazon order ID.
     *
     * The parameters are passed to the parent constructor, which are
     * in turn passed to the AmazonCore constructor. See it for more information
     * on these parameters and common methods.
     * @param \Amazon\Sdk\Api\ConfigInterface $config = null,
     * @param \Psr\Log\LoggerInterface $logger = null,
     * @param boolean $mockMode [optional] <p>This is a flag for enabling Mock Mode.
     * This defaults to <b>FALSE</b>.</p>
     * @param array|string $mockFiles [optional] <p>The files (or file) to use in Mock Mode.</p>
     * @throws \Amazon\Sdk\Api\Exception\InvalidConfigValue
     */
    public function __construct(
        \Amazon\Sdk\Api\ConfigInterface $config = null,
        \Psr\Log\LoggerInterface $logger = null,
        $mockMode = false,
        $mockFiles = null
    )
    {
        parent::__construct($config, $logger, $mockMode, $mockFiles);

        $this->options['Action'] = 'CreateShipment';
    }

    /**
     * Sets the Amazon order ID. (Required)
     *
     * This method sets the Amazon Order ID to be sent in the next request.
     * This parameter is required for getting a shipment Id and shipping label.
     * Amazon order ID can be generated using the <i>GetOrder</i> object.
     * @param string $s <p>Maximum 50 characters.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setAmazonOrderId($s)
    {
        if (is_string($s)) {
            $this->options['ShipmentRequestDetails.AmazonOrderId'] = $s;
            return true;
        }

        return false;
    }

    /**
     * Sets the Seller order ID. (Optional)
     *
     * This method sets the Seller Order ID to be sent in the next request.
     * Seller order ID is a seller-defined order identifier..
     * @param string $s <p>Maximum 64 characters.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setSellerOrderId($s)
    {
        if (is_string($s)) {
            $this->options['ShipmentRequestDetails.SellerOrderId'] = $s;
            return true;
        }

        return false;
    }

    /**
     * Sets the items. (Required)
     *
     * This parameter is required for getting a shipment Id and shipping label from Amazon.
     * The array provided should contain a list of arrays, each with the following fields:
     * <ul>
     * <li><b>OrderItemId</b> An Amazon-defined identifier for an individual item in an order.
     * Used in the XML response to an order query request (Order API/Order XML).</li>
     * <li><b>Quantity</b> - numeric</li>
     * <li><b>ItemWeight</b> (optional array) - The weight of the item.</li>
     * <ul>
     * <li><b>Value</b> - The decimal weight of the object.</li>
     * <li><b>Unit</b> - The unit of measurement for weight.Unit values: ounces, gram. string</li>
     * </ul>
     * <li><b>ItemDescription</b> (optional) - string</li>
     * <li><b>TransparencyCodeList</b> (optional array) </li>
     * <ul>
     * <li><b>TransparencyCode</b> - string.</li>
     * </ul>
     * @param array $a <p>See above.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setItems($a)
    {
        if (is_null($a) || is_string($a) || !$a) {
            $this->log("Tried to set Items to invalid values", 'WARNING');
            return false;
        }
        $this->resetItems();
        $i = 1;
        foreach ($a as $x) {
            if (is_array($x) and
                array_key_exists('OrderItemId', $x) && array_key_exists('Quantity', $x)) {
                $this->options['ShipmentRequestDetails.ItemList.Item.' . $i . '.OrderItemId'] = $x['OrderItemId'];
                $this->options['ShipmentRequestDetails.ItemList.Item.' . $i . '.Quantity'] = $x['Quantity'];
                if (array_key_exists('ItemWeight', $x)) {
                    $this->options['ShipmentRequestDetails.ItemList.Item.' . $i . '.ItemWeight.Value'] = $x['ItemWeight']['Value'];
                    $this->options['ShipmentRequestDetails.ItemList.Item.' . $i . '.ItemWeight.Unit'] = $x['ItemWeight']['Unit'];
                }
                if (array_key_exists('ItemDescription', $x)) {
                    $this->options['ShipmentRequestDetails.ItemList.Item.' . $i . '.ItemDescription'] = $x['ItemDescription'];
                }
                if (array_key_exists('TransparencyCodeList', $x)) {
                    $this->options['ShipmentRequestDetails.ItemList.Item.' . $i . '.TransparencyCodeList.TransparencyCode'] =
                        $x['TransparencyCodeList']['TransparencyCode'];
                }
                $i++;
            } else {
                $this->resetItems();
                $this->log("Tried to set Items with invalid array", 'WARNING');
                return false;
            }
        }
        return true;
    }

    /**
     * Resets the item options.
     *
     * Since the list of items is a required parameter, these options should not be removed
     * without replacing them, so this method is not public.
     */
    protected function resetItems()
    {
        foreach ($this->options as $op => $junk) {
            if (preg_match("#Item#", $op)) {
                unset($this->options[$op]);
            }
        }
    }

    /**
     * Sets the address. (Required)
     *
     * This method sets the shipping from address to be sent in the next request.
     * This parameter is required getting a shipment Id and shipping label from Amazon.
     * The array provided should have the following fields:
     * <ul>
     * <li><b>Name</b> - max: 50 char</li>
     * <li><b>AddressLine1</b> - max: 180 char</li>
     * <li><b>AddressLine2</b> (optional) - max: 60 char</li>
     * <li><b>AddressLine3</b> (optional) - max: 60 char</li>
     * <li><b>DistrictOrCounty</b> (optional) - max: 150 char</li>
     * <li><b>Email</b> - max: 256 char</li>
     * <li><b>City</b> - max: 30 char</li>
     * <li><b>StateOrProvinceCode</b> - max: 30 char</li>
     * <li><b>CountryCode</b> - 2 digits</li>
     * <li><b>PostalCode</b> - max: 30 char</li>
     * <li><b>Phone</b> - max: 30 char</li>
     * </ul>
     * @param array $a <p>See above.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setAddress($a)
    {
        if (is_null($a) || is_string($a) || !$a) {
            $this->log("Tried to set address to invalid values", 'WARNING');
            return false;
        }
        $this->resetAddress();
        if (is_array($a) and
            array_key_exists('Name', $a) && array_key_exists('AddressLine1', $a) &&
            array_key_exists('Email', $a) && array_key_exists('City', $a) &&
            array_key_exists('StateOrProvinceCode', $a) && array_key_exists('PostalCode', $a) &&
            array_key_exists('CountryCode', $a) && array_key_exists('Phone', $a)) {
            $this->options['ShipmentRequestDetails.ShipFromAddress.Name'] = $a['Name'];
            $this->options['ShipmentRequestDetails.ShipFromAddress.AddressLine1'] = $a['AddressLine2'];
            if (array_key_exists('AddressLine2', $a)) {
                $this->options['ShipmentRequestDetails.ShipFromAddress.AddressLine2'] = $a['AddressLine2'];
            }
            if (array_key_exists('AddressLine3', $a)) {
                $this->options['ShipmentRequestDetails.ShipFromAddress.AddressLine3'] = $a['AddressLine3'];
            }
            if (array_key_exists('DistrictOrCounty', $a)) {
                $this->options['ShipFromAddress.DistrictOrCounty'] = $a['DistrictOrCounty'];
            }
            $this->options['ShipmentRequestDetails.ShipFromAddress.Email'] = $a['Email'];
            $this->options['ShipmentRequestDetails.ShipFromAddress.City'] = $a['City'];
            $this->options['ShipmentRequestDetails.ShipFromAddress.StateOrProvinceCode'] = $a['StateOrProvinceCode'];
            $this->options['ShipmentRequestDetails.ShipFromAddress.CountryCode'] = $a['CountryCode'];
            $this->options['ShipmentRequestDetails.ShipFromAddress.PostalCode'] = $a['PostalCode'];
            if (array_key_exists('Phone', $a)) {
                $this->options['ShipmentRequestDetails.ShipFromAddress.Phone'] = $a['Phone'];
            }
        } else {
            $this->resetAddress();
            $this->log("Tried to set Address with invalid array", 'WARNING');
            return false;
        }
        return true;
    }

    /**
     * Resets the address options.
     *
     * Since address is a required parameter, these options should not be removed
     * without replacing them, so this method is not public.
     */
    protected function resetAddress()
    {
        unset($this->options['ShipFromAddress.Name']);
        unset($this->options['ShipFromAddress.AddressLine1']);
        unset($this->options['ShipFromAddress.AddressLine2']);
        unset($this->options['ShipFromAddress.AddressLine3']);
        unset($this->options['ShipFromAddress.DistrictOrCounty']);
        unset($this->options['ShipFromAddress.Email']);
        unset($this->options['ShipFromAddress.City']);
        unset($this->options['ShipFromAddress.StateOrProvinceCode']);
        unset($this->options['ShipFromAddress.CountryCode']);
        unset($this->options['ShipFromAddress.PostalCode']);
        unset($this->options['ShipFromAddress.PhoneNumber']);
    }

    /**
     * Sets the package Dimensions. (Required)
     *
     * This method sets the package dimensions to be sent in the next request.
     * This parameter is required for getting a shipment Id and shipping label from Amazon.
     * The array provided should have the following fields:
     * <ul>
     * <li><b>Length</b> - decimal</li>
     * <li><b>Width</b> - decimal</li>
     * <li><b>Height</b> - decimal</li>
     * <li><b>Unit</b>  - string</li>
     * <li><b>PredefinedPackageDimensions</b> (optional)- string.
     * If you specify the PredefinedPackageDimensions request parameter,
     * you must not specify the Length/Width/Height/Unit request parameter</li>
     * </ul>
     * @param array $a <p>See above.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setPackageDimension($a)
    {
        if (is_null($a) || is_string($a) || !$a) {
            $this->log("Tried to set package dimension to invalid values", 'WARNING');
            return false;
        }
        $this->resetPackageDimension();
        if (is_array($a) and
            ((array_key_exists('Length', $a) && array_key_exists('Width', $a) &&
                    array_key_exists('Height', $a) && array_key_exists('Unit', $a)) ||
                (array_key_exists('ShipmentRequestDetails.PredefinedPackageDimensions', $a)))) {

            if (array_key_exists('Length', $a)) {
                $this->options['ShipmentRequestDetails.PackageDimensions.Length'] = $a['Length'];
            }
            if (array_key_exists('Width', $a)) {
                $this->options['ShipmentRequestDetails.PackageDimensions.Width'] = $a['Width'];
            }
            if (array_key_exists('Height', $a)) {
                $this->options['ShipmentRequestDetails.PackageDimensions.Height'] = $a['Height'];
            }
            if (array_key_exists('Unit', $a)) {
                $this->options['ShipmentRequestDetails.PackageDimensions.Unit'] = $a['Unit'];
            }
            if (array_key_exists('PredefinedPackageDimensions', $a)) {
                $this->options['ShipmentRequestDetails.PackageDimensions.PredefinedPackageDimensions'] = $a['PredefinedPackageDimensions'];
            }
        } else {
            $this->resetPackageDimension();
            $this->log("Tried to set package dimension with invalid array", 'WARNING');
            return false;
        }
        return true;
    }

    /**
     * Resets the package dimension options.
     *
     * Since package dimension is a required parameter, these options should not be removed
     * without replacing them, so this method is not public.
     */
    protected function resetPackageDimension()
    {
        unset($this->options['ShipmentRequestDetails.PackageDimensions.Length']);
        unset($this->options['ShipmentRequestDetails.PackageDimensions.Width']);
        unset($this->options['ShipmentRequestDetails.PackageDimensions.Height']);
        unset($this->options['ShipmentRequestDetails.PackageDimensions.Unit']);
        unset($this->options['ShipmentRequestDetails.PackageDimensions.PredefinedPackageDimensions']);

    }

    /**
     * Sets Weight. (Required)
     *
     * This method sets the Weight to be sent in the next request.
     * This parameter is required for getting a shipment Id and shipping label from Amazon.
     * The array provided should have the following fields:
     * <ul>
     * <li><b>Value</b> - decimal</li>
     * <li><b>Unit</b> - Unit values: ounces, grams. String</li>
     * </ul>
     * @param array $a <p>See above.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setWeight($a)
    {
        if (is_null($a) || is_string($a) || !$a) {
            $this->log("Tried to set weight to invalid values", 'WARNING');
            return false;
        }
        $this->resetWeight();
        if (is_array($a) and
            array_key_exists('Value', $a) && array_key_exists('Unit', $a)) {
            $this->options['ShipmentRequestDetails.Weight.Value'] = $a['Value'];
            $this->options['ShipmentRequestDetails.Weight.Unit'] = $a['Unit'];
        } else {
            $this->resetWeight();
            $this->log("Tried to set weight with invalid array", 'WARNING');
            return false;
        }
        return true;
    }

    /**
     * Resets the weight options.
     *
     * Since weight is a required parameter, these options should not be removed
     * without replacing them, so this method is not public.
     */
    protected function resetWeight()
    {
        unset($this->options['ShipmentRequestDetails.Weight.Value']);
        unset($this->options['ShipmentRequestDetails.Weight.Unit']);

    }

    /**
     * Sets the MustArriveByDate. (Optional)
     *
     * This method sets the MustArriveByDate to be sent in the next request.
     * @param string $s <p>Maximum 50 characters.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setMustArriveByDate($s)
    {
        if (is_string($s)) {
            $this->options['ShipmentRequestDetails.MustArriveByDate'] = $this->genTime($s);
            return true;
        }

        return false;
    }

    /**
     * Sets the ShipDate. (Optional)
     *
     * This method sets the ShipDate to be sent in the next request.
     * @param string $s <p>Maximum 50 characters.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setShipDate($s)
    {
        if (is_string($s)) {
            $this->options['ShipmentRequestDetails.ShipDate'] = $this->genTime($s);
            return true;
        }

        return false;
    }

    /**
     * Sets the Shipping Service options. (Required)
     *
     * This method sets the Shipping Service options to be sent in the next request.
     * This parameter is required getting a shipment Id and shipping label from Amazon.
     * The array provided should have the following fields:
     * <ul>
     * <li><b>DeliveryExperience</b> - DeliveryExperience values:</li>
     * DeliveryConfirmationWithAdultSignature - Delivery confirmation with adult signature.
     * DeliveryConfirmationWithSignature - Delivery confirmation with signature. Required for DPD (UK).
     * DeliveryConfirmationWithoutSignature - Delivery confirmation without signature.
     * NoTracking - No delivery confirmation.</li>
     * <li><b>DeclaredValue</b> (optional array) - </li>
     * <ul>
     * <li><b>CurrencyCode</b> - string </li>
     * <li><b>Amount</b> - decimal <li>
     * </ul>
     * <li><b>Height</b> - decimal</li>
     * <li><b>Unit</b>  - string</li>
     * <li><b>CarrierWillPickUp</b> - boolean.
     * </ul>
     * @param array $a <p>See above.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setShippingServiceOptions($a)
    {
        if (is_null($a) || is_string($a) || !$a) {
            $this->log("Tried to set shipping service options to invalid values", 'WARNING');
            return false;
        }
        $this->resetShippingServiceOptions();
        if (is_array($a) and
            array_key_exists('DeliveryExperience', $a) && array_key_exists('CarrierWillPickUp', $a)) {
            $this->options['ShipmentRequestDetails.ShippingServiceOptions.DeliveryExperience'] = $a['DeliveryExperience'];
            $this->options['ShipmentRequestDetails.ShippingServiceOptions.CarrierWillPickUp'] = $a['CarrierWillPickUp'];
            if (array_key_exists('DeclaredValue', $a)) {
                $this->options['ShipmentRequestDetails.ShippingServiceOptions.DeclaredValue.CurrencyCode'] = $a['DeclaredValue']['CurrencyCode'];
                $this->options['ShipmentRequestDetails.ShippingServiceOptions.DeclaredValue.Amount'] = $a['DeclaredValue']['Amount'];
            }
        } else {
            $this->resetShippingServiceOptions();
            $this->log("Tried to set shipping service options with invalid array", 'WARNING');
            return false;
        }
        return true;
    }

    /**
     * Resets the weight options.
     *
     * Since weight is a required parameter, these options should not be removed
     * without replacing them, so this method is not public.
     */
    protected function resetShippingServiceOptions()
    {
        unset($this->options['ShipmentRequestDetails.ShippingServiceOptions.DeliveryExperience']);
        unset($this->options['ShipmentRequestDetails.ShippingServiceOptions.CarrierWillPickUp']);
        unset($this->options['ShipmentRequestDetails.ShippingServiceOptions.DeclaredValue.CurrencyCode']);
        unset($this->options['ShipmentRequestDetails.ShippingServiceOptions.DeclaredValue.Amount']);
    }

    /**
     * Sets the Shipping Service ID. (Required)
     *
     * This method sets the Shipping service ID to be sent in the next request.
     * This parameter is required for getting a shipment Id and shipping label.
     * Shipping Service ID can be generated using the <i>GetEligibleShippingServices</i> object.
     * @param string $s <p>Maximum 50 characters.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setShippingServiceId($s)
    {
        if (is_string($s)) {
            $this->options['ShippingServiceId'] = $s;
            return true;
        }
        return false;
    }

    /**
     * Sets the Shipping Service Offer ID. (Optional)
     *
     * This method sets the Shipping service Offer ID to be sent in the next request.
     * Shipping Service Offer ID can be generated using the <i>GetEligibleShippingServices</i> object.
     * @param string $s <p>Maximum 50 characters.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setShippingServiceOfferId($s)
    {
        if (is_string($s)) {
            $this->options['ShippingServiceOfferId'] = $s;
            return true;
        }
        return false;
    }

    /**
     * Sets the Hazmat Type. (Optional)
     *
     * This method sets the Hazmat type to be sent in the next request.
     * HazmatType enumeration :
     * None:This package does not contain hazardous material.
     * LQHazmat:This package contains limited quantities of hazardous material.
     * @param string $s <p>Maximum 50 characters.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setHazmatType($s)
    {
        if (is_string($s)) {
            $this->options['HazmatType'] = $s;
            return true;
        }
        return false;
    }

    /**
     * Sets the LabelFormatOption. (Optional)
     *
     * This method sets the LabelFormatOption to be sent in the next request.
     * @param boolean $s
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setLabelFormatOption($s)
    {
        if (is_bool($s)) {
            $this->options['LabelFormatOption.IncludePackingSlipWithLabel'] = $s;
            return true;
        }
        return false;
    }

    /**
     * Fetches Shipping Label.
     *
     * @return boolean <b>FALSE</b> if error is returned
     * @throws \Amazon\Sdk\Api\Exception\InvalidConfigValue
     */
    public function fetchShippingLabel()
    {
        if (!array_key_exists('ShipmentRequestDetails.AmazonOrderId', $this->options)) {
            $this->log("Amazon OrderID must be set in order to get shipping label", 'WARNING');
            return false;
        }
        if (!array_key_exists('ShipmentRequestDetails.ItemList.Item.1.OrderItemId', $this->options)) {
            $this->log("Item list must be set in order to get shipping label", 'WARNING');
            return false;
        }
        if (!array_key_exists('ShipmentRequestDetails.ShipFromAddress.Name', $this->options)) {
            $this->log("ShipFromAddress must be set in order to get shipping label", 'WARNING');
            return false;
        }
        if (!array_key_exists('ShipmentRequestDetails.Weight.Value', $this->options)) {
            $this->log("Weight must be set in order to get shipping label", 'WARNING');
            return false;
        }
        if (!array_key_exists('ShipmentRequestDetails.ShippingServiceOptions.DeliveryExperience', $this->options)) {
            $this->log("ShippingServiceOptions must be set in order to get shipping label", 'WARNING');
            return false;
        }
        if (!array_key_exists('ShippingServiceId', $this->options)) {
            $this->log("ShippingServiceId must be set in order to get shipping label", 'WARNING');
            return false;
        }

        $url = $this->urlbase . $this->urlbranch;

        $query = $this->genQuery();
        $path = $this->options['Action'] . 'Result';
        if ($this->mockMode) {
            $response = $this->fetchMockFile(self::MOCK_FILE_MERCHANT_FULFILLMENT_CREATE_SHIPMENT);
            if ($response) {
                $xml = $response->$path;
            } else {
                return false;
            }
        } else {
            $response = $this->sendRequest($url, ['Post' => $query]);
            if (!$this->checkResponse($response)) {
                return false;
            }
            $xml = simplexml_load_string($response['body'])->$path;
        }
        $this->parseXML($xml);

        return true;
    }

    /**
     * Parses XML response into array.
     *
     * This is what reads the response XML and converts it into an array.
     * @param \SimpleXMLElement $xml <p>The XML response from Amazon.</p>
     * @return boolean <b>FALSE</b> if no XML data is found
     * @throws \Amazon\Sdk\Api\Exception\InvalidConfigValue
     */
    protected function parseXML($xml)
    {
        if (!$xml) {
            return false;
        }

        if (isset($xml->Shipment)) {
            $data = $xml->Shipment;
            $this->shippingLabelResponse['Insurance'] = [
                'CurrencyCode' => (string)$data->Insurance->CurrencyCode,
                'Amount' => (string)$data->Insurance->Amount
            ];
            $this->shippingLabelResponse['ShipToAddress'] = [
                'City' => (string)$data->ShipToAddress->City,
                'Phone' => (string)$data->ShipToAddress->Phone,
                'CountryCode' => (string)$data->ShipToAddress->CountryCode,
                'PostalCode' => (string)$data->ShipToAddress->PostalCode,
                'Name' => (string)$data->ShipToAddress->Name,
                'AddressLine1' => (string)$data->ShipToAddress->AddressLine1,
                'StateOrProvinceCode' => (string)$data->ShipToAddress->StateOrProvinceCode,
                'Email' => (string)$data->ShipToAddress->Email,
            ];
            $this->shippingLabelResponse['AmazonOrderId'] = (string)$data->AmazonOrderId;
            $this->shippingLabelResponse['Weight'] = [
                'Value' => (string)$data->Weight->Value,
                'Unit' => (string)$data->Weight->Unit
            ];
            $this->shippingLabelResponse['Label'] = [
                'CustomTextForLabel' => (string)$data->Label->CustomTextForLabel,
                'LabelFormat' => (string)$data->Label->LabelFormat,
                'FileContents' => [
                    'Checksum' => (string)$data->Label->FileContents->Checksum,
                    'Contents' => (string)$data->Label->FileContents->Contents,
                    'FileType' => (string)$data->Label->FileContents->FileType,
                    'LabelFormat' => (string)$data->Label->FileContents->LabelFormat
                ],
                'Dimensions' => [
                    'Width' => (string)$data->Label->Dimensions->Width,
                    'Length' => (string)$data->Label->Dimensions->Length,
                    'Unit' => (string)$data->Label->Dimensions->Unit,
                ],
                'StandardIdForLabel' => (string)$data->Label->StandardIdForLabel
            ];
            $this->shippingLabelResponse['ShippingService'] = [
                'CarrierName' => (string)$data->ShippingService->CarrierName,
                'ShippingServiceOptions' => [
                    'CarrierWillPickUp' => (string)$data->ShippingService->ShippingServiceOptions->CarrierWillPickUp,
                    'DeclaredValue' => [
                        'CurrencyCode' => (string)$data->ShippingService->ShippingServiceOptions->DeclaredValue->CurrencyCode,
                        'Amount' => (string)$data->ShippingService->ShippingServiceOptions->DeclaredValue->Amount
                    ],
                    'DeliveryExperience' => (string)$data->ShippingService->ShippingServiceOptions->DeliveryExperience
                ],
                'ShippingServiceId' => (string)$data->ShippingService->ShippingServiceId,
                'Rate' => [
                    'CurrencyCode' => (string)$data->ShippingService->Rate->CurrencyCode,
                    'Amount' => (string)$data->ShippingService->Rate->Amount
                ],
                'LatestEstimatedDeliveryDate' => (string)$data->ShippingService->LatestEstimatedDeliveryDate,
                'EarliestEstimatedDeliveryDate' => (string)$data->ShippingService->EarliestEstimatedDeliveryDate,
                'ShippingServiceOfferId' => (string)$data->ShippingService->ShippingServiceOfferId,
                'ShipDate' => (string)$data->ShippingService->ShipDate,
                'ShippingServiceName' => (string)$data->ShippingService->ShippingServiceName
            ];
            $this->shippingLabelResponse['PackageDimensions'] = [
                'Width' => (string)$data->PackageDimensions->Width,
                'Length' => (string)$data->PackageDimensions->Length,
                'Unit' => (string)$data->PackageDimensions->Unit,
                'Height' => (string)$data->PackageDimensions->Height,
            ];
            $this->shippingLabelResponse['CreatedDate'] = (string)$data->CreatedDate;
            $this->shippingLabelResponse['ShipFromAddress'] = [
                  'City' => (string)$data->ShipFromAddress->City,
                  'Phone' => (string)$data->ShipFromAddress->Phone,
                  'CountryCode' => (string)$data->ShipFromAddress->CountryCode,
                  'PostalCode' => (string)$data->ShipFromAddress->PostalCode,
                  'Name' => (string)$data->ShipFromAddress->Name,
                  'AddressLine1' => (string)$data->ShipFromAddress->AddressLine1,
                  'StateOrProvinceCode' => (string)$data->ShipFromAddress->StateOrProvinceCode
            ];
            $this->shippingLabelResponse['ShipmentId'] = (string)$data->ShipmentId;
            foreach ($data->ItemList->children() as $key => $item) {
                $this->shippingLabelResponse['ItemList'][$this->index]['Item'] = [
                      'OrderItemId' => (string)$item->Item->OrderItemId,
                      'Quantity' => (string)$item->Item->Quantity,
                      'ItemWeight' => (string)$item->Item->ItemWeight,
                      'ItemDescription' => (string)$item->Item->ItemDescription,
                ];
                $this->index++;
            }
            $this->shippingLabelResponse['Status'] = (string)$data->Status;
            $this->shippingLabelResponse['TrackingId'] = (string)$data->TrackingId;
        }
        return true;
    }

    /**
     * Returns the list of shipping Label Response.
     * @return array|boolean shipping label response, or <b>FALSE</b> if list not filled yet
     */
    public function getShippingLabelResponse()
    {
        if (isset($this->shippingLabelResponse)) {
            return $this->shippingLabelResponse;
        }
        return false;
    }

    /**
     * Returns the Shipment Id.
     * @return string|boolean Shipment Id, or <b>FALSE</b> if shipment Id not found
     */
    public function getShipmentId()
    {
        if (isset($this->shippingLabelResponse['ShipmentId']) &&
            $this->shippingLabelResponse['ShipmentId']) {
            $this->shipmentId = $this->shippingLabelResponse['ShipmentId'];
            return $this->shipmentId;
        } else {
            return false;
        }
    }

    public function getTrackingId()
    {
        if (isset($this->shippingLabelResponse['TrackingId']) && $this->shippingLabelResponse['TrackingId']) {
            $this->trackingId = $this->shippingLabelResponse['TrackingId'];
            return $this->trackingId;
        } else {
            return false;
        }
    }

    public function getStatus()
    {
        if (isset($this->shippingLabelResponse['Status']) && $this->shippingLabelResponse['Status']) {
            $this->status = $this->shippingLabelResponse['Status'];
            return $this->status;
        } else {
            return false;
        }
    }

    /**
     * Saves Shipping Label in provided path.
     * @param string $gzipFile <p>File path for gzip file </p>
     * @param string $pdfFile <p>File path for pdf file </p>
     * @return boolean
     */
    public function getShippingLabel($gzipFile, $pdfFile)
    {
        if ($this->mockMode) {
            $path = realpath(__DIR__ . '/../../test-cases/mock/'.self::MOCK_SHIPPING_LABEL);
            if (file_exists($path)) {
                copy($path, $pdfFile);
            }
        } else {
            if (isset($this->shippingLabelResponse['Label']['FileContents']['Contents'])) {
                $contents = $this->shippingLabelResponse['Label']['FileContents']['Contents'];
                $decodedString = base64_decode($contents);

                //copying data in gz file
                $gzipFileOpened = fopen($gzipFile, 'w');
                fwrite($gzipFileOpened, $decodedString);
                fclose($gzipFileOpened);

                //extracting shipping label
                $buffer_size = 4096;
                $pdfFileOpen = fopen($pdfFile, 'wb');
                $openGzFile = gzopen($gzipFile, 'rb');

                // Keep repeating until the end of the input file
                while (!gzeof($openGzFile)) {
                    // Read buffer-size bytes
                    // Both fwrite and gzread and binary-safe
                    fwrite($pdfFileOpen, gzread($openGzFile, $buffer_size));
                }

                // Files are done, close files
                fclose($pdfFileOpen);
                gzclose($openGzFile);
                return true;
            }
            return false;
        }
    }

    public function getPackingSlip($pdfFile) {
        if ($this->mockMode) {
            $path = realpath(__DIR__ . '/../../test-cases/mock/' . self::MOCK_PACKING_SLIP);
            if (file_exists($path)) {
                copy($path, $pdfFile);
            }
        } else {
            return false;
        }
    }

}
