<?php

/**
 * CedCommerce
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the End User License Agreement (EULA)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://cedcommerce.com/license-agreement.txt
 *
 * @category    Ced
 * @package     Ced_Amazon
 * @author      CedCommerce Core Team <connect@cedcommerce.com>
 * @copyright   Copyright © 2019 CedCommerce. All rights reserved.
 * @license     EULA http://cedcommerce.com/license-agreement.txt
 */

namespace Amazon\Sdk\Api\Service;

use Amazon\Sdk\Api\Data\PingResultInterface;
use DateTime;
use Psr\Log\LoggerInterface;
use Amazon\Sdk\Api\ConfigInterface;
use Amazon\Sdk\Api\Data\PingResult;

class Ping implements PingInterface
{
    /** @var ConfigInterface  */
    public $config;

    /** @var null|LoggerInterface  */
    public $logger;

    public $client;

    public function __construct(
        ConfigInterface $config,
        LoggerInterface $logger = null
    ) {
        $this->config = $config;
        $this->logger = $logger;
    }

    /**
     * Ping the Amazon API
     * @return \Amazon\Sdk\Api\Data\PingResultInterface
     */
    public function ping()
    {
        $client = $this->getClient();
        $this->addOptions($client);
        return $this->execute($client);
    }

    private function getClient()
    {
        if (!isset($this->client)) {
            $this->client = curl_init();
        }

        return $this->client;
    }

    private function addOptions($client)
    {
        $options = [
            CURLOPT_URL => $this->config->getServiceUrl(),
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_RETURNTRANSFER => true,
        ];

        curl_setopt_array($client, $options);
    }

    private function execute($client)
    {
        /** @var \Amazon\Sdk\Api\Data\PingResultInterface $result */
        $result = new PingResult();
        $result->setStatus(PingResultInterface::STATUS_INACTIVE);
        $timestamp = new DateTime();
        $response = curl_exec($client);

        $value = null;
        if (!empty($response) && is_string($response)) {
            if (preg_match('/timestamp="(.*?)"/', $response, $match) == 1) {
                $value = $match[1];
                $timestamp->modify($value);
                $result->setStatus(PingResultInterface::STATUS_ACTIVE);
            }
        }

        $this->log("Ping service executed.", ['timestamp' => $value]);

        $result->setTimestamp($timestamp);

        return $result;
    }

    private function log($value, $context = [], $type = 'INFO')
    {
        if ($this->logger instanceof LoggerInterface && !$this->config->getMuteLog()) {
            $this->logger->log($type, $value, $context);
        }
    }
}
