<?php
/**
 * Copyright 2013 CPI Group, LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 *
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

namespace Amazon\Sdk\Api\Report;

/**
 * Sends a report request to Amazon.
 *
 * This AmazonReportsCore object makes a request to Amazon to generate a report.
 * In order to do this, a report type is required. Other parameters are also
 * available to limit the scope of the report.
 */
class Request extends \Amazon\Sdk\Api\Report\ReportsCore
{
    /** @var string Open Inventory Report */
    const REPORT_TYPE_INVENTORY = "_GET_FLAT_FILE_OPEN_LISTINGS_DATA_";

    /** @var string All Listings Report */
    const REPORT_TYPE_OPEN_LISTING_ALL_DATA = "_GET_MERCHANT_LISTINGS_ALL_DATA_";
    /** @var string Active Listings Report */
    const REPORT_TYPE_LISTING_ACTIVE_DATA = "_GET_MERCHANT_LISTINGS_DATA_";
    /** @var string Inactive Listings Report */
    const REPORT_TYPE_LISTING_INACTIVE_DATA = "_GET_MERCHANT_LISTINGS_INACTIVE_DATA_";
    /** @var string Open Listings Report */
    const REPORT_TYPE_OPEN_LISTING_COMPACT = "_GET_MERCHANT_LISTINGS_DATA_BACK_COMPAT_";
    /** @var string Open Listings Report Lite */
    const REPORT_TYPE_OPEN_LISTING_LITE = "_GET_MERCHANT_LISTINGS_DATA_LITE_";
    /** @var string Open Listings Report Liter */
    const REPORT_TYPE_OPEN_LISTING_LITER = "_GET_MERCHANT_LISTINGS_DATA_LITER_";
    /** @var string Canceled Listings Report */
    const REPORT_TYPE_CANCELED_LISTING_DATA = "_GET_MERCHANT_CANCELLED_LISTINGS_DATA_";
    /** @var string Sold Listings Report */
    const REPORT_TYPE_SOLD_LISTING_DATA = "_GET_CONVERGED_FLAT_FILE_SOLD_LISTINGS_DATA_";
    /** @var string Listing Quality and Suppressed Listing Report */
    const REPORT_TYPE_DEFECT_LISTING_DATA = "_GET_MERCHANT_LISTINGS_DEFECT_DATA_";
    /** @var string Pan-European Eligibility: FBA ASINs */
    const REPORT_TYPE_EURO_AFN_LISTING_STATUS = "_GET_PAN_EU_OFFER_STATUS_";
    /** @var string Pan-European Eligibility: Self-fulfilled ASINs */
    const REPORT_TYPE_EURO_MFN_LISTING_STATUS  = "_GET_MFN_PAN_EU_OFFER_STATUS_";
    /**
     * @var string Global Expansion Opportunities Report
     * Tab-delimited flat file report that contains products that a seller lists which have a high sales potential
     * in other Amazon marketplaces. This report is only available in the US marketplace.
     */
    const REPORT_TYPE_GLOBAL_OPPORTUNITIES  = "_GET_FLAT_FILE_GEO_OPPORTUNITIES_";
    /**
     * @var string Referral Fee Preview Report
     * Tab-delimited flat file that contains your open listings as well as the price and estimated referral
     * fees for each SKU.
     */
    const REPORT_TYPE_REFERRAL_FEE_PREVIEW  = "_GET_REFERRAL_FEE_PREVIEW_REPORT_";

    /** @var string Requested or Scheduled Flat File Order Report string  */
    const REPORT_TYPE_ORDER_DATA  = "_GET_FLAT_FILE_ORDER_REPORT_DATA_";
    /** @var string Unshipped Orders Report */
    const REPORT_TYPE_UNSHIPPED_ORDER_DATA  = "_GET_FLAT_FILE_ACTIONABLE_ORDER_DATA_";

    /** @var string Flat File All Orders Shipments Report by Order Date */
    const REPORT_TYPE_FBA_SHIPMENTS_DATA = "_GET_AMAZON_FULFILLED_SHIPMENTS_DATA_";
    /** @var string Flat File All Orders Report by Order Date */
    const REPORT_TYPE_FBA_FLAT_ORDER_DATA  = "_GET_FLAT_FILE_ALL_ORDERS_DATA_BY_ORDER_DATE_";
    /** @var string XML All Orders Report by Order Date */
    const REPORT_TYPE_FBA_XML_ORDER_DATA  = "_GET_XML_ALL_ORDERS_DATA_BY_ORDER_DATE_";

    private $response;

    /**
     * AmazonReportRequest sends a report request to Amazon.
     *
     * The parameters are passed to the parent constructor, which are
     * in turn passed to the AmazonCore constructor. See it for more information
     * on these parameters and common methods.
     * @param \Amazon\Sdk\Api\ConfigInterface|null $config ,
     * @param \Psr\Log\LoggerInterface|null $logger ,
     * @param boolean $mockMode [optional] <p>This is a flag for enabling Mock Mode.
     * This defaults to <b>FALSE</b>.</p>
     * @param array|string $mockFiles [optional] <p>The files (or file) to use in Mock Mode.</p>
     */
    public function __construct(
        \Amazon\Sdk\Api\ConfigInterface $config = null,
        \Psr\Log\LoggerInterface $logger = null,
        $mockMode = false,
        $mockFiles = null
    ) {
        parent::__construct($config, $logger, $mockMode, $mockFiles);

        $this->options['Action'] = 'RequestReport';
        $marketplaceIds = $this->config->getMarketplaceId();
        if (isset($marketplaceIds) && !empty($marketplaceIds)) {
            // Applicable for NA, EU only
            $this->setMarketplaceListIds($marketplaceIds);
        } else {
            $this->log("Marketplace List Id is missing", 'ERROR');
        }

        $this->throttleLimit = \Amazon\Sdk\Api\Environment::THROTTLE_LIMIT_REPORTREQUEST;
        $this->throttleTime = \Amazon\Sdk\Api\Environment::THROTTLE_TIME_REPORTREQUEST;
        $this->throttleGroup = 'RequestReport';
    }

    /**
     * Set Start Date
     * @param string $s
     * @return bool
     */
    public function setStartDate($s)
    {
        if (is_string($s) && $s) {
            $s = date('Y-m-d\TH:i:sO', strtotime($s));
            $this->options['StartDate'] = $s;
            return true;
        }
        return false;
    }

    /**
     * Set End Date
     * @param string $s
     * @return bool
     */
    public function setEndDate($s)
    {
        if (is_string($s) && $s) {
            $s = date('Y-m-d\TH:i:sO', strtotime($s));
            $this->options['EndDate'] = $s;
            return true;
        }
        return false;
    }

    /**
     * Sets the report type. (Required)
     *
     * This method sets the report type to be sent in the next request.
     * This parameter is required for fetching the report from Amazon.
     * @param string|integer $s <p>See comment inside for a list of valid values.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setReportType($s)
    {
        if (is_string($s) && $s) {
            $this->options['ReportType'] = $s;
            return true;
        }
        return false;

        /*
         * List of valid Report Types:
         * Listings Reports:
         *      Open Listings Report ~ _GET_FLAT_FILE_OPEN_LISTINGS_DATA_
         *      Open Listings Report ~ _GET_MERCHANT_LISTINGS_DATA_BACK_COMPAT_
         *      Merchant Listings Report ~ _GET_MERCHANT_LISTINGS_DATA_
         *      Merchant Listings Lite Report ~ _GET_MERCHANT_LISTINGS_DATA_LITE_
         *      Merchant Listings Liter Report ~ _GET_MERCHANT_LISTINGS_DATA_LITER_
         *      Canceled Listings Report ~ _GET_MERCHANT_CANCELLED_LISTINGS_DATA_
         *      Quality Listing Report ~ _GET_MERCHANT_LISTINGS_DEFECT_DATA_
         * Order Reports:
         *      Unshipped Orders Report ~ _GET_FLAT_FILE_ACTIONABLE_ORDER_DATA_
         *      Flat File Order Report ~ _GET_FLAT_FILE_ORDER_REPORT_DATA_
         *      Requested Flat File Order Report ~ _GET_FLAT_FILE_ORDERS_DATA_
         *      Flat File Order Report ~ _GET_CONVERGED_FLAT_FILE_ORDER_REPORT_DATA_
         * Order Tracking Reports:
         *      Flat File Orders By Last Update Report ~ _GET_FLAT_FILE_ALL_ORDERS_DATA_BY_LAST_UPDATE_
         *      Flat File Orders By Order Date Report ~ _GET_FLAT_FILE_ALL_ORDERS_DATA_BY_ORDER_DATE_
         *      XML Orders By Last Update Report ~ _GET_XML_ALL_ORDERS_DATA_BY_LAST_UPDATE_
         *      XML Orders By Order Date Report ~ _GET_XML_ALL_ORDERS_DATA_BY_ORDER_DATE_
         * Pending Order Reports:
         *      Flat File Pending Orders Report ~ _GET_FLAT_FILE_PENDING_ORDERS_DATA_
         *      XML Pending Orders Report ~ _GET_PENDING_ORDERS_DATA_
         *      Converged Flat File Pending Orders Report ~ GET_CONVERGED_FLAT_FILE_PENDING_ORDERS_DATA_
         * Performance Reports:
         *      Flat File Feedback Report ~ _GET_SELLER_FEEDBACK_DATA_
         * FBA Reports:
         *      Flat File All Orders Report by Last Update ~ _GET_FLAT_FILE_ALL_ORDERS_DATA_BY_LAST_UPDATE_
         *      Flat File All Orders Report by Order Date ~ _GET_FLAT_FILE_ALL_ORDERS_DATA_BY_ORDER_DATE_
         *      XML All Orders Report by Last Update ~ _GET_XML_ALL_ORDERS_DATA_BY_LAST_UPDATE_
         *      XML All Orders Report by Order Date ~ _GET_XML_ALL_ORDERS_DATA_BY_ORDER_DATE_
         *      FBA Inventory Report ~ _GET_AFN_INVENTORY_DATA_
         *      FBA Fulfilled Shipments Report ~ _GET_AMAZON_FULFILLED_SHIPMENTS_DATA_
         *      FBA Returns Report ~ _GET_FBA_FULFILLMENT_CUSTOMER_RETURNS_
         *      FBA Customer Shipment Sales Report ~ _GET_FBA_FULFILLMENT_CUSTOMER_SHIPMENT_SALES_DATA_
         *      Customer Taxes ~ _GET_FBA_FULFILLMENT_CUSTOMER_TAXES_DATA_
         *      FBA Promotions Report ~ _GET_FBA_FULFILLMENT_CUSTOMER_SHIPMENT_PROMOTION_DATA_
         *      FBA Inbound Compliance Report ~ _GET_FBA_FULFILLMENT_INBOUND_NONCOMPLIANCE_DATA_
         *      FBA Daily Inventory History Report ~ _GET_FBA_FULFILLMENT_CURRENT_INVENTORY_DATA_
         *      FBA Monthly Inventory History Repoty ~ _GET_FBA_FULFILLMENT_MONTHLY_INVENTORY_DATA_
         *      FBA Received Inventory Report ~ _GET_FBA_FULFILLMENT_INVENTORY_RECEIPTS_DATA_
         *      FBA Inventory Event Detail Report ~ _GET_FBA_FULFILLMENT_INVENTORY_SUMMARY_DATA_
         *      FBA Inventory Adjustments Report ~ _GET_FBA_FULFILLMENT_INVENTORY_ADJUSTMENTS_DATA_
         *      FBA Inventory Health Report ~ _GET_FBA_FULFILLMENT_INVENTORY_HEALTH_DATA_
         *      FBA Manage Inventory ~ _GET_FBA_MYI_UNSUPPRESSED_INVENTORY_DATA_
         *      FBA Manage Inventory - Archived ~ _GET_FBA_MYI_ALL_INVENTORY_DATA_
         *      FBA Replacements Report ~ _GET_FBA_FULFILLMENT_CUSTOMER_SHIPMENT_REPLACEMENT_DATA_
         *      FBA Cross-Border Inventory Movement Report ~ _GET_FBA_FULFILLMENT_CROSS_BORDER_INVENTORY_MOVEMENT_DATA_
         *      FBA Recommended Removal Report ~ _GET_FBA_RECOMMENDED_REMOVAL_DATA_
         * Amazon Product Ads Report:
         *      Product Ads Listings Report ~ _GET_NEMO_MERCHANT_LISTINGS_DATA_
         *      Product Ads Daily Performance by SKU Report, flat file ~ _GET_PADS_PRODUCT_PERFORMANCE_OVER_TIME_DAILY_DATA_TSV_
         *      Product Ads Daily Performance by SKU Report, XML ~ _GET_PADS_PRODUCT_PERFORMANCE_OVER_TIME_DAILY_DATA_XML_
         *      Product Ads Weekly Performance by SKU Report, flat file ~ _GET_PADS_PRODUCT_PERFORMANCE_OVER_TIME_WEEKLY_DATA_TSV_
         *      Product Ads Weekly Performance by SKU Report, XML ~ _GET_PADS_PRODUCT_PERFORMANCE_OVER_TIME_WEEKLY_DATA_XML_
         *      Product Ads Monthly Performance by SKU Report, flat file ~ _GET_PADS_PRODUCT_PERFORMANCE_OVER_TIME_MONTHLY_DATA_TSV_
         *      Product Ads Monthly Performance by SKU Report, XML ~ _GET_PADS_PRODUCT_PERFORMANCE_OVER_TIME_MONTHLY_DATA_XML_
         */
    }

    /**
     * Sets the time frame options. (Optional)
     *
     * This method sets the start and end times for the report request. If this
     * parameter is set, the report will only contain data that was updated
     * between the two times given. If these parameters are not set, the report
     * will only contain the most recent data.
     * The parameters are passed through <i>strtotime</i>, so values such as "-1 hour" are fine.
     * @param string $s [optional] <p>A time string for the earliest time.</p>
     * @param string $e [optional] <p>A time string for the latest time.</p>
     */
    public function setTimeLimits($s = null, $e = null)
    {
        if ($s && is_string($s)) {
            $times = $this->genTime($s);
            $this->options['StartDate'] = $times;
        }
        if ($e && is_string($e)) {
            $timee = $this->genTime($e);
            $this->options['EndDate'] = $timee;
        }
        if (isset($this->options['StartDate']) &&
            isset($this->options['EndDate']) &&
            $this->options['StartDate'] > $this->options['EndDate']
        ) {
            $this->setTimeLimits($this->options['EndDate'] . ' - 1 second');
        }
    }

    /**
     * Removes time limit options.
     *
     * Use this in case you change your mind and want to remove the time limit
     * parameters you previously set.
     */
    public function resetTimeLimits()
    {
        unset($this->options['StartDate']);
        unset($this->options['EndDate']);
    }

    /**
     * Sets whether or not the report should return the Sales Channel column. (Optional)
     *
     * Setting this parameter to <b>TRUE</b> adds the Sales Channel column to the report.
     * @param string|boolean $s <p>"true" or "false", or boolean</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setShowSalesChannel($s)
    {
        if ($s == 'true' || (is_bool($s) && $s == true)) {
            $this->options['ReportOptions=ShowSalesChannel'] = 'true';
        } elseif ($s == 'false' || (is_bool($s) && $s == false)) {
            $this->options['ReportOptions=ShowSalesChannel'] = 'false';
        } else {
            return false;
        }
    }

    /**
     * Sets the marketplace ID(s). (Optional)
     *
     * This method sets the list of marketplace IDs to be sent in the next request.
     * If this parameter is set, the report will only contain data relevant to the
     * marketplaces listed.
     * @param array|string $s <p>A list of marketplace IDs, or a single ID string.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setMarketplaces($s)
    {
        if (is_string($s)) {
            $this->resetMarketplaces();
            $this->options[self::URL_INDEX_MARKETPLACE_LIST_ID . '1'] = $s;
        } elseif (is_array($s)) {
            $this->resetMarketplaces();
            $i = 1;
            foreach ($s as $x) {
                $this->options[self::URL_INDEX_MARKETPLACE_LIST_ID . $i] = $x;
                $i++;
            }
        } else {
            return false;
        }
    }

    /**
     * Removes marketplace ID options.
     *
     * Use this in case you change your mind and want to remove the Marketplace ID
     * parameters you previously set.
     */
    public function resetMarketplaces()
    {
        foreach ($this->options as $op => $junk) {
            if (preg_match("#" . self::URL_INDEX_MARKETPLACE_LIST_ID . "#", $op)) {
                unset($this->options[$op]);
            }
        }
    }

    /**
     * Sends a report request to Amazon.
     *
     * Submits a <i>RequestReport</i> request to Amazon. In order to do this,
     * a Report Type is required. Amazon will send info back as a response,
     * which can be retrieved using <i>getResponse</i>.
     * Other methods are available for fetching specific values from the list.
     * @return boolean <b>FALSE</b> if something goes wrong
     */
    public function requestReport()
    {
        if (!array_key_exists('ReportType', $this->options)) {
            $this->log("Report Type must be set in order to request a report!", 'WARNING');
            return false;
        }

        $url = $this->urlbase . $this->urlbranch;

        $query = $this->genQuery();

        $path = $this->options['Action'] . 'Result';
        if ($this->mockMode) {
            $response = $this->fetchMockFile(self::MOCK_FILE_REQUEST_REPORT);
            if ($response) {
                $xml = $response->$path;
            } else {
                return false;
            }
        } else {
            $response = $this->sendRequest($url, ['Post' => $query]);

            if (!$this->checkResponse($response)) {
                return false;
            }

            $xml = simplexml_load_string($response['body'])->$path;
        }

        $this->parseXML($xml->ReportRequestInfo);
        return true;
    }

    /**
     * Parses XML response into array.
     *
     * This is what reads the response XML and converts it into an array.
     * @param SimpleXMLObject $xml <p>The XML response from Amazon.</p>
     * @return boolean <b>FALSE</b> if no XML data is found
     */
    protected function parseXML($xml)
    {
        if (!$xml) {
            return false;
        }

        $this->response = [];
        $this->response['ReportRequestId'] = (string)$xml->ReportRequestId;
        $this->response['ReportType'] = (string)$xml->ReportType;
        $this->response['StartDate'] = (string)$xml->StartDate;
        $this->response['EndDate'] = (string)$xml->EndDate;
        $this->response['Scheduled'] = (string)$xml->Scheduled;
        $this->response['SubmittedDate'] = (string)$xml->SubmittedDate;
        $this->response['ReportProcessingStatus'] = (string)$xml->ReportProcessingStatus;
        return true;
    }

    /**
     * Returns the full response.
     *
     * This method will return <b>FALSE</b> if the response data has not yet been filled.
     * The returned array will have the following fields:
     * <ul>
     * <li><b>ReportRequestId</b></li>
     * <li><b>ReportType</b></li>
     * <li><b>StartDate</b></li>
     * <li><b>EndDate</b></li>
     * <li><b>Scheduled</b> - "true" or "false"</li>
     * <li><b>SubmittedDate</b></li>
     * <li><b>ReportProcessingStatus</b></li>
     * </ul>
     * @return array|boolean data array, or <b>FALSE</b> if list not filled yet
     */
    public function getResponse()
    {
        if (isset($this->response)) {
            return $this->response;
        } else {
            return false;
        }
    }

    /**
     * Returns the report request ID from the response.
     *
     * This method will return <b>FALSE</b> if the response data has not yet been filled.
     * @return string|boolean single value, or <b>FALSE</b> if Non-numeric index
     */
    public function getReportRequestId()
    {
        if (isset($this->response)) {
            return $this->response['ReportRequestId'];
        } else {
            return false;
        }
    }

    /**
     * Returns the report type from the response.
     *
     * This method will return <b>FALSE</b> if the response data has not yet been filled.
     * @return string|boolean single value, or <b>FALSE</b> if Non-numeric index
     */
    public function getReportType()
    {
        if (isset($this->response)) {
            return $this->response['ReportType'];
        } else {
            return false;
        }
    }

    /**
     * Returns the start date for the report from the response.
     *
     * This method will return <b>FALSE</b> if the response data has not yet been filled.
     * @return string|boolean single value, or <b>FALSE</b> if Non-numeric index
     */
    public function getStartDate()
    {
        if (isset($this->response)) {
            return $this->response['StartDate'];
        } else {
            return false;
        }
    }

    /**
     * Returns the end date for the report from the response.
     *
     * This method will return <b>FALSE</b> if the response data has not yet been filled.
     * @return string|boolean single value, or <b>FALSE</b> if Non-numeric index
     */
    public function getEndDate()
    {
        if (isset($this->response)) {
            return $this->response['EndDate'];
        } else {
            return false;
        }
    }

    /**
     * Returns whether or not the report is scheduled from the response.
     *
     * This method will return <b>FALSE</b> if the response data has not yet been filled.
     * @return string|boolean "true" or "false", or <b>FALSE</b> if Non-numeric index
     */
    public function getIsScheduled()
    {
        if (isset($this->response)) {
            return $this->response['Scheduled'];
        } else {
            return false;
        }
    }

    /**
     * Returns the date the report was submitted from the response.
     *
     * This method will return <b>FALSE</b> if the response data has not yet been filled.
     * @return string|boolean single value, or <b>FALSE</b> if Non-numeric index
     */
    public function getSubmittedDate()
    {
        if (isset($this->response)) {
            return $this->response['SubmittedDate'];
        } else {
            return false;
        }
    }

    /**
     * Returns the report processing status from the response.
     *
     * This method will return <b>FALSE</b> if the response data has not yet been filled.
     * @param int $i [optional] <p>List index to retrieve the value from. Defaults to 0.</p>
     * @return string|boolean single value, or <b>FALSE</b> if Non-numeric index
     */
    public function getStatus()
    {
        if (isset($this->response)) {
            return $this->response['ReportProcessingStatus'];
        } else {
            return false;
        }
    }
}
