<?php
/**
 * Copyright 2013 CPI Group, LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 *
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

namespace Amazon\Sdk\Api;

/**
 * Fetches a report from Amazon
 *
 * This Amazon Reports Core object retrieves the results of a report from Amazon.
 * In order to do this, a report ID is required. The results of the report can
 * then be saved to a file.
 */
class Report extends \Amazon\Sdk\Api\Report\ReportsCore
{
    private $rawreport;

    /**
     * AmazonReport fetches a report from Amazon.
     *
     * The parameters are passed to the parent constructor, which are
     * in turn passed to the AmazonCore constructor. See it for more information
     * on these parameters and common methods.
     * Please note that an extra parameter comes before the usual Mock Mode parameters,
     * so be careful when setting up the object.
     * @param string $id [optional] <p>The report ID to set for the object.</p>
     * @param \Amazon\Sdk\Api\ConfigInterface|null $config ,
     * @param \Psr\Log\LoggerInterface|null $logger ,
     * @param boolean $mockMode [optional] <p>This is a flag for enabling Mock Mode.
     * This defaults to <b>FALSE</b>.</p>
     * @param array|string $mockFiles [optional] <p>The files (or file) to use in Mock Mode.</p>
     */
    public function __construct(
        $id = null,
        \Amazon\Sdk\Api\ConfigInterface $config = null,
        \Psr\Log\LoggerInterface $logger = null,
        $mockMode = false,
        $mockFiles = null
    )
    {
        parent::__construct($config, $logger, $mockMode, $mockFiles);

        if ($id) {
            $this->setReportId($id);
        }

        $this->options['Action'] = 'GetReport';
        $this->throttleLimit = \Amazon\Sdk\Api\Environment::THROTTLE_LIMIT_REPORT;
        $this->throttleTime = \Amazon\Sdk\Api\Environment::THROTTLE_TIME_REPORT;

    }

    /**
     * Sets the report ID. (Required)
     *
     * This method sets the report ID to be sent in the next request.
     * This parameter is required for fetching the report from Amazon.
     * @param string|integer $n <p>Must be numeric</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setReportId($n)
    {
        if (is_numeric($n)) {
            $this->options['ReportId'] = $n;
            return true;
        } else {
            return false;
        }
    }

    /**
     * Sends a request to Amazon for a report.
     *
     * Submits a <i>GetReport</i> request to Amazon. In order to do this,
     * a report ID is required. Amazon will send
     * the data back as a response, which can be saved using <i>saveReport</i>.
     * @return boolean <b>FALSE</b> if something goes wrong
     */
    public function fetchReport()
    {
        if (!array_key_exists('ReportId', $this->options)) {
            $this->log("Report Id must be set in order to fetch it!", 'WARNING', ['path' => __METHOD__]);
            return false;
        }

        $url = $this->urlbase . $this->urlbranch;

        $query = $this->genQuery();

        if ($this->mockMode) {
            $this->rawreport = $this->fetchMockFile(self::MOCK_FILE_FETCH_REPORT, false);
        } else {
            $response = $this->sendRequest($url, ['Post' => $query]);

            if (!$this->checkResponse($response)) {
                return false;
            }

            $this->rawreport = $response['body'];
        }
        return true;
    }

    /**
     * Saves the raw report data to a path you specify
     * @param string $path <p>filename to save the file in</p>
     * @return boolean
     */
    public function saveReport($path)
    {
        $status = false;
        if (!isset($this->rawreport)) {
            return $status;
        }
        try {
            $status = file_put_contents($path, $this->rawreport);
            $this->log(
                "Successfully saved report #" . $this->options['ReportId'] . " at $path",
                'DEBUG',
                ['path' => __METHOD__]
            );
        } catch (\Exception $e) {
            $this->log(
                "Unable to save report #" . $this->options['ReportId'] . " at $path: " . $e->getMessage(),
                'ERROR',
                ['path' => __METHOD__, 'exception' => $e->getTraceAsString()]
            );
        }

        return $status;
    }

    public function getReportContent()
    {
        if (isset($this->rawreport)) {
            return $this->rawreport;
        } else {
            return false;
        }
    }
}
