<?php

namespace Amazon\Sdk\Api\Order;

/**
 * Copyright 2013 CPI Group, LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 *
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * Pulls a list of Orders and turn them into an array of Amazon\Sdk\Api\Order objects.
 *
 * This Amazon Orders Core object can retrieve a list of orders from Amazon
 * and store them in an array of AmazonOrder objects. A number of filters
 * are available to narrow the number of orders returned, but none of them
 * are required. This object can use tokens when retrieving the list.
 */
class OrderList extends \Amazon\Sdk\Api\Order\Core implements \Iterator
{
    protected $tokenFlag = false;
    protected $tokenUseFlag = false;
    private $orderList = [];
    private $i = 0;
    private $index = 0;

    /**
     * Amazon Order Lists pull a set of Orders and turn them into an array of <i>AmazonOrder</i> objects.
     *
     * The parameters are passed to the parent constructor, which are
     * in turn passed to the AmazonCore constructor. See it for more information
     * on these parameters and common methods.
     * @param \Amazon\Sdk\Api\ConfigInterface|null $config ,
     * @param \Psr\Log\LoggerInterface|null $logger ,
     * @param boolean $mockMode [optional] <p>This is a flag for enabling Mock Mode.
     * This defaults to <b>FALSE</b>.</p>
     * @param array|string $mockFiles [optional] <p>The files (or file) to use in Mock Mode.</p>
     * @throws \Amazon\Sdk\Api\Exception\InvalidConfigValue
     */
    public function __construct(
        \Amazon\Sdk\Api\ConfigInterface $config = null,
        \Psr\Log\LoggerInterface $logger = null,
        $mockMode = false,
        $mockFiles = null
    ) {
        parent::__construct($config, $logger, $mockMode, $mockFiles);

        $marketplaceIds = $this->config->getMarketplaceId();
        if (isset($marketplaceIds) && !empty($marketplaceIds)) {
            $this->setMarketplaceIds($marketplaceIds);
        } else {
            $this->log("Marketplace Id is missing", 'ERROR');
        }

        $this->throttleLimit = \Amazon\Sdk\Api\Environment::THROTTLE_LIMIT_ORDERLIST;
        $this->throttleTime = \Amazon\Sdk\Api\Environment::THROTTLE_TIME_ORDERLIST;
        $this->throttleGroup = 'ListOrders';
    }

    /**
     * Returns whether or not a token is available.
     * @return boolean
     */
    public function hasToken()
    {
        return $this->tokenFlag;
    }

    /**
     * Sets whether or not the object should automatically use tokens if it receives one.
     *
     * If this option is set to <b>TRUE</b>, the object will automatically perform
     * the necessary operations to retrieve the rest of the list using tokens. If
     * this option is off, the object will only ever retrieve the first section of
     * the list.
     * @param boolean $b [optional] <p>Defaults to <b>TRUE</b></p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setUseToken($b = true)
    {
        if (is_bool($b)) {
            $this->tokenUseFlag = $b;
            return true;
        }
        return false;
    }

    /**
     * Sets the order status(es). (Optional)
     *
     * This method sets the list of Order Statuses to be sent in the next request.
     * Setting this parameter tells Amazon to only return Orders with statuses that match
     * those in the list. If this parameter is not set, Amazon will return
     * Orders of any status.
     * @param array|string $list <p>A list of Order Statuses, or a single status string.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setOrderStatusFilter($list)
    {
        if (is_string($list) && in_array($list, self::ORDER_STATUS)) {
            //if single string, set as filter
            $this->resetOrderStatusFilter();
            $this->options['OrderStatus.Status.1'] = $list;
            return true;
        } elseif (is_array($list)) {
            //if array of strings, set all filters
            $this->resetOrderStatusFilter();
            $i = 1;
            foreach ($list as $x) {
                if (in_array($x, self::ORDER_STATUS)) {
                    $this->options['OrderStatus.Status.' . $i] = $x;
                    $i++;
                }
            }
            return true;
        }
        return false;
    }

    public function setAmazonOrderIdFilter($list)
    {
        if (is_string($list)) {
            //if single string, set as filter
            $this->resetOrderStatusFilter();
            $this->options['AmazonOrderId.Id.1'] = $list;
            return true;
        } elseif (is_array($list)) {
            //if array of strings, set all filters
            $this->resetAmazonOrderIdFilter();
            $i = 1;
            foreach ($list as $x) {
                if ($i > 50) {
                    break;
                }
                if (!empty($x)) {
                    $this->options['AmazonOrderId.Id.' . $i] = $x;
                    $i++;
                }
            }
            return true;
        }
        return false;
    }

    /**
     * Removes order status options.
     *
     * Use this in case you change your mind and want to remove the Order Status
     * parameters you previously set.
     */
    public function resetOrderStatusFilter()
    {
        foreach ($this->options as $op => $junk) {
            if (preg_match("#OrderStatus#", $op)) {
                unset($this->options[$op]);
            }
        }
    }

    public function resetAmazonOrderIdFilter()
    {
        foreach ($this->options as $op => $junk) {
            if (preg_match("#AmazonOrderId#", $op)) {
                unset($this->options[$op]);
            }
        }
    }

    /**
     * Sets the order BuyerEmail. (Optional)
     *
     * This method sets the list of Order BuyerEmail to be sent in the next request.
     * Setting this parameter tells Amazon to only return Orders with BuyerEmail that matches.
     * If this parameter is not set, Amazon will return
     * all BuyerEmail orders.
     * @param string $buyerEmail <p>a single BuyerEmail string.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setBuyerEmail($buyerEmail)
    {
        if (is_string($buyerEmail)) {
            //if single string, set as filter
            $this->resetBuyerEmailFilter();
            $this->options['BuyerEmail'] = $buyerEmail;
            return true;
        }
        return false;
    }

    /**
     * Removes order buyer email options.
     *
     * Use this in case you change your mind and want to remove the Order BuyerEmail
     * parameters you previously set.
     */
    public function resetBuyerEmailFilter()
    {
        foreach ($this->options as $op => $junk) {
            if (preg_match("#BuyerEmail#", $op)) {
                unset($this->options[$op]);
            }
        }
    }

    /**
     * Set seller order id, differ from amazon order id
     * @param $orderId
     * @return bool
     */
    public function setOrderId($orderId)
    {
        if (is_string($orderId)) {
            //if single string, set as filter as
            // - SellerOrderId cannot be combined with
            // LastUpdatedAfter, LastUpdatedBefore, FulfillmentChannel, PaymentMethod,
            // OrderStatus, BuyerEmail or TFMShipmentStatus as a filter option
            $this->resetOrderIdFilter();
            $this->options['SellerOrderId'] = $orderId;
            return true;
        }
        return false;
    }

    public function resetOrderIdFilter()
    {
        foreach ($this->options as $op => $junk) {
            if (preg_match("#SellerOrderId#", $op)) {
                unset($this->options[$op]);
            }
        }

        unset($this->options['LastUpdatedAfter']);
        unset($this->options['LastUpdatedBefore']);
        unset($this->options['FulfillmentChannel']);
        unset($this->options['PaymentMethod']);
        unset($this->options['OrderStatus.Status.1']);
        unset($this->options['OrderStatus.Status.2']);
        unset($this->options['BuyerEmail']);
        unset($this->options['TFMShipmentStatus']);
    }

    /**
     * Sets the payment method(s). (Optional)
     *
     * This method sets the list of Payment Methods to be sent in the next request.
     * Setting this parameter tells Amazon to only return Orders with payment methods
     * that match those in the list. If this parameter is not set, Amazon will return
     * Orders with any payment method.
     * @param $list
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setPaymentMethodFilter($list)
    {
        if (is_string($list)) {
            //if single string, set as filter
            $this->resetPaymentMethodFilter();
            $this->options['PaymentMethod.1'] = $list;
            return true;
        } elseif (is_array($list)) {
            //if array of strings, set all filters
            $this->resetPaymentMethodFilter();
            $i = 1;
            foreach ($list as $x) {
                $this->options['PaymentMethod.' . ($i++)] = $x;
            }
            return true;
        }
        return false;
    }

    /**
     * Removes payment method options.
     *
     * Use this in case you change your mind and want to remove the Payment Method
     * parameters you previously set.
     */
    public function resetPaymentMethodFilter()
    {
        foreach ($this->options as $op => $junk) {
            if (preg_match("#PaymentMethod#", $op)) {
                unset($this->options[$op]);
            }
        }
    }

    /**
     * Sets the maximum response per page count. (Optional)
     *
     * This method sets the maximum number of Feed Submissions for Amazon to return per page.
     * If this parameter is not set, Amazon will send 100 at a time.
     * @param $num
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setMaxResultsPerPage($num)
    {
        if (is_int($num) && $num <= 100 && $num >= 1) {
            $this->options['MaxResultsPerPage'] = $num;
            return true;
        }
        return false;
    }

    /**
     * Fetches orders from Amazon and puts them in an array of <i>AmazonOrder</i> objects.
     *
     * Submits a <i>ListOrders</i> request to Amazon. Amazon will send
     * the list back as a response, which can be retrieved using <i>getList</i>.
     * This operation can potentially involve tokens.
     * @param boolean <p>When set to <b>FALSE</b>, the function will not recurse, defaults to <b>TRUE</b></p>
     * @return boolean <b>FALSE</b> if something goes wrong
     * @throws \Amazon\Sdk\Api\Exception\InvalidConfigValue
     */
    public function fetchOrders($r = true)
    {
        if (!array_key_exists('CreatedAfter', $this->options) &&
            !array_key_exists('LastUpdatedAfter', $this->options)
        ) {
            $this->setLimits('Created');
        }

        $this->prepareToken();

        $url = $this->urlbase . $this->urlbranch;

        $query = $this->genQuery();

        $path = $this->options['Action'] . 'Result';
        if ($this->mockMode) {
            $response = $this->fetchMockFile(self::MOCK_FILE_FETCH_ORDER_LIST);
            if ($response) {
                $xml = $response->$path;
            } else {
                return false;
            }
        } else {
            $response = $this->sendRequest($url, ['Post' => $query]);
            if (!$this->checkResponse($response)) {
                return false;
            }

            $xml = simplexml_load_string($response['body'])->$path;
        }

        $this->parseXML($xml);

        $this->checkToken($xml);

        if ($this->tokenFlag && $this->tokenUseFlag && $r === true) {
            while ($this->tokenFlag) {
                $this->log("Recursively fetching more orders");
                $this->fetchOrders(false);
            }
        }

        return true;
    }

    /**
     * Sets the time frame for the orders fetched. (Optional)
     *
     * Sets the time frame for the orders fetched. If no times are specified, times default to the current time.
     * @param string $mode <p>"Created" or "Modified"</p>
     * @param string $lower [optional] <p>A time string for the earliest time.</p>
     * @param string $upper [optional] <p>A time string for the latest time.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setLimits($mode, $lower = null, $upper = null)
    {
        try {
            if ($upper) {
                $before = $this->genTime($upper);
            } else {
                $before = $this->genTime('- 2 min');
            }
            if ($lower) {
                $after = $this->genTime($lower);
            } else {
                $after = $this->genTime('- 2 min');
            }
            if ($after > $before) {
                $after = $this->genTime($upper . ' - 150 sec');
            }
            if ($mode == 'Created') {
                $this->options['CreatedAfter'] = $after;
                if ($before) {
                    $this->options['CreatedBefore'] = $before;
                }
                unset($this->options['LastUpdatedAfter']);
                unset($this->options['LastUpdatedBefore']);
            } elseif ($mode == 'Modified') {
                $this->options['LastUpdatedAfter'] = $after;
                if ($before) {
                    $this->options['LastUpdatedBefore'] = $before;
                }
                unset($this->options['CreatedAfter']);
                unset($this->options['CreatedBefore']);
            } else {
                $this->log('First parameter should be either "Created" or "Modified".', 'WARNING');
                return false;
            }
        } catch (\Exception $e) {
            $this->log('Error: ' . $e->getMessage(), 'WARNING');
            return false;
        }
        return true;
    }

    /**
     * Sets up options for using tokens.
     *
     * This changes key options for switching between simply fetching a list and
     * fetching the rest of a list using a token. Please note: because the
     * operation for using tokens does not use any other parameters, all other
     * parameters will be removed.
     */
    protected function prepareToken()
    {
        if ($this->tokenFlag && $this->tokenUseFlag) {
            $this->options['Action'] = 'ListOrdersByNextToken';

            //When using tokens, only the NextToken option should be used
            unset($this->options['SellerOrderId']);
            $this->resetOrderStatusFilter();
            $this->resetPaymentMethodFilter();
            $this->setFulfillmentChannelFilter(null);
            $this->setSellerOrderIdFilter(null);
            $this->setEmailFilter(null);
            unset($this->options['LastUpdatedAfter']);
            unset($this->options['LastUpdatedBefore']);
            unset($this->options['CreatedAfter']);
            unset($this->options['CreatedBefore']);
            unset($this->options['MaxResultsPerPage']);
        } else {
            if (isset($this->options['AmazonOrderId.Id.1'])) {
                $this->options['Action'] = 'GetOrder';
            } else {
                $this->options['Action'] = 'ListOrders';
            }
            unset($this->options['NextToken']);
            $this->index = 0;
            $this->orderList = [];
        }
    }

    /**
     * Sets (or resets) the Fulfillment Channel Filter
     * @param string $filter <p>'AFN' or 'MFN' or NULL</p>
     * @return boolean <b>FALSE</b> on failure
     */
    public function setFulfillmentChannelFilter($filter)
    {
        if ($filter == 'AFN' || $filter == 'MFN') {
            $this->options['FulfillmentChannel.Channel.1'] = $filter;
        } elseif (is_null($filter)) {
            unset($this->options['FulfillmentChannel.Channel.1']);
        }
        return false;
    }

    /**
     * Sets (or resets) the seller order ID(s). (Optional)
     *
     * This method sets the list of seller order IDs to be sent in the next request.
     * Setting this parameter tells Amazon to only return Orders with addresses
     * that match those in the list. If this parameter is set, the following options
     * will be removed: BuyerEmail, OrderStatus, PaymentMethod, FulfillmentChannel, LastUpdatedAfter, LastUpdatedBefore.
     * @param $filter
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setSellerOrderIdFilter($filter)
    {
        if (is_string($filter)) {
            $this->options['SellerOrderId'] = $filter;
            //these fields must be disabled
            unset($this->options['BuyerEmail']);
            $this->resetOrderStatusFilter();
            $this->resetPaymentMethodFilter();
            $this->setFulfillmentChannelFilter(null);
            unset($this->options['LastUpdatedAfter']);
            unset($this->options['LastUpdatedBefore']);
        } elseif (is_null($filter)) {
            unset($this->options['SellerOrderId']);
        } else {
            return false;
        }
    }

    /**
     * Sets (or resets) the email address. (Optional)
     *
     * This method sets the email address to be sent in the next request.
     * Setting this parameter tells Amazon to only return Orders with addresses
     * that match the address given. If this parameter is set, the following options
     * will be removed: SellerOrderId, OrderStatus, PaymentMethod, FulfillmentChannel,
     * LastUpdatedAfter, LastUpdatedBefore.
     * @param $filter
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setEmailFilter($filter)
    {
        if (is_string($filter)) {
            $this->options['BuyerEmail'] = $filter;
            //these fields must be disabled
            unset($this->options['SellerOrderId']);
            $this->resetOrderStatusFilter();
            $this->resetPaymentMethodFilter();
            $this->setFulfillmentChannelFilter(null);
            unset($this->options['LastUpdatedAfter']);
            unset($this->options['LastUpdatedBefore']);
            return true;
        } elseif (is_null($filter)) {
            unset($this->options['BuyerEmail']);
            return true;
        }
        return false;
    }

    /**
     * Parses XML response into array.
     *
     * This is what reads the response XML and converts it into an array.
     * @param \SimpleXMLElement $xml <p>The XML response from Amazon.</p>
     * @return boolean <b>FALSE</b> if no XML data is found
     * @throws \Amazon\Sdk\Api\Exception\InvalidConfigValue
     */
    protected function parseXML($xml)
    {
        if (!$xml) {
            return false;
        }

        foreach ($xml->Orders->children() as $key => $data) {
            if ($key != 'Order') {
                break;
            }
            $this->orderList[$this->index] = new \Amazon\Sdk\Api\Order(
                null,
                $data,
                $this->config,
                $this->logger,
                $this->mockMode,
                $this->mockFiles
            );
            $this->index++;
        }
        return true;
    }

    public function loadByReport($path)
    {
        if (file_exists($path)) {
            $headers = [];
            $first = true;

            $file = fopen($path, 'r');
            while (($line = fgetcsv($file, 0, "\t")) !== false) {
                if ($first) {
                    $headers = $line;
                    $first = false;
                } else {
                    $order = [];
                    foreach ($headers as $i => $header) {
                        $order[$header] = $line[$i];
                    }

                    $this->addOrder($order);
                }
            }
            fclose($file);
        }
    }

    /**
     * Add Order or Order Item to OrderList
     * @param array $order
     * @return bool
     * @throws \Amazon\Sdk\Api\Exception\InvalidConfigValue
     */
    public function addOrder(array $order = [])
    {
        if (isset($order["amazon-order-id"])) {
            $amazonOrderId = $order["amazon-order-id"];
            if ($this->getList() == false) {
                $this->orderList = [];
            }

            /**
             * @var int $key
             * @var \Amazon\Sdk\Api\Order $value
             */
            foreach ($this->orderList as $key => $value) {
                if ($value->getAmazonOrderId() == $amazonOrderId) {
                    $value->addOrderItem($order);
                    return true;
                }
            }

            $item = new \Amazon\Sdk\Api\Order(
                null,
                null,
                $this->config,
                $this->logger,
                $this->mockMode,
                $this->mockFiles
            );
            $item->loadByReport($order);

            $this->orderList[$this->index] = $item;
            $this->index++;
            return true;
        }

        return false;
    }

    /**
     * Returns array of item lists or a single item list.
     *
     * If <i>$i</i> is not specified, the method will fetch the items for every
     * order in the list. Please note that for lists with a high number of orders,
     * this operation could take a while due to throttling. (Two seconds per order when throttled.)
     * @param boolean $token [optional] <p>whether or not to automatically use tokens when fetching items.</p>
     * @param int $i [optional] <p>List index to retrieve the value from. Defaults to null.</p>
     * @return array|boolean <i>AmazonOrderItemList</i> object or array of objects,
     * or <b>FALSE</b> if non-numeric index
     * @throws \Amazon\Sdk\Api\Exception\InvalidConfigValue
     */
    public function fetchItems($token = false, $i = null)
    {
        if (!isset($this->orderList)) {
            return false;
        }
        if (!is_bool($token)) {
            $token = false;
        }
        if (is_int($i)) {
            return $this->orderList[$i]->fetchItems($token);
        } else {
            $a = [];
            /** @var \Amazon\Sdk\Api\Order $x */
            foreach ($this->orderList as $x) {
                $a[] = $x->fetchItems($token);
            }
            return $a;
        }
    }

    /**
     * Returns the list of orders.
     * @return array|boolean array of <i>AmazonOrder</i> objects, or <b>FALSE</b> if list not filled yet
     */
    public function getList()
    {
        if (isset($this->orderList)) {
            return $this->orderList;
        }
        return false;
    }

    /**
     * Iterator function
     * @return \Amazon\Sdk\Api\Order
     */
    public function current()
    {
        return $this->orderList[$this->i];
    }

    /**
     * Iterator function
     */
    public function rewind()
    {
        $this->i = 0;
    }

    /**
     * Iterator function
     * @return int
     */
    public function key()
    {
        return $this->i;
    }

    /**
     * Iterator function
     */
    public function next()
    {
        $this->i++;
    }

    /**
     * Iterator function
     * @return boolean
     */
    public function valid()
    {
        return isset($this->orderList[$this->i]);
    }
}
