<?php
/**
 * CedCommerce
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the End User License Agreement (EULA)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://cedcommerce.com/license-agreement.txt
 *
 * @category    Ced
 * @package     Ced_Amazon_sdk
 * @author      CedCommerce Core Team <connect@cedcommerce.com>
 * @copyright   Copyright © 2018 CedCommerce. All rights reserved.
 * @license     EULA http://cedcommerce.com/license-agreement.txt
 */

namespace Amazon\Sdk\Api\Merchant\Fulfillment;

/**
 * Submits a request to get a list of shipping service offers.
 *
 * This Amazon object can submit a request to Amazon to
 * get a list of shipping service offers. In order to get
 * shipping service, Amazon Order Id is needed.
 */
class GetEligibleShippingServices extends \Amazon\Sdk\Api\Core
{
    private $shippingServiceList = [];
    private $rejectedShippingServiceList= [];
    private $temporarilyUnavailableCarrierList = [];
    private $termsAndConditionsNotAcceptedCarrierList = [];
    private $index = 0;
    /**
     * AmazonMerchantFulfillment GetEligibleShippingServices gets a list of shipping service offers. You need a amazon order ID.
     *
     * The parameters are passed to the parent constructor, which are
     * in turn passed to the AmazonCore constructor. See it for more information
     * on these parameters and common methods.
     * @param \Amazon\Sdk\Api\ConfigInterface $config = null,
     * @param \Psr\Log\LoggerInterface $logger = null,
     * @param boolean $mockMode [optional] <p>This is a flag for enabling Mock Mode.
     * This defaults to <b>FALSE</b>.</p>
     * @param array|string $mockFiles [optional] <p>The files (or file) to use in Mock Mode.</p>
     * @throws \Amazon\Sdk\Api\Exception\InvalidConfigValue
     */
    public function __construct(
        \Amazon\Sdk\Api\ConfigInterface $config = null,
        \Psr\Log\LoggerInterface $logger = null,
        $mockMode = false,
        $mockFiles = null
    )
    {
        parent::__construct($config, $logger, $mockMode, $mockFiles);

        $this->options['Action'] = 'GetEligibleShippingServices';
    }

    /**
     * Sets the Amazon order ID. (Required)
     *
     * This method sets the Amazon Order ID to be sent in the next request.
     * This parameter is required for getting a list of shipping services from Amazon.
     * Amazon order ID can be generated using the <i>GetOrder</i> object.
     * @param string $s <p>Maximum 50 characters.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setAmazonOrderId($s)
    {
        if (is_string($s)) {
            $this->options['ShipmentRequestDetails.AmazonOrderId'] = $s;
            return true;
        }

        return false;
    }

    /**
     * Sets the Seller order ID. (Optional)
     *
     * This method sets the Seller Order ID to be sent in the next request.
     * Seller order ID is a seller-defined order identifier..
     * @param string $s <p>Maximum 64 characters.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setSellerOrderId($s)
    {
        if (is_string($s)) {
            $this->options['ShipmentRequestDetails.SellerOrderId'] = $s;
            return true;
        }

        return false;
    }

    /**
     * Sets the items. (Required)
     *
     * This parameter is required for getting a list of shipping services offers from Amazon.
     * The array provided should contain a list of arrays, each with the following fields:
     * <ul>
     * <li><b>OrderItemId</b> An Amazon-defined identifier for an individual item in an order.
     * Used in the XML response to an order query request (Order API/Order XML).</li>
     * <li><b>Quantity</b> - numeric</li>
     * <li><b>ItemWeight</b> (optional array) - The weight of the item.</li>
     * <ul>
     * <li><b>Value</b> - The decimal weight of the object.</li>
     * <li><b>Unit</b> - The unit of measurement for weight.Unit values: ounces, gram. string</li>
     * </ul>
     * <li><b>ItemDescription</b> (optional) - string</li>
     * <li><b>TransparencyCodeList</b> (optional array) </li>
     * <ul>
     * <li><b>TransparencyCode</b> - string.</li>
     * </ul>
     * @param array $a <p>See above.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setItems($a)
    {
        if (is_null($a) || is_string($a) || !$a) {
            $this->log("Tried to set Items to invalid values", 'WARNING');
            return false;
        }
        $this->resetItems();
        $i = 1;
        foreach ($a as $x) {
            if (is_array($x) and
                array_key_exists('OrderItemId', $x) && array_key_exists('Quantity', $x)) {
                $this->options['ShipmentRequestDetails.ItemList.Item.' . $i . '.OrderItemId'] = $x['OrderItemId'];
                $this->options['ShipmentRequestDetails.ItemList.Item.' . $i . '.Quantity'] = $x['Quantity'];
                if (array_key_exists('ItemWeight', $x)) {
                    $this->options['ShipmentRequestDetails.ItemList.Item.' . $i . '.ItemWeight.Value'] = $x['ItemWeight']['Value'];
                    $this->options['ShipmentRequestDetails.ItemList.Item.' . $i . '.ItemWeight.Unit'] = $x['ItemWeight']['Unit'];
                }
                if (array_key_exists('ItemDescription', $x)) {
                    $this->options['ShipmentRequestDetails.ItemList.Item.' . $i . '.ItemDescription'] = $x['ItemDescription'];
                }
                if (array_key_exists('TransparencyCodeList', $x)) {
                    $this->options['ShipmentRequestDetails.ItemList.Item.' . $i . '.TransparencyCodeList.TransparencyCode'] =
                        $x['TransparencyCodeList']['TransparencyCode'];
                }
                $i++;
            } else {
                $this->resetItems();
                $this->log("Tried to set Items with invalid array", 'WARNING');
                return false;
            }
        }
        return true;
    }

    /**
     * Resets the item options.
     *
     * Since the list of items is a required parameter, these options should not be removed
     * without replacing them, so this method is not public.
     */
    protected function resetItems()
    {
        foreach ($this->options as $op => $junk) {
            if (preg_match("#Item#", $op)) {
                unset($this->options[$op]);
            }
        }
    }

    /**
     * Sets the address. (Required)
     *
     * This method sets the shipping from address to be sent in the next request.
     * This parameter is required getting a list of shipping services offers from Amazon.
     * The array provided should have the following fields:
     * <ul>
     * <li><b>Name</b> - max: 50 char</li>
     * <li><b>AddressLine1</b> - max: 180 char</li>
     * <li><b>AddressLine2</b> (optional) - max: 60 char</li>
     * <li><b>AddressLine3</b> (optional) - max: 60 char</li>
     * <li><b>DistrictOrCounty</b> (optional) - max: 150 char</li>
     * <li><b>Email</b> - max: 256 char</li>
     * <li><b>City</b> - max: 30 char</li>
     * <li><b>StateOrProvinceCode</b> - max: 30 char</li>
     * <li><b>CountryCode</b> - 2 digits</li>
     * <li><b>PostalCode</b> - max: 30 char</li>
     * <li><b>Phone</b> - max: 30 char</li>
     * </ul>
     * @param array $a <p>See above.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setAddress($a)
    {
        if (is_null($a) || is_string($a) || !$a) {
            $this->log("Tried to set address to invalid values", 'WARNING');
            return false;
        }
        $this->resetAddress();
        if (is_array($a) and
            array_key_exists('Name', $a) && array_key_exists('AddressLine1', $a) &&
            array_key_exists('Email', $a) && array_key_exists('City', $a) &&
            array_key_exists('StateOrProvinceCode', $a) && array_key_exists('PostalCode', $a) &&
            array_key_exists('CountryCode', $a) && array_key_exists('Phone', $a)) {
            $this->options['ShipmentRequestDetails.ShipFromAddress.Name'] = $a['Name'];
            $this->options['ShipmentRequestDetails.ShipFromAddress.AddressLine1'] = $a['AddressLine1'];
            if (array_key_exists('AddressLine2', $a)) {
                $this->options['ShipmentRequestDetails.ShipFromAddress.AddressLine2'] = $a['AddressLine2'];
            }
            if (array_key_exists('AddressLine3', $a)) {
                $this->options['ShipmentRequestDetails.ShipFromAddress.AddressLine3'] = $a['AddressLine3'];
            }
            if (array_key_exists('DistrictOrCounty', $a)) {
                $this->options['ShipFromAddress.DistrictOrCounty'] = $a['DistrictOrCounty'];
            }
            $this->options['ShipmentRequestDetails.ShipFromAddress.Email'] = $a['Email'];
            $this->options['ShipmentRequestDetails.ShipFromAddress.City'] = $a['City'];
            $this->options['ShipmentRequestDetails.ShipFromAddress.StateOrProvinceCode'] = $a['StateOrProvinceCode'];
            $this->options['ShipmentRequestDetails.ShipFromAddress.CountryCode'] = $a['CountryCode'];
            $this->options['ShipmentRequestDetails.ShipFromAddress.PostalCode'] = $a['PostalCode'];
            if (array_key_exists('Phone', $a)) {
                $this->options['ShipmentRequestDetails.ShipFromAddress.Phone'] = $a['Phone'];
            }
        } else {
            $this->resetAddress();
            $this->log("Tried to set Address with invalid array", 'WARNING');
            return false;
        }
        return true;
    }

    /**
     * Resets the address options.
     *
     * Since address is a required parameter, these options should not be removed
     * without replacing them, so this method is not public.
     */
    protected function resetAddress()
    {
        unset($this->options['ShipFromAddress.Name']);
        unset($this->options['ShipFromAddress.AddressLine1']);
        unset($this->options['ShipFromAddress.AddressLine2']);
        unset($this->options['ShipFromAddress.AddressLine3']);
        unset($this->options['ShipFromAddress.DistrictOrCounty']);
        unset($this->options['ShipFromAddress.Email']);
        unset($this->options['ShipFromAddress.City']);
        unset($this->options['ShipFromAddress.StateOrProvinceCode']);
        unset($this->options['ShipFromAddress.CountryCode']);
        unset($this->options['ShipFromAddress.PostalCode']);
        unset($this->options['ShipFromAddress.PhoneNumber']);
    }

    /**
     * Sets the package Dimensions. (Required)
     *
     * This method sets the package dimensions to be sent in the next request.
     * This parameter is required getting a list of shipping services offers from Amazon.
     * The array provided should have the following fields:
     * <ul>
     * <li><b>Length</b> - decimal</li>
     * <li><b>Width</b> - decimal</li>
     * <li><b>Height</b> - decimal</li>
     * <li><b>Unit</b>  - string</li>
     * <li><b>PredefinedPackageDimensions</b> (optional)- string.
     * If you specify the PredefinedPackageDimensions request parameter,
     * you must not specify the Length/Width/Height/Unit request parameter</li>
     * </ul>
     * @param array $a <p>See above.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setPackageDimension($a)
    {
        if (is_null($a) || is_string($a) || !$a) {
            $this->log("Tried to set package dimension to invalid values", 'WARNING');
            return false;
        }
        $this->resetPackageDimension();
        if (is_array($a) and
            ((array_key_exists('Length', $a) && array_key_exists('Width', $a) &&
              array_key_exists('Height', $a) && array_key_exists('Unit', $a)) ||
                (array_key_exists('ShipmentRequestDetails.PredefinedPackageDimensions', $a)))) {

            if (array_key_exists('Length', $a)) {
                $this->options['ShipmentRequestDetails.PackageDimensions.Length'] = $a['Length'];
            }
            if (array_key_exists('Width', $a)) {
                $this->options['ShipmentRequestDetails.PackageDimensions.Width'] = $a['Width'];
            }
            if (array_key_exists('Height', $a)) {
                $this->options['ShipmentRequestDetails.PackageDimensions.Height'] = $a['Height'];
            }
            if (array_key_exists('Unit', $a)) {
                $this->options['ShipmentRequestDetails.PackageDimensions.Unit'] = $a['Unit'];
            }
            if (array_key_exists('PredefinedPackageDimensions', $a)) {
                $this->options['ShipmentRequestDetails.PackageDimensions.PredefinedPackageDimensions'] = $a['PredefinedPackageDimensions'];
            }
        } else {
            $this->resetPackageDimension();
            $this->log("Tried to set package dimension with invalid array", 'WARNING');
            return false;
        }
        return true;
    }

    /**
     * Resets the package dimension options.
     *
     * Since package dimension is a required parameter, these options should not be removed
     * without replacing them, so this method is not public.
     */
    protected function resetPackageDimension()
    {
        unset($this->options['ShipmentRequestDetails.PackageDimensions.Length']);
        unset($this->options['ShipmentRequestDetails.PackageDimensions.Width']);
        unset($this->options['ShipmentRequestDetails.PackageDimensions.Height']);
        unset($this->options['ShipmentRequestDetails.PackageDimensions.Unit']);
        unset($this->options['ShipmentRequestDetails.PackageDimensions.PredefinedPackageDimensions']);

    }

    /**
     * Sets Weight. (Required)
     *
     * This method sets the Weight to be sent in the next request.
     * This parameter is required for getting a list of shipping services from Amazon.
     * The array provided should have the following fields:
     * <ul>
     * <li><b>Value</b> - decimal</li>
     * <li><b>Unit</b> - Unit values: ounces, grams. String</li>
     * </ul>
     * @param array $a <p>See above.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setWeight($a)
    {
        if (is_null($a) || is_string($a) || !$a) {
            $this->log("Tried to set weight to invalid values", 'WARNING');
            return false;
        }
        $this->resetWeight();
        if (is_array($a) and
            array_key_exists('Value', $a) && array_key_exists('Unit', $a)) {
                $this->options['ShipmentRequestDetails.Weight.Value'] = $a['Value'];
                $this->options['ShipmentRequestDetails.Weight.Unit'] = $a['Unit'];
        } else {
            $this->resetWeight();
            $this->log("Tried to set weight with invalid array", 'WARNING');
            return false;
        }
        return true;
    }

    /**
     * Resets the weight options.
     *
     * Since weight is a required parameter, these options should not be removed
     * without replacing them, so this method is not public.
     */
    protected function resetWeight()
    {
        unset($this->options['ShipmentRequestDetails.Weight.Value']);
        unset($this->options['ShipmentRequestDetails.Weight.Unit']);

    }

    /**
     * Sets the MustArriveByDate. (Optional)
     *
     * This method sets the MustArriveByDate to be sent in the next request.
     * @param string $s <p>Maximum 50 characters.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setMustArriveByDate($s)
    {
        if (is_string($s)) {
            $this->options['ShipmentRequestDetails.MustArriveByDate'] = $this->genTime($s);
            return true;
        }

        return false;
    }

    /**
     * Sets the ShipDate. (Optional)
     *
     * This method sets the ShipDate to be sent in the next request.
     * @param string $s <p>Maximum 50 characters.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setShipDate($s)
    {
        if (is_string($s)) {
            $this->options['ShipmentRequestDetails.ShipDate'] = $this->genTime($s);
            return true;
        }

        return false;
    }

    /**
     * Sets the Shipping Service options. (Required)
     *
     * This method sets the Shipping Service options to be sent in the next request.
     * This parameter is required getting a list of shipping services offers from Amazon.
     * The array provided should have the following fields:
     * <ul>
     * <li><b>DeliveryExperience</b> - DeliveryExperience values:</li>
     * DeliveryConfirmationWithAdultSignature - Delivery confirmation with adult signature.
     * DeliveryConfirmationWithSignature - Delivery confirmation with signature. Required for DPD (UK).
     * DeliveryConfirmationWithoutSignature - Delivery confirmation without signature.
     * NoTracking - No delivery confirmation.</li>
     * <li><b>DeclaredValue</b> (optional array) - </li>
     * <ul>
     * <li><b>CurrencyCode</b> - string </li>
     * <li><b>Amount</b> - decimal <li>
     * </ul>
     * <li><b>Height</b> - decimal</li>
     * <li><b>Unit</b>  - string</li>
     * <li><b>CarrierWillPickUp</b> - boolean.
     * </ul>
     * @param array $a <p>See above.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setShippingServiceOptions($a)
    {
        if (is_null($a) || is_string($a) || !$a) {
            $this->log("Tried to set shipping service options to invalid values", 'WARNING');
            return false;
        }
        $this->resetShippingServiceOptions();
        if (is_array($a) and
            array_key_exists('DeliveryExperience', $a) && array_key_exists('CarrierWillPickUp', $a)) {
                $this->options['ShipmentRequestDetails.ShippingServiceOptions.DeliveryExperience'] = $a['DeliveryExperience'];
                $this->options['ShipmentRequestDetails.ShippingServiceOptions.CarrierWillPickUp'] = $a['CarrierWillPickUp'];
            if (array_key_exists('DeclaredValue', $a)) {
                $this->options['ShipmentRequestDetails.ShippingServiceOptions.DeclaredValue.CurrencyCode'] = $a['DeclaredValue']['CurrencyCode'];
                $this->options['ShipmentRequestDetails.ShippingServiceOptions.DeclaredValue.Amount'] = $a['DeclaredValue']['Amount'];
            }
        } else {
            $this->resetShippingServiceOptions();
            $this->log("Tried to set shipping service options with invalid array", 'WARNING');
            return false;
        }
        return true;
    }

    /**
     * Resets the weight options.
     *
     * Since weight is a required parameter, these options should not be removed
     * without replacing them, so this method is not public.
     */
    protected function resetShippingServiceOptions()
    {
        unset($this->options['ShipmentRequestDetails.ShippingServiceOptions.DeliveryExperience']);
        unset($this->options['ShipmentRequestDetails.ShippingServiceOptions.CarrierWillPickUp']);
        unset($this->options['ShipmentRequestDetails.ShippingServiceOptions.DeclaredValue.CurrencyCode']);
        unset($this->options['ShipmentRequestDetails.ShippingServiceOptions.DeclaredValue.Amount']);
    }

    /**
     * Specifies whether to include Complex Shipping Options or not. (Optional)
     *
     * This method Specifies whether to include Complex Shipping Options or not in the next request.
     * @param string $s <p> -boolean.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function IncludeComplexShippingOptions($s)
    {
        if (is_bool($s)) {
            $this->options['ShippingOfferingFilter.IncludeComplexShippingOptions'] = $s;
            return true;
        }

        return false;
    }

    /**
     * Fetches Shipping Services.
     *
     * @return boolean <b>FALSE</b> if error is returned
     * @throws \Amazon\Sdk\Api\Exception\InvalidConfigValue
     */
    public function fetchShippingServices()
    {
        if (!array_key_exists('ShipmentRequestDetails.AmazonOrderId', $this->options)) {
            $this->log("Amazon OrderID must be set in order to get amazon shipping services", 'WARNING');
            return false;
        }
        if (!array_key_exists('ShipmentRequestDetails.ItemList.Item.1.OrderItemId', $this->options)) {
            $this->log("Item list must be set in order to get amazon shipping services", 'WARNING');
            return false;
        }
        if (!array_key_exists('ShipmentRequestDetails.ShipFromAddress.Name', $this->options)) {
            $this->log("ShipFromAddress must be set in order to get amazon shipping services", 'WARNING');
            return false;
        }
        if (!array_key_exists('ShipmentRequestDetails.Weight.Value', $this->options)) {
            $this->log("Weight must be set in order to get amazon shipping services", 'WARNING');
            return false;
        }
        if (!array_key_exists('ShipmentRequestDetails.ShippingServiceOptions.DeliveryExperience', $this->options)) {
            $this->log("ShippingServiceOptions must be set in order to get amazon shipping services", 'WARNING');
            return false;
        }

        $url = $this->urlbase . $this->urlbranch;

        $query = $this->genQuery();
        $path = $this->options['Action'] . 'Result';
        if ($this->mockMode) {
            if (isset($this->options['ShippingOfferingFilter.IncludeComplexShippingOptions']) &&
                $this->options['ShippingOfferingFilter.IncludeComplexShippingOptions']) {
                $response = $this->fetchMockFile(self::MOCK_FILE_MERCHANT_FULFILLMENT_INTERNATIONAL);
            } else {
                $response = $this->fetchMockFile(self::MOCK_FILE_MERCHANT_FULFILLMENT_DOMESTIC);
            }
            if ($response) {
                $xml = $response->$path;
            } else {
                return false;
            }
        } else {
            $response = $this->sendRequest($url, ['Post' => $query]);
            if (!$this->checkResponse($response)) {
                return false;
            }
            $xml = simplexml_load_string($response['body'])->$path;
        }

        $this->parseXML($xml);

        return true;
    }

    /**
     * Parses XML response into array.
     *
     * This is what reads the response XML and converts it into an array.
     * @param \SimpleXMLElement $xml <p>The XML response from Amazon.</p>
     * @return boolean <b>FALSE</b> if no XML data is found
     * @throws \Amazon\Sdk\Api\Exception\InvalidConfigValue
     */
    protected function parseXML($xml)
    {
        if (!$xml) {
            return false;
        }

        foreach ($xml->ShippingServiceList->children() as $key => $data) {
            if ($key != 'ShippingService') {
                break;
            }
            $this->shippingServiceList[$this->index] = [
                'ShippingServiceName' => (string)$data->ShippingServiceName,
                'CarrierName' => (string)$data->CarrierName,
                'ShippingServiceId' => (string)$data->ShippingServiceId,
                'ShippingServiceOfferId' => (string)$data->ShippingServiceOfferId,
                'ShipDate' => (string)$data->ShipDate,
                'EarliestEstimatedDeliveryDate' => (string)$data->EarliestEstimatedDeliveryDate,
                'LatestEstimatedDeliveryDate' => (string)$data->LatestEstimatedDeliveryDate,
                'Rate' => [
                    'CurrencyCode' => (string)$data->Rate->CurrencyCode,
                    'Amount' => (string)$data->Rate->Amount
                ],
                'ShippingServiceOptions' => [
                    'DeliveryExperience' => (string)$data->ShippingServiceOptions->DeliveryExperience,
                    'CarrierWillPickUp' => (string)$data->ShippingServiceOptions->CarrierWillPickUp
                ],
                'AvailableLabelFormats' => [
                    'LabelFormat' => (array)$data->AvailableLabelFormats->LabelFormat
                ],
                'RequiresAdditionalSellerInputs' => false
            ];
            $this->index++;
        }

        if (!empty($xml->RejectedShippingServiceList->RejectedShippingService)) {
            $this->rejectedShippingServiceList = (array)$xml->RejectedShippingServiceList->RejectedShippingService;
        }

        foreach ($xml->TemporarilyUnavailableCarrierList->TemporarilyUnavailableCarrier as $key => $data) {
            $this->temporarilyUnavailableCarrierList['CarrierName'][] = (string)$data->CarrierName;
        }

        foreach ($xml->TermsAndConditionsNotAcceptedCarrierList->TermsAndConditionsNotAcceptedCarrier as $key => $data) {
            $this->termsAndConditionsNotAcceptedCarrierList['CarrierName'][] = (string)$data->CarrierName;
        }

        return true;
    }

    /**
     * Returns the list of shipping services.
     * @return array|boolean array of shipping service list, or <b>FALSE</b> if list not filled yet
     */
    public function getAvailableShippingServices()
    {
        if (isset($this->shippingServiceList)) {
            return $this->shippingServiceList;
        }
        return false;
    }

    public function getAllShippingServiceList()
    {
        $servicesList = [];
        if (!empty($this->shippingServiceList)) {
            $servicesList['AvailableShippingServices'] = $this->shippingServiceList;
        }
        if (!empty($this->rejectedShippingServiceList)) {
            $servicesList['RejectedList'] = $this->rejectedShippingServiceList;
        }
        if (!empty($this->temporarilyUnavailableCarrierList)) {
            $servicesList['TemporarilyUnavailable'] = $this->temporarilyUnavailableCarrierList;
        }
        if (!empty($this->termsAndConditionsNotAcceptedCarrierList)) {
            $servicesList['TermsAndConditionsNotAccepted'] = $this->termsAndConditionsNotAcceptedCarrierList;
        }
        return $servicesList;
    }
}
