<?php
/**
 * Copyright 2013 CPI Group, LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 *
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

namespace Amazon\Sdk\Api\Feed;

/**
 * Retrieves feeds from Amazon.
 *
 * This Amazon Feeds Core object can retrieve the results of a
 * processed feed from Amazon, which can then be saved to a file
 * specified by the user or retrieved as a single string.
 * In order to fetch feed results, the feed's ID must be given.
 */
class Result extends \Amazon\Sdk\Api\Feed\FeedsCore
{
    private $rawFeed;

    /**
     * AmazonFeedResult gets the result of a Feed from Amazon.
     *
     * The parameters are passed to the parent constructor, which are
     * in turn passed to the AmazonCore constructor. See it for more information
     * on these parameters and common methods.
     * Please note that an extra parameter comes before the usual Mock Mode parameters,
     * so be careful when setting up the object.
     * @param string $id [optional] <p>The Feed Submission ID to set for the object.</p>
     * @param \Amazon\Sdk\Api\ConfigInterface $config = null,
     * @param \Psr\Log\LoggerInterface $logger = null,
     * @param boolean $mockMode [optional] <p>This is a flag for enabling Mock Mode.
     * This defaults to <b>FALSE</b>.</p>
     * @param array|string $mockFiles [optional] <p>The files (or file) to use in Mock Mode.</p>
     */
    public function __construct(
        $id = null,
        \Amazon\Sdk\Api\ConfigInterface $config = null,
        \Psr\Log\LoggerInterface $logger = null,
        $mockMode = false,
        $mockFiles = null
    )
    {
        parent::__construct($config, $logger, $mockMode, $mockFiles);

        if ($id) {
            $this->options['FeedSubmissionId'] = $id;
        }

        $this->options['Action'] = 'GetFeedSubmissionResult';

        $this->throttleLimit = \Amazon\Sdk\Api\Environment::THROTTLE_LIMIT_FEEDRESULT;
        $this->throttleTime = \Amazon\Sdk\Api\Environment::THROTTLE_TIME_FEEDRESULT;
        $this->throttleGroup = 'GetFeedSubmissionResult';
    }

    /**
     * Sets the feed submission ID for the next request. (Required)
     *
     * This method sets the feed submission ID to be sent in the next request. This
     * parameter is required in order to retrieve a feed from Amazon.
     * @param string|integer $n <p>Must be numeric</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setFeedId($n)
    {
        if (is_numeric($n)) {
            $this->options['FeedSubmissionId'] = $n;
            return true;
        }

        return false;
    }

    /**
     * Sends a request to Amazon for a feed.
     *
     * Submits a <i>GetFeedSubmissionResult</i> request to Amazon. In order to
     * do this, a feed submission ID is required. Amazon will send back the raw results
     * of the feed as a response, which can be saved to a file using <i>saveFeed</i>.
     * @return boolean <b>FALSE</b> if something goes wrong
     */
    public function fetchFeedResult()
    {
        if (!array_key_exists('FeedSubmissionId', $this->options)) {
            $this->log("Feed Submission Id must be set in order to fetch it.", 'WARNING');
            return false;
        }

        $url = $this->urlbase . $this->urlbranch;

        $query = $this->genQuery();

        if ($this->mockMode) {
            $this->rawFeed = $this->fetchMockFile(self::MOCK_FILE_FETCH_FEED_RESULT, false);
        } else {
            $response = $this->sendRequest($url, ['Post' => $query]);

            if (!$this->checkResponse($response)) {
                return false;
            }

            $this->rawFeed = $response['body'];
        }

        return true;
    }

    /**
     * Saves the raw report data to a path you specify.
     *
     * This method will record in the log whether or not the save was successful.
     * @param string $path <p>path for the file to save the feed data in</p>
     * @return boolean <b>FALSE</b> if something goes wrong
     */
    public function saveFeed($path)
    {
        if (!isset($this->rawFeed)) {
            return false;
        }
        try {
            $saved = @file_put_contents($path, $this->rawFeed);
            if (!$saved) {
                throw new \Exception('Invalid file save path : ' . $path);
            }
            $this->log("Successfully saved feed #" . $this->options['FeedSubmissionId'] . " at $path");
            return true;
        } catch (\Exception $e) {
            $this->log(
                "Unable to save feed #" . $this->options['FeedSubmissionId'] . " " . $e->getMessage(),
                'ERROR',
                ['exception' => $e->getMessage(), 'file' => $path, 'path' => __METHOD__]
            );
            return false;
        }
    }

    /**
     * Returns the entire raw report data.
     *
     * This is useful for handling the report with an external file management system.
     * @return string|boolean The raw report data as a string, or <b>FALSE</b> if there is no data
     */
    public function getRawFeed()
    {
        if (!isset($this->rawFeed)) {
            return false;
        }
        return $this->rawFeed;
    }
}
