<?php
/**
 * Copyright 2013 CPI Group, LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 *
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

namespace Amazon\Sdk\Api;

/**
 * Submits feeds to Amazon.
 *
 * This Amazon Feeds Core object can submit feeds to Amazon.
 * In order to submit a feed, the feed's contents (as direct input or from a file)
 * and feed type must be set. Once the feed has been submitted,
 * the response from Amazon can be viewed with <i>getResponse</i>.
 */
class Feed extends \Amazon\Sdk\Api\Feed\FeedsCore
{

    /**
     *  Product Feed ~ _POST_PRODUCT_DATA_
     */
    const PRODUCT = '_POST_PRODUCT_DATA_';


    const ORDER_INVOICE_UPLOAD = '_UPLOAD_VAT_INVOICE_';

    /**
     *  Inventory Feed ~ _POST_INVENTORY_AVAILABILITY_DATA_
     */
    const PRODUCT_INVENTORY = '_POST_INVENTORY_AVAILABILITY_DATA_';

    /**
     *  Shipping Override Feed ~ _POST_PRODUCT_OVERRIDES_DATA_
     */
    const PRODUCT_OVERRIDES = '_POST_PRODUCT_OVERRIDES_DATA_';

    /**
     *  Pricing Feed ~ _POST_PRODUCT_PRICING_DATA_
     */
    const PRODUCT_PRICING = '_POST_PRODUCT_PRICING_DATA_';

    /**
     *  Product Images Feed ~ _POST_PRODUCT_IMAGE_DATA_
     */
    const PRODUCT_IMAGE = '_POST_PRODUCT_IMAGE_DATA_';

    /**
     *  Relationships Feed ~ _POST_PRODUCT_RELATIONSHIP_DATA_
     */
    const PRODUCT_RELATIONSHIP = '_POST_PRODUCT_RELATIONSHIP_DATA_';

    /**
     *  Order Acknowledgement Feed ~ _POST_ORDER_ACKNOWLEDGEMENT_DATA_
     */
    const ORDER_ACKNOWLEDGEMENT = '_POST_ORDER_ACKNOWLEDGEMENT_DATA_';

    /**
     *  Order Adjustment Feed ~ _POST_PAYMENT_ADJUSTMENT_DATA_
     */
    const ORDER_PAYMENT_ADJUSTMENT = '_POST_PAYMENT_ADJUSTMENT_DATA_';

    /**
     *  Order Fulfillment Feed ~ _POST_ORDER_FULFILLMENT_DATA_
     */
    const ORDER_FULFILLMENT = '_POST_ORDER_FULFILLMENT_DATA_';

    /**
     *  FBA Shipment Injection Fulfillment Feed~  _POST_FULFILLMENT_ORDER_REQUEST_DATA_
     */
    const ORDER_FULFILLMENT_FBA = '_POST_FULFILLMENT_ORDER_REQUEST_DATA_';

    /**
     *  FBA Fulfillment Order Cancellation Feed ~ _POST_FULFILLMENT_ORDER_CANCELLATION_REQUEST_DATA_
     */
    const ORDER_FULFILLMENT_CANCELLATION_FBA = '_POST_FULFILLMENT_ORDER_CANCELLATION_REQUEST_DATA_';

    /**
     *  (China only)
     *  Invoice Confirmation Feed ~ _POST_INVOICE_CONFIRMATION_DATA_
     */
    const ORDER_INVOICE_CONFIRMATION = '_POST_INVOICE_CONFIRMATION_DATA_';

    const MOCK_FEED = '_MOCK_FEED_';

    private $response;

    private $feedContent;

    private $feedMD5;

    /**
     * Feed submits a Feed to Amazon.
     *
     * The parameters are passed to the parent constructor, which are
     * in turn passed to the AmazonCore constructor. See it for more information
     * on these parameters and common methods.
     * @param \Amazon\Sdk\Api\ConfigInterface|null $config ,
     * @param \Psr\Log\LoggerInterface|null $logger ,
     * @param boolean $mockMode [optional] <p>This is a flag for enabling Mock Mode.
     * This defaults to <b>FALSE</b>.</p>
     * @param array|string $mockFiles [optional] <p>The files (or file) to use in Mock Mode.</p>
     */
    public function __construct(
        \Amazon\Sdk\Api\ConfigInterface $config = null,
        \Psr\Log\LoggerInterface $logger = null,
        $mockMode = false,
        $mockFiles = null
    ) {
        parent::__construct($config, $logger, $mockMode, $mockFiles);

        $marketplaceIds = $this->config->getMarketplaceId();
        if (isset($marketplaceIds) && !empty($marketplaceIds)) {
            $this->setMarketplaceListIds($marketplaceIds);
        } else {
            $this->log("Marketplace List Id is missing", 'ERROR');
        }

        $this->options['Action'] = 'SubmitFeed';

        $this->throttleLimit = \Amazon\Sdk\Api\Environment::THROTTLE_LIMIT_FEEDSUBMIT;
        $this->throttleTime = \Amazon\Sdk\Api\Environment::THROTTLE_TIME_FEEDSUBMIT;
        $this->throttleGroup = 'SubmitFeed';
    }

    /**
     * Sets the Feed Content. (Required)
     *
     * Thie method sets the feed's contents from direct input.
     * This parameter is required in order to submit a feed to Amazon.
     * @param string $s <p>The contents to put in the file.</p>
     * It can be relative or absolute.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setFeedContent($s)
    {
        if (is_string($s) && $s) {
            $this->feedContent = $s;
            $this->feedMD5 = base64_encode(md5($this->feedContent, true));
        }

        return false;
    }

    /**
     * Sets the Feed Content. (Required)
     *
     * This method loads the contents of a file to send as the feed. This
     * parameter is required in order to submit a feed to Amazon.
     * @param string $url <p>The path to a file you want to use.
     * It can be relative or absolute.</p>
     */
    public function loadFeedFile($path)
    {
        if (file_exists($path)) {
            $p = strpos($path, '/');
            if ($p == 0) {
                $this->feedContent = file_get_contents($path);
            } else {
                $url = __DIR__ . '/../../' . $path; //todo: change to current install dir
                $this->feedContent = file_get_contents($url);
            }
            $this->feedMD5 = base64_encode(md5($this->feedContent, true));
        }
    }

    /**
     * Sets the Feed Type. (Required)
     *
     * This method sets the Feed Type to be sent in the next request. This tells
     * Amazon how the Feed should be processsed.
     * This parameter is required in order to submit a feed to Amazon.
     * @param string $s <p>A value from the list of valid Feed Types.
     * See the comment inside the function for the complete list.</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setFeedType($s)
    {
        if (is_string($s) && $s) {
            $this->options['FeedType'] = $s;
            return true;
        }

        return false;

        /**
         * @link https://docs.developer.amazonservices.com/en_UK/feeds/Feeds_FeedType.html
         * List of valid Feed Types:
         * XML Feeds:
         *      Product Feed ~ _POST_PRODUCT_DATA_
         *      Relationships Feed ~ _POST_PRODUCT_RELATIONSHIP_DATA_
         *      Single Format Item Feed ~ _POST_ITEM_DATA_
         *      Shipping Override Feed ~ _POST_PRODUCT_OVERRIDES_DATA_
         *      Product Images Feed ~ _POST_PRODUCT_IMAGE_DATA_
         *      Pricing Feed ~ _POST_PRODUCT_PRICING_DATA_
         *      Inventory Feed ~ _POST_INVENTORY_AVAILABILITY_DATA_
         *      Order Acknowledgement Feed ~ _POST_ORDER_ACKNOWLEDGEMENT_DATA_
         *      Order Fulfillment Feed ~ _POST_ORDER_FULFILLMENT_DATA_
         *      FBA Shipment Injection Fulfillment Feed~  _POST_FULFILLMENT_ORDER_REQUEST_DATA_
         *      FBA Shipment Injection ~ _POST_FULFILLMENT_ORDER_CANCELLATION_
         *      Cancellation Feed ~ _REQUEST_DATA_
         *      Order Adjustment Feed ~ _POST_PAYMENT_ADJUSTMENT_DATA_
         *      Invoice Confirmation Feed ~ _POST_INVOICE_CONFIRMATION_DATA_
         * Tab Delimited Feeds:
         *      Flat File Listings Feed ~ _POST_FLAT_FILE_LISTINGS_DATA_
         *      Flat File Order Acknowledgement Feed ~ _POST_FLAT_FILE_ORDER_ACKNOWLEDGEMENT_DATA_
         *      Flat File Order Fulfillment Feed ~ _POST_FLAT_FILE_FULFILLMENT_DATA_
         *      Flat File FBA Shipment Injection Fulfillment Feed ~ _POST_FLAT_FILE_FULFILLMENT_ORDER_REQUEST_DATA_
         *      Flat File FBA Shipment Injection Cancellation Feed ~
         *                                                  _POST_FLAT_FILE_FULFILLMENT_ORDER_CANCELLATION_REQUEST_DATA_
         *      FBA Flat File Create Inbound Shipment Feed ~ _POST_FLAT_FILE_FBA_CREATE_INBOUND_SHIPMENT_
         *      FBA Flat File Update Inbound Shipment Feed ~ _POST_FLAT_FILE_FBA_UPDATE_INBOUND_SHIPMENT_
         *      FBA Flat File Shipment Notification Feed ~ _POST_FLAT_FILE_FBA_SHIPMENT_NOTIFICATION_FEED_
         *      Flat File Order Adjustment Feed ~ _POST_FLAT_FILE_PAYMENT_ADJUSTMENT_DATA_
         *      Flat File Invoice Confirmation Feed ~ _POST_FLAT_FILE_INVOICE_CONFIRMATION_DATA_
         *      Flat File Inventory Loader Feed ~ _POST_FLAT_FILE_INVLOADER_DATA_
         *      Flat File Music Loader File ~ _POST_FLAT_FILE_CONVERGENCE_LISTINGS_DATA_
         *      Flat File Book Loader File ~ _POST_FLAT_FILE_BOOKLOADER_DATA_
         *      Flat File Video Loader File ~ _POST_FLAT_FILE_LISTINGS_DATA_
         *      Flat File Price and Quantity Update File ~ _POST_FLAT_FILE_PRICEANDQUANTITYONLY_UPDATE_DATA_
         *      Product Ads Flat File Feed ~ _POST_FLAT_FILE_SHOPZILLA_DATA_
         * Universal Information Exchange Environment (UIEE) Feeds:
         *      UIEE Inventory File ~ _POST_UIEE_BOOKLOADER_DATA_
         */
    }

    /**
     * Turns on or off Purge mode. (Optional)
     *
     *
     * <b>WARNING! This parameter can only be used once every 24 hours!</b>
     *
     * This method sets whether or not the tab delimited feed you provide should
     * completely replace old data. Use this parameter only in exceptional cases.
     * If this is not set, Amazon assumes it to be false.
     * @param boolean|string $s [optional] <p>The value "true" or "false", either as
     * a boolean or a string. It defaults to "true".</p>
     * @return boolean <b>FALSE</b> if improper input
     */
    public function setPurge($s = 'true')
    {
        if ($s == 'true' || ($s && is_bool($s))) {
            $this->log("Purge mode activated.", 'WARNING', ['path' => __METHOD__]);
            $this->options['PurgeAndReplace'] = 'true';
            $this->throttleTime = 86400;
            return true;
        } elseif ($s == 'false' || (!$s && is_bool($s))) {
            $this->log("Purge mode deactivated.", 'WARNING', ['path' => __METHOD__]);
            $this->options['PurgeAndReplace'] = 'false';
            $this->throttleTime = \Amazon\Sdk\Api\Environment::THROTTLE_TIME_FEEDSUBMIT;
            return true;
        }

        return false;
    }

    /**
     * @param $options
     * @return bool
     */
    public function setFeedOptions($options){
        if (is_string($options) && $options) {
            $this->options['FeedOptions'] = $options;
            return true;
        }

        return false;
    }

    /**
     * Submits a feed to Amazon.
     *
     * Submits a <i>SubmitFeed</i> request to Amazon. In order to do this, both
     * the feed's contents and feed type are required. The request will not be
     * sent if either of these are not set. Amazon will send a response back,
     * which can be retrieved using <i>getResponse</i>.
     * @return boolean <b>FALSE</b> if something goes wrong
     */
    public function submitFeed()
    {
        if (!$this->feedContent) {
            $this->log("Feed's contents must be set in order to submit it!", 'WARNING', ['path' => __METHOD__]);
            return false;
        }
        if (!array_key_exists('FeedType', $this->options)) {
            $this->log("Feed Type must be set in order to submit a feed!", 'WARNING', ['path' => __METHOD__]);
            return false;
        }

        $url = $this->urlbase . $this->urlbranch;

        $query = $this->genQuery();

        $path = $this->options['Action'] . 'Result'; //SubmitFeedResult
        if ($this->mockMode) {
            $response = $this->fetchMockFile(self::MOCK_FILE_SUBMIT_FEED);
            if ($response) {
                $xml = $response->$path;
            } else {
                return false;
            }
        } else {
            $headers = $this->genHeader();
            $response = $this->sendRequest(
                "$url?$query",
                ['Header' => $headers, 'Post' => $this->feedContent]
            );

            if (!$this->checkResponse($response)) {
                return false;
            }

            if (isset($response['code']) && $response['code'] == '200') {
                $body = strstr($response['body'], '<');
                $xml = simplexml_load_string($body)->$path;
            } else {
                $this->log("Unexpected response: " . var_export($response, true), 'WARNING', ['path' => __METHOD__]);
                $xml = simplexml_load_string($response['body'])->$path;
            }
        }

        $this->parseXML($xml->FeedSubmissionInfo);

        return true;
    }

    /**
     * Generates array for Header.
     *
     * This method creates the Header array to use with cURL. It contains the Content MD5.
     * @return array
     */
    protected function genHeader()
    {
        $return[0] = "Content-MD5:" . $this->feedMD5;
        return $return;
    }

    /**
     * Checks whether or not the response is OK.
     *
     * Verifies whether or not the HTTP response has the 200 OK code. If the code
     * is not 200, the incident and error message returned are logged. This method
     * is different than the ones used by other objects due to Amazon sending
     * 100 Continue responses in addition to the usual response.
     * @param array $r <p>The HTTP response array. Expects the array to have
     * the fields <i>code</i>, <i>body</i>, and <i>error</i>.</p>
     * @return boolean <b>TRUE</b> if the status is 200 OK, <b>FALSE</b> otherwise.
     */
    protected function checkResponse($r)
    {
        if (!is_array($r)) {
            $this->log("No Response found.", 'WARNING', ['response' => $r]);
            return false;
        }
        //for dealing with 100 response
        if (array_key_exists('error', $r) && $r['ok'] == 0) {
            $this->log("Response Error: " . $r['error'], 'ERROR', ['response' => $r]);
            return false;
        } else {
            $this->log("Response Success.", 'DEBUG');
            return true;
        }
    }

    /**
     * Parses XML response into array.
     *
     * This is what reads the response XML and converts it into an array.
     * @param \SimpleXMLElement $xml <p>The XML response from Amazon.</p>
     * @return boolean <b>FALSE</b> if no XML data is found
     */
    protected function parseXML($xml)
    {
        if (!$xml) {
            return false;
        }

        $this->response = [];
        $this->response['FeedSubmissionId'] = (string)$xml->FeedSubmissionId;
        $this->response['FeedType'] = (string)$xml->FeedType;
        $this->response['SubmittedDate'] = (string)$xml->SubmittedDate;
        $this->response['FeedProcessingStatus'] = (string)$xml->FeedProcessingStatus;

        $this->log(
            "Successfully submitted feed #" . $this->response['FeedSubmissionId'] .
            ' (' . $this->response['FeedType'] . ')'
        );

        return true;
    }

    /**
     * Returns the response data in array.
     *
     * It will contain the following fields:
     * <ul>
     * <li><b>FeedSubmissionId</b> - Unique ID for the feed submission</li>
     * <li><b>FeedType</b> - Same as the feed type you gave</li>
     * <li><b>SubmittedDate</b> - The timestamp for when the Feed was received</li>
     * <li><b>FeedProcessingStatus</b> - The status of the feed, likely "_SUBMITTED_"</li>
     * </ul>
     * @return array|boolean
     */
    public function getResponse()
    {
        if (isset($this->response)) {
            return $this->response;
        }

        return false;
    }
}
