# Phalcon\Logger\Adapter

Usage examples of the adapters available here:

## Database

Adapter to store logs in a database table:

```php
use Phalcon\Db\Adapter\Pdo\Mysql;
use Phalcon\Logger\Adapter\Database as DbLogger;

$di->set(
    'logger',
    function () {
        $connection = new Mysql(
            [
                'host'     => 'localhost',
                'username' => 'root',
                'password' => 'secret',
                'dbname'   => 'audit',
            ]
        );

        $logger = new DbLogger(
            'errors',
            [
                'db'    => $connection,
                'table' => 'logs',
            ]
        );

        return $logger;
    }
);
```

The following table is used to store the logs:

```sql
CREATE TABLE `logs` (
  `id` int(10) unsigned NOT NULL AUTO_INCREMENT,
  `name` varchar(32) DEFAULT NULL,
  `type` int(3) NOT NULL,
  `content` text,
  `created_at` int(18) unsigned NOT NULL,
  PRIMARY KEY (`id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8
```

## Firelogger

Adapter to send messages to the [Firelogger][1] console inside the [Firebug][2] in your browser.

```php
use Phalcon\Logger\Adapter\Firelogger;

$logger = new Firelogger('debug');

$logger->log('Plain Message');
$logger->info('Info Message');
// etc
```

## UDP logger

Adapter to send messages by UDP protocol to external server

```php
use Phalcon\Logger\Adapter\Udplogger as UdpLogger;

$di->set(
    'logger',
    function () {
        $logger = new UdpLogger(
            'errors',
            [
                'url'  => $url,
                'port' => $port,
            ]
        );

        return $logger;
    }
);
```

## Multiple file logger

Adapter `Phalcon\Logger\Adapter\File\Multiple` can be used to log messages to multiple files. This is similar to the core
`Phalcon\Logger\Adapter\File` adapter with two important distinctions:
* messages are logged into separate files by their log levels
* prefix for log files can be specified, thus allowing us to have separate log files for individual applications in a multi-app project

```php
use Phalcon\Logger\Adapter\File\Multiple as MultipleLogger;
use Phalcon\Logger;

$logger = new MultipleLogger(__DIR__ . '/../logs');

$logger->debug('Hello world');      // this is logged into debug.log
$logger->info('Hello world');       // this is logged into info.log
$logger->warning('Hello world');    // this is logged into warning.log
$logger->error('Hello world');      // this is logged into error.log
$logger->log('Hello world', Logger::CRITICAL);  // this is logged into critical.log

```

### Message grouping by their level

Note that similar-level logs are logged into the same file.
The log level groups are defined within the `getTypeString()` method. You may overload this method to fit your needs:

```php
private function getTypeString($type)
{
    switch ($type) {
        case Logger::EMERGENCY:
        case Logger::EMERGENCE:
        case Logger::CRITICAL:
            // emergence, critical
            return 'critical';
        case Logger::ALERT:
        case Logger::ERROR:
            // error, alert
            return 'error';
        case Logger::WARNING:
            // warning
            return 'warning';
        case Logger::NOTICE:
        case Logger::INFO:
            // info, notice
            return 'info';
        case Logger::DEBUG:
        case Logger::CUSTOM:
        case Logger::SPECIAL:
        default:
            // debug, log, custom, special
            return 'debug';
    }
}
```

Thus, by default both errors and alerts are logged into error.log, and both emergence and critical messages are logged into critical.log, etc.

### Configuration options

Optionally, you may pass configuration options to the logger constructor:

```php
use Phalcon\Logger\Adapter\File\Multiple as MultipleLogger;

$logger = new MultipleLogger(
    __DIR__ . '/../logs',
    [
        // filename prefix to all logs generated by this logger. Defaults to ""
        'prefix' => 'myapp-',
        // filename extension to all logs generated by this logger. Defaults to "log"
        'extension' => 'txt',
    ]
);
```


[1]: http://firelogger.binaryage.com/
[2]: https://getfirebug.com/
