<?php

/**
 * CedCommerce
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the End User License Agreement (EULA)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://cedcommerce.com/license-agreement.txt
 *
 * @category    Ced
 * @package     Ced_Woocommercewebapi
 * @author      CedCommerce Core Team <connect@cedcommerce.com>
 * @copyright   Copyright CEDCOMMERCE (http://cedcommerce.com/)
 * @license     http://cedcommerce.com/license-agreement.txt
 */

namespace App\Woocommercewebapi\Components;

use App\Woocommercewebapi\Components\Core\Common As CoreCommon;
use App\Apiconnect\Models\Apps\Shop;

/**
 * Class Webhook
 * @package App\Woocommercewebapi\Components
 */
class Webhook extends CoreCommon
{
    /**
     * @param array $data
     * @return mixed
     */
    public function getAllWebhooks($data = []) {
        $shopData = $this->getCurrentShop();
        if (!isset($shopData[self::SHOP_WEBHOOKS_FIELD])) {
            return [
                'code' => 200,
                'success' => true,
                'message' => 'Succeeded',
                'data' => []
            ];
        }

        if (count($shopData[self::SHOP_WEBHOOKS_FIELD]) == 0) {
            return [
                'code' => 200,
                'success' => true,
                'message' => 'Succeeded',
                'data' => []
            ];
        }
        $data['include'] = array_values($shopData[self::SHOP_WEBHOOKS_FIELD]);
        return $this->rest(
            'GET',
            self::GET_WEBHOOKS_END_POINT,
            $data,
            []
        );
    }

    /**
     * @param array $data
     * @return mixed
     */
    public function getWebhookByTopic($data = []) {
        $shopData = $this->getCurrentShop();
        if (!isset($shopData[self::SHOP_WEBHOOKS_FIELD])) {
            return [
                'success' => false,
                'message' => 'No webhook found.'
            ];
        }
        $topic = strtolower(trim($data['topic']));
        if (!isset($shopData[self::SHOP_WEBHOOKS_FIELD][$topic])) {
            return [
                'success' => false,
                'message' => 'Webhook not available.'
            ];
        }
        $endPoint = str_replace(
            '{webhook_id}',
            $shopData[self::SHOP_WEBHOOKS_FIELD][$topic],
            self::GET_RETRIEVE_A_WEBHOOKS_END_POINT
        );
        return $this->rest(
            'GET',
            $endPoint,
            $data,
            []
        );
    }

    /**
     * Will create webhook at woocommerce.
     *
     * @param array $data array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function createWebhook( $data = [] ) {
        $response = $this->rest(
            'POST',
            self::CREATE_A_WEBHOOK_END_POINT,
            $data,
            []
        );
        return [
            'success' => $this->success,
            'data'    => $response
            ];
    }

    /**
     * @param array $data
     * @return mixed
     */
    public function updateWebhook($data = []) {
        $shopData = $this->getCurrentShop();
        if (!isset($shopData[self::SHOP_WEBHOOKS_FIELD])) {
            return [
                'success' => false,
                'message' => 'No webhook found.'
            ];
        }
        $topic = strtolower(trim($data['topic']));
        if (!isset($shopData[self::SHOP_WEBHOOKS_FIELD][$topic])) {
            return [
                'success' => false,
                'message' => 'Webhook not available.'
            ];
        }
        $endPoint = str_replace(
            '{webhook_id}',
            $shopData[self::SHOP_WEBHOOKS_FIELD][$topic],
            self::UPDATE_A_WEBHOOK_END_POINT
        );
        return $this->rest(
            'PUT',
            $endPoint,
            $data,
            []
        );
    }
    /**
     * @param array $data
     * @return mixed
     */
    public function deleteWebhook($data = []) {
        
        $data['force'] = 'true'; /* permanent delete webhook */
        $endPoint = str_replace(
            '{webhook_id}',
            $data['id'],
            self::DELETE_A_WEBHOOK_END_POINT
        );
        return $this->rest(
            'DELETE',
            $endPoint,
            $data,
            []
        );
    }
    // /**
    //  * @param array $data
    //  * @return mixed
    //  */
    // public function deleteWebhook($data = []) {
    //     $shopData = $this->getCurrentShop();
    //     if (!isset($shopData[self::SHOP_WEBHOOKS_FIELD])) {
    //         return [
    //             'success' => false,
    //             'message' => 'No webhook found.'
    //         ];
    //     }
    //     $topic = strtolower(trim($data['topic']));
    //     if (!isset($shopData[self::SHOP_WEBHOOKS_FIELD][$topic])) {
    //         return [
    //             'success' => false,
    //             'message' => 'Webhook not available.'
    //         ];
    //     }
    //     $data['force'] = 'true'; /* permanent delete webhook */
    //     $endPoint = str_replace(
    //         '{webhook_id}',
    //         $shopData[self::SHOP_WEBHOOKS_FIELD][$topic],
    //         self::DELETE_A_WEBHOOK_END_POINT
    //     );
    //     $response = $this->rest(
    //         'DELETE',
    //         $endPoint,
    //         $data,
    //         []
    //     );
    //     if ($response['success']) {
    //         $webhookData[self::SHOP_WEBHOOKS_FIELD] = $shopData[self::SHOP_WEBHOOKS_FIELD];
    //         unset($webhookData[self::SHOP_WEBHOOKS_FIELD][$topic]);
    //         $this->updateWebhooksInShop($webhookData);
    //     }
    //     return $response;
    // }

    /**
     * @param array $data
     * @return mixed
     */
    public function batchUpdateWebhooks($data = []) {
        /* check topic must be available & should be unique */
        $webhookTopics = [];
        $createWebhookTopics = [];
        $updateWebhookTopics = [];
        $deleteWebhookTopics = [];
        if (isset($data['create'])) {
            foreach ($data['create'] as $createData) {
                if (!isset($createData['name']) ||
                    !isset($createData['topic']) ||
                    !isset($createData['delivery_url'])
                ) {
                    return [
                        'success' => false,
                        'message' => '\'topic\', \'name\', \'delivery_url\' must be available to create webhooks.'
                    ];
                }
                $topic = strtolower(trim($createData['topic']));
                if (in_array($topic, $webhookTopics)) {
                    return [
                        'success' => false,
                        'message' => '\'topic\' must be unique.'
                    ];
                } else {
                    $webhookTopics[] = $topic;
                    $createWebhookTopics[] = $topic;
                }
            }
        }
        if (isset($data['update'])) {
            foreach ($data['update'] as $updateData) {
                if (!isset($updateData['topic'])) {
                    return [
                        'success' => false,
                        'message' => '\'topic\' must be available to update webhooks.'
                    ];
                }
                $topic = strtolower(trim($updateData['topic']));
                if (in_array($topic, $webhookTopics)) {
                    return [
                        'success' => false,
                        'message' => '\'topic\' must be unique.'
                    ];
                } else {
                    $webhookTopics[] = $topic;
                    $updateWebhookTopics[] = $topic;
                }
            }
        }
        if (isset($data['delete'])) {
            foreach ($data['delete'] as $deleteData) {
                if (!isset($deleteData['topic'])) {
                    return [
                        'success' => false,
                        'message' => '\'topic\' must be available to delete webhooks.'
                    ];
                }
                $topic = strtolower(trim($deleteData['topic']));
                if (in_array($topic, $webhookTopics)) {
                    return [
                        'success' => false,
                        'message' => '\'topic\' must be unique.'
                    ];
                } else {
                    $webhookTopics[] = $topic;
                    $deleteWebhookTopics[] = $topic;
                }
            }
        }
        unset($webhookTopics);
        /* check topic must be available & should be unique */

        $shopData = $this->getCurrentShop();

        /* topic must be available in shop for update/delete*/
        if (isset($data['update']) || isset($data['delete'])) {
            if (!isset($shopData[self::SHOP_WEBHOOKS_FIELD])) {
                return [
                    'success' => false,
                    'message' => 'No webhook found to update or delete.'
                ];
            }
            $notAvailableWebhooks = [];
            foreach ($updateWebhookTopics as $topic) {
                if (!isset($shopData[self::SHOP_WEBHOOKS_FIELD][$topic])) {
                    $notAvailableWebhooks[] = $topic;
                }
            }
            foreach ($deleteWebhookTopics as $topic) {
                if (!isset($shopData[self::SHOP_WEBHOOKS_FIELD][$topic])) {
                    $notAvailableWebhooks[] = $topic;
                }
            }
            unset($updateWebhookTopics);
            unset($deleteWebhookTopics);
            if (count($notAvailableWebhooks)) {
                return [
                    'success' => false,
                    'message' => implode(', ', $notAvailableWebhooks) . ' webhook(s) not available to update or delete.'
                ];
            }
        }
        /* topic must be available in shop update/delete */

        /* new webhooks topic must not be available in shop */
        if (isset($data['create'])) {
            if (isset($shopData[self::SHOP_WEBHOOKS_FIELD])) {
                $alreadyCreatedWebhooks = [];
                foreach ($createWebhookTopics as $topic) {
                    if (isset($shopData[self::SHOP_WEBHOOKS_FIELD][$topic])) {
                        $alreadyCreatedWebhooks[] = $topic;
                    }
                }
                unset($createWebhookTopics);
                if (count($alreadyCreatedWebhooks)) {
                    return [
                        'success' => false,
                        'message' => implode(', ', $alreadyCreatedWebhooks) . ' webhook(s) already created.'
                    ];
                }
            }
        }
        /* new webhooks topic must not be available in shop */

        /* set id to update webhook */
        if (isset($data['update'])) {
            foreach ($data['update'] as $key => $updateData) {
                $topic = strtolower(trim($updateData['topic']));
                $data['update'][$key]['id'] = $shopData[self::SHOP_WEBHOOKS_FIELD][$topic];
            }
        }
        if (isset($data['delete'])) {
            foreach ($data['delete'] as $key => $deleteData) {
                $topic = strtolower(trim($deleteData['topic']));
                $data['delete'][$key] = $shopData[self::SHOP_WEBHOOKS_FIELD][$topic];
            }
        }
        /* set id to update webhook */
        $response = $this->rest(
            'POST',
            self::BATCH_UPDATE_WEBHOOKS_END_POINT,
            $data,
            []
        );

        /* update webhook data in db */
        if ($this->success) {
            $webhookData[self::SHOP_WEBHOOKS_FIELD] = $shopData[self::SHOP_WEBHOOKS_FIELD];
            if (isset($response['data']['create'])) {
                foreach ($response['data']['create'] as $createData) {
                    $webhookData[self::SHOP_WEBHOOKS_FIELD][strtolower(trim($createData['topic']))] = $createData['id'];
                }
            }
            if (isset($response['data']['delete'])) {
                foreach ($response['data']['delete'] as $deleteData) {
                    if (isset($webhookData[self::SHOP_WEBHOOKS_FIELD][strtolower(trim($deleteData['topic']))]))
                        unset($webhookData[self::SHOP_WEBHOOKS_FIELD][strtolower(trim($deleteData['topic']))]);
                }
            }
            $this->updateWebhooksInShop($webhookData);
        }
        /* update webhook data in db */

        return $response;
    }

    /**
     * @param array $webhookData
     * @return bool
     */
    private function updateWebhooksInShop($webhookData = []) {
        $shopData = $this->getCurrentShop();
        $shopObject = $this->di->getObjectManager()->get(Shop::class);
        $result = $shopObject->getCollection()->updateOne(['_id' => $shopData['_id']],['$set' => $webhookData]);
        if ($result->getMatchedCount()) {
            $shopData = $shopObject->getById($shopData['_id']);
            $this->di->getRegistry()->setCurrentShop($shopData);
            return true;
        }
        return false;
    }

}