<?php

/**
 * CedCommerce
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the End User License Agreement (EULA)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://cedcommerce.com/license-agreement.txt
 *
 * @category    Ced
 * @package     Ced_Woocommercewebapi
 * @author      CedCommerce Core Team <connect@cedcommerce.com>
 * @copyright   Copyright CEDCOMMERCE (http://cedcommerce.com/)
 * @license     http://cedcommerce.com/license-agreement.txt
 */

namespace App\Woocommercewebapi\Components;
use App\Woocommercewebapi\Components\Core\Common As CoreCommon;

/**
 * Class Product
 * @package App\Woocommercewebapi\Components
 */
class Product extends CoreCommon {
    
    /**
     * Get all products.
     *
     * @param array $data request params.
     * @since 1.0.0
     * @return array
     */
    public function getAllProducts( $data = [] ) {
        $response = $this->rest( 'GET', self::GET_PRODUCTS_END_POINT, $data, [], true );
        if ( isset( $response['data'] ) && ! empty( $response['headers'] ) ) {
            $finalData = [ 'products' => $response['data'] ];
            $headers   = $response['headers'];
            foreach ( $headers as $key => $headerData ) {
                if ( 'x-wp-total' == strtolower( $key ) ) {
                    $finalData['totalProducts'] = (int) ( $headerData[0] ?? 0 );
                }
                if ( 'x-wp-totalpages' == strtolower( $key ) ) {
                    $finalData['totalPages'] = (int) ( $headerData[0] ?? 0 );
                }
            }
        } else {
            $finalData = $response;
        }
        return [
            'success' => $this->success,
            'data'    => $finalData
        ];
    }

    /**
     * Will get product variations.
     *
     * @param array $data array containing request data.
     * @since 1.0.0
     * @return array
     */
    public function getProductVariations( $data = [] ) {
        $data = array_merge(
            array(
                'per_page'   => 100,
                'page'       => 1,
                'product_id' => false
            ),
            $data
        );
        $endPoint = str_replace(
            '{product_id}',
            $data['product_id'],
            self::GET_PRODUCT_VARIATION_END_POINT
        );
        $response = $this->rest( 'GET', $endPoint, $data, [], true );
        if ( isset( $response['data'] ) && ! empty( $response['headers'] ) ) {
            $finalData = [ 'products' => $response['data'] ];
            $headers   = $response['headers'];
            foreach ( $headers as $key => $headerData ) {
                if ( 'x-wp-total' == strtolower( $key ) ) {
                    $finalData['totalProducts'] = (int) ( $headerData[0] ?? 0 );
                }
                if ( 'x-wp-totalpages' == strtolower( $key ) ) {
                    $finalData['totalPages'] = (int) ( $headerData[0] ?? 0 );
                }
            }
        } else {
            $finalData = $response;
        }
        return [
            'success' => $this->success,
            'data'    => $finalData
        ];
    }

    /**
     * Will get single product data.
     *
     * @param array $data array containing the product id.
     * @since 1.0.0
     * @return array
     */
    public function getProductById( $data = [] ) {
        $endPoint = str_replace(
            '{product_id}',
            $data['product_id'],
            self::GET_RETRIEVE_A_PRODUCT_END_POINT
        );
        $response = $this->rest(
            'GET',
            $endPoint,
            $data,
            []
        );
        return [
            'success' => $this->success,
            'data'    => $response
        ];
    }

    /**
     * @param array $data
     * @return mixed
     */
    public function getMetaAttributes($data = []) {
        $response = $this->rest(
            'GET',
            self::GET_META_ATTRIBUTES,
            $data,
            []
        );
        $data = [];
        if ( ! empty($response['meta_data'])) {
                $metaData = base64_decode($response['meta_data']);
                $req_data = base64_decode($response['req_data']);
                $unit_data = base64_decode($response['unit_data']);
                $data = explode(',', $metaData);
                $req_data = explode(',', $req_data);
                $unit_data = explode(',', $unit_data);
        }
        $response['success' ] = $this->success;
        $response[ 'data']['meta_data'] = $data;
        $response[ 'data']['req_data']  = $req_data;
        $response[ 'data']['unit_data'] = $unit_data;
        return $response;
    }

    /**
     * Will get product categories.
     *
     * @param array $data array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function getProductsCategories( $data = [] ) {
        $response = $this->rest( 'GET', self::GET_PRODUCTS_CATEGORIES, $data, [], true );
        if ( ! empty( $response['data'] ) && ! empty( $response['headers'] ) ) {
            $finalData = [ 'categories' => $response['data'] ];
            $headers   = $response['headers'];
            foreach ( $headers as $key => $headerData ) {
                if ( 'x-wp-total' == $key ) {
                    $finalData['totalCategories'] = (int) ( $headerData[0] ?? 0 );
                }
                if ( 'x-wp-totalpages' == $key ) {
                    $finalData['totalPages'] = (int) ( $headerData[0] ?? 0 );
                }
            }
        } else {
            $finalData = $response;
        }
        
        return [
            'success' => $this->success,
            'data'    => $finalData
        ];
    }

    /**
     * Get product attributes.
     *
     * @param array $data array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function getProductsAttributes($data = []) {
        $response = $this->rest(
            'GET',
            self::GET_PRODUCTS_ATTRIBUTES,
            $data,
            []
        );
        return [
            'success' => $this->success,
            'data'    => $response
        ];
    }

    /**
     * Get product attributes.
     *
     * @param array $data array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function getAttributeValues($data = []) {
        $endPoint = str_replace(
            '{attribute_id}',
            $data['attribute_id'],
            self::GET_ATTRIBUTE_VALUES
        );
        $response = $this->rest(
            'GET',
            $endPoint,
            $data,
            []
        );
        return [
            'success' => $this->success,
            'data' => $response
        ];
    }

    /**
     * @param array $data
     * @return mixed
     */
    public function updateProduct($data = []) {
        $this->di->getLog()->logContent('UpdateInv | updateProduct ' . json_encode($data), 'info', 'inv-remote-data.log');
        $params = array();
        $params = $data['data'];
        $endPoint = str_replace(
            '{product_id}',
            $data['product_id'],
            self::UPDATE_A_PRODUCT_END_POINT
        );
        return $this->rest(
            'PUT',
            $endPoint,
            $params,
            []
        );
    }


    /**
     * @param array $data
     * @return mixed
     */
    public function updateProductVariation($data = []) {
        $this->di->getLog()->logContent('UpdateInv | updateProductVariation ' . json_encode($data), 'info', 'inv-remote-data.log');
        $params = array();
        $params = $data['data'];
        $endPoint = str_replace(
            '{variation_id}',
            $data['variation_id'],
            self::UPDATE_A_PRODUCT_VARIATION_END_POINT
        );
        $endPoint = str_replace("{product_id}", $data['product_id'], $endPoint );
        return $this->rest(
            'PUT',
            $endPoint,
            $params,
            []
        );
    }

    /**
     * Will get woocommerce product settings.
     *
     * @param array $params array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function getProductSettings( $params = array() ) {
        $response = $this->rest(
            'GET',
            self::GET_PRODUCT_SETTINGS,
            $params,
            []
        );
        return [
            'success' => $this->success,
            'data'    => $response
        ];
    }


    /**
     * Will get woocommerce general settings.
     *
     * @param array $params array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function getGeneralSettings( $params = array() ) {
        $response = $this->rest(
            'GET',
            self::GET_GENERAL_SETTINGS,
            $params,
            []
        );
        return [
            'success' => $this->success,
            'data'    => $response
        ];
    }

    /**
     * Will upsert products at woocommerce.
     *
     * @param array $params array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function upsertProducts($params = array())
    {
        if (empty($params['products'])) {
            return [
                'success' => false,
                'message' => 'Oops! Products are required to upsert.'
            ];
        }
        $response = $this->rest(
            'POST',
            self::CREATE_UPDATE_PRODUCTS_BATCH,
            $params['products'],
            []
        );
        return [
            'success' => $this->success,
            'data'    => $response
        ];
    }

    /**
     * Will upsert variation products at woocommerce.
     *
     * @param array $params array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function upsertVariation($params = array())
    {
        if (empty($params['products']) || empty($params['parent_id'])) {
            return [
                'success' => false,
                'message' => 'Oops! Products are required to upsert.'
            ];
        }
        $response = $this->rest(
            'POST',
            str_replace('{product_id}', $params['parent_id'], self::CREATE_UPDATE_VARIATION_PRODUCTS_BATCH),
            $params['products'],
            []
        );
        return [
            'success' => $this->success,
            'data'    => $response
        ];
    }
}