<?php

/**
 * CedCommerce
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the End User License Agreement (EULA)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://cedcommerce.com/license-agreement.txt
 *
 * @category    Ced
 * @package     Ced_Woocommercewebapi
 * @author      CedCommerce Core Team <connect@cedcommerce.com>
 * @copyright   Copyright CEDCOMMERCE (http://cedcommerce.com/)
 * @license     http://cedcommerce.com/license-agreement.txt
 */

namespace App\Woocommercewebapi\Components\Core;
use Exception;
use GuzzleHttp\Client;

/**
 * Common class to handle all requests.
 *
 * @since 1.0.0
 */
class Common extends \App\Apiconnect\Components\Authenticate\Common {

    const REST_URL_PREFIX = 'woocommerce/';

    const WOOCOMMERCE_SERVERLESS_ENDPOINT = 'https://9dndz71zua.execute-api.ap-southeast-1.amazonaws.com/liveWoocommerce/';

    const WOOCOMMERCE_DOMAIN_FIELD_NAME = 'domain';

    const CONSUMER_KEY_FIELD_NAME = 'consumer_key';

    const CONSUMER_SECRET_FIELD_NAME = 'consumer_secret';

    const AUTHORIZATION_PREFIX = 'Basic ';

    const SHOP_NOT_FOUND_ERROR_MESSAGE = 'Shop not found.';

    const AUTHENTICATION_ERROR_MESSAGE = 'You are unable to access. Please authenticate first.';

    const SHOP_WEBHOOKS_FIELD = 'webhooks';

    /**
     * End Point's Suffix start
     */

    const GET_PRODUCTS_END_POINT = '/wp-json/wc/v3/products';

    const GET_RETRIEVE_A_PRODUCT_END_POINT = '/wp-json/wc/v3/products/{product_id}';

    const GET_PRODUCT_VARIATION_END_POINT = '/wp-json/wc/v3/products/{product_id}/variations';

    const UPDATE_A_PRODUCT_VARIATION_END_POINT = '/wp-json/wc/v3/products/{product_id}/variations/{variation_id}';

    const UPDATE_A_PRODUCT_END_POINT = '/wp-json/wc/v3/products/{product_id}';
    
    const GET_WEBHOOKS_END_POINT = '/wp-json/wc/v3/webhooks';

    const GET_RETRIEVE_A_WEBHOOKS_END_POINT = '/wp-json/wc/v3/webhooks/{webhook_id}';

    const CREATE_A_WEBHOOK_END_POINT = '/wp-json/wc/v3/webhooks';

    const UPDATE_A_WEBHOOK_END_POINT = '/wp-json/wc/v3/webhooks/{webhook_id}';

    const DELETE_A_WEBHOOK_END_POINT = '/wp-json/wc/v3/webhooks/{webhook_id}';

    const BATCH_UPDATE_WEBHOOKS_END_POINT = '/wp-json/wc/v3/webhooks/batch';

    const CREATE_AN_ORDER_END_POINT = '/wp-json/wc/v3/orders';

    const RETRIEVE_AN_ORDER_END_POINT = '/wp-json/wc/v3/orders/{order_id}';

    const LIST_ALL_ORDERS_END_POINT = '/wp-json/wc/v3/orders';

    const LIST_ALL_CUSTOMERS_END_POINT = '/wp-json/wc/v3/customers';

    const RETRIEVE_AN_CUSTOMER_END_POINT = '/wp-json/wc/v3/customers/{customer_id}';

    const UPDATE_AN_ORDER_END_POINT = '/wp-json/wc/v3/orders/{order_id}';

    const DELETE_AN_ORDER_END_POINT = '/wp-json/wc/v3/orders/{order_id}';

    const BATCH_UPDATE_ORDERS_END_POINT = '/wp-json/wc/v3/orders/batch';

    
    const GET_META_ATTRIBUTES = '/wp-json/wc/v3/ced/get-meta_attr';

    const UPDATE_CEDCOMMERCE_ORDER_DATA = '/wp-json/wc/v3/ced/update_order';

    const GET_PRODUCTS_CATEGORIES = '/wp-json/wc/v3/products/categories';

    const GET_PRODUCTS_ATTRIBUTES = '/wp-json/wc/v3/products/attributes';

    const GET_CURRENT_CURRENCY = '/wp-json/wc/v3/data/currencies/current';

    const GET_ATTRIBUTE_VALUES = '/wp-json/wc/v3/products/attributes/{attribute_id}/terms';

    const GET_PRODUCT_SETTINGS = '/wp-json/wc/v3/settings/products';

    const GET_GENERAL_SETTINGS = '/wp-json/wc/v3/settings/general';

    const GET_ORDER_STATUS  = '/wp-json/wc/v3/ced/get_order_status';
  
    const UPDATE_ORDER_NOTE = '/wp-json/wc/v3/orders/{order_id}/notes';

    const GET_TAX_CLASSES = '/wp-json/wc/v3/taxes/classes';

    const GET_ORDER_STATUSES = '/wp-json/cedcommerce/v1/get_order_statuses';
    
    const GET_TAXES = '/wp-json/wc/v3/taxes';

    const GET_A_TAXES = '/wp-json/wc/v3/taxes/{tax_id}';

    const UPDATE_SHIPPING_CARRIER = '/wp-json/wc/v3/ced/update_shipping_carrier';

    const CREATE_UPDATE_PRODUCTS_BATCH = '/wp-json/wc/v3/products/batch';

    const CREATE_UPDATE_VARIATION_PRODUCTS_BATCH = '/wp-json/wc/v3/products/{product_id}/variations/batch';

    /**
     * Current API execution status.
     *
     * @var boolean
     * @since 1.0.0
     */
    protected $success = true;

    /**
     * End Point's Suffix end
     */

    /**
     * Constructor.
     *
     * @param bool $private If this is a private or public app
     *
     * @return self
     */
    public function _construct() {
        parent::_construct();
    }

    /**
     * Runs a request to the Shopify API.
     *
     * @param string     $type   The type of request... GET, POST, PUT, DELETE
     * @param string     $path   The woocommerce API path... /admin/xxxx/xxxx.json
     * @param array|null $params Optional parameters to send with the request
     *
     * @return array An array of the Guzzle response, and JSON-decoded body
     */
    public function rest( string $type, string $path, array $params = [], $headers = [], $getResponseHeaders = false ) {
        try {
            if ( is_array( $headers ) ) {
                if ( empty( $headers ) || ! isset( $headers['Authorization'] ) ) {
                    $headers['Authorization'] = self::AUTHORIZATION_PREFIX . $this->getBasicAuthText();
                }
            }

            return $this->call( $path, $headers, $params, $type, $getResponseHeaders );
        } catch ( Exception $e ) {
            return $e->getMessage();
        }
    }

    /**
     * Will call the desired endpoint using the right request method.
     *
     * @param string  $url current endpoint.
     * @param array   $headers header params.
     * @param array   $data body params.
     * @param string  $type current method to request at.
     * @param boolean $getResponseHeaders either to get request headers in the response.
     * @since 1.0.0
     * @return array
     */
    public function call( $url, $headers = [], $data = [], $type = 'GET', $getResponseHeaders = false ) {
        $url        = $this->getDomain() . $url;
        $shop       = $this->getCurrentShopData();
        $isSecurApi = $shop['is_secure_api'] ?? false;
        if ( '0' === $isSecurApi ) {
            if ( isset( $shop[ self::CONSUMER_KEY_FIELD_NAME ] ) && isset( $shop[ self::CONSUMER_SECRET_FIELD_NAME ] ) ) {
                $secretData = "consumer_key=" . $shop[ self::CONSUMER_KEY_FIELD_NAME ] . "&consumer_secret=" . $shop[ self::CONSUMER_SECRET_FIELD_NAME ];
                $url        = parse_url( $url, PHP_URL_QUERY ) ? $url . '&' . $secretData : $url . '?' . $secretData;
                if ( 'GET' === $type ) {
                    $url .= '&' . http_build_query( $data );
                }
            }
        }
        $client = new Client(
            [
                "verify" => false
            ]
        );
        $headers['Content-Type'] = 'application/json';
        $headers['Accept']       = 'application/json';
        switch ( $type ) {
            case 'DELETE':
                $response = $client->delete(
                    $url,
                    [
                        'headers'     => $headers,
                        'query'       => $data,
                        'http_errors' => false
                    ]
                );
                $receivedData = $response->getBody()->getContents();
                break;
            case 'DELETE/FORM':
                $response = $client->delete(
                    $url,
                    [
                        'headers'     => $headers,
                        'form_params' => $data,
                        'http_errors' => false
                    ]
                );
                $receivedData = $response->getBody()->getContents();
                break;

            case 'POST':
                $response = $client->post(
                    $url,
                    [
                        'headers'     => $headers,
                        'body'        => json_encode($data),
                        'http_errors' => false
                    ]
                );
                $receivedData = $response->getBody()->getContents();
                break;

            case 'PUT':
                $response = $client->put(
                    $url,
                    [
                        'headers' => $headers,
                        'json' => $data,
                        'http_errors' => false
                    ]
                );
                $receivedData = $response->getBody()->getContents();
                break;

            default:
                $requestParams = [
                    'headers'     => $headers,
                    'query'       => $data,
                    'http_errors' => false
                ];
                if ( '0' === $isSecurApi ) {
                    unset( $requestParams['query'] );
                }
                $response = $client->get(
                     $url,
                    $requestParams
                );

                $receivedData = $response->getBody()->getContents();
                break;
        }
        $statusCode    = $response->getStatusCode();
        $this->success = in_array($statusCode, array( 200, 201, 300, 301 ));
        if ( $getResponseHeaders ) {
            return $this->success
            ? [
                'data'    => @json_decode($receivedData, true),
                'headers' => $response->getHeaders()
            ]
            : $receivedData;
        }
        return @json_decode($receivedData, true);
    }

    /**
     * Will get current shop data.
     *
     * @since 1.0.0
     * @return array
     */
    public function getCurrentShopData()
    {
        $shop      = $this->di->getRegistry()->getCurrentShop();
        $appConfig = $this->di->getRegistry()->getAppConfig();
        if (empty($shop['_id']) || empty($appConfig['_id'])) {
            return [];
        }
        $shopData = $this->di->getObjectManager()->get('App\Apiconnect\Models\Apps\Shop')
                ->getAppInShopById($shop['_id'], $appConfig['_id']);
        if (! isset($shopData['token_data'])) {
            $shopData['token_data'] = $shopData;
        }
        return $shopData;
    }

    /**
     * Will get domain by removing http and www from URL.
     *
     * @param string $woocommerceDomain current domain.
     * @since 1.0.0
     * @return string
     */
    protected function formatDomain(&$woocommerceDomain)
    {
        if (empty($woocommerceDomain)) {
            return $woocommerceDomain;
        }
        $woocommerceDomain = preg_replace('#^(https?://)#i', '', $woocommerceDomain);
        $woocommerceDomain = preg_replace('#^www\.#i', '', $woocommerceDomain);
        return $woocommerceDomain;
    }

    /**
     * @return string
     */
    private function getDomain() {
        $shop = $this->getCurrentShopData();
        return $shop['token_data'][ self::WOOCOMMERCE_DOMAIN_FIELD_NAME ] ?? '';
    }

    /**
     * @return string
     */
    private function getBasicAuthText() {
        $shop = $this->getCurrentShopData();
        if ( isset($shop[self::CONSUMER_KEY_FIELD_NAME]) && isset($shop[self::CONSUMER_SECRET_FIELD_NAME])) {
            return $this->getBase64EncodedText(
                $shop[self::CONSUMER_KEY_FIELD_NAME] . ':' . $shop[self::CONSUMER_SECRET_FIELD_NAME]
            );
        }
        return '';
    }

    /**
     * @param $string
     * @return string
     */
    private function getBase64EncodedText($string) {
        return base64_encode( $string );
    }

    /**
     * @return mixed
     */
    public function getCurrentShop() {
        return $this->di->getRegistry()->getCurrentShop();
    }

    /**
     * Get app credentials.
     *
     * @since 1.0.0
     * @return array
     */
    public function getAppCredentials(){
        $requestData = $this->di->getRequest()->get();
        if ( ! empty( $requestData['sandbox'] ) ) {
            $mode = 'sandbox';
        } else {
            $mode = 'live';
        }
        $configData = $this->di->getRegistry()->getAppConfig();

        return [
            'redirect_uri' => ! empty( $configData[ $mode ]['redirect_uri'] ) ? $configData[ $mode ]['redirect_uri'] : false
        ];
    }

    /**
     * Get Particular app from apps_shop using shopID and appCode
     */
    public function getAppShopUsingAppCode($shopId, $appCode)
    {
        $appShopCollection = $this->di->getObjectManager()->create('\App\Core\Models\BaseMongo')->getCollectionForTable('apps_shop');
        
        $query = [
            [
                '$match' => [
                    '_id' => $shopId   //Stage 1 - Get app using remote shop Id
                ]
            ],
            [
                '$unwind' => '$apps'  //Stage 2 - Unwind apps column from the document
            ],
            [
                '$match' => [
                    'apps.app_code' => $appCode  //Stage 3 - Match the apps column appCode with config appCode
                ]
            ]
        ];
        $response = $appShopCollection->aggregate($query)->toArray();
        return $response;
    }

}
