<?php

/**
 * CedCommerce
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the End User License Agreement (EULA)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://cedcommerce.com/license-agreement.txt
 *
 * @category    Ced
 * @package     Ced_Woocommercewebapi
 * @author      CedCommerce Core Team <connect@cedcommerce.com>
 * @copyright   Copyright CEDCOMMERCE (http://cedcommerce.com/)
 * @license     http://cedcommerce.com/license-agreement.txt
 */

namespace App\Woocommercewebapi\Api;

use App\Woocommercewebapi\Api\Base;
use App\Woocommercewebapi\Components\Webhook As ComponentWebhook;
use App\Shopifywebapi\Components\Webhook\Sqs;
use Aws\DynamoDb\DynamoDbClient;

/**
 * Class Webhook
 * @package App\Woocommercewebapi\Api
 */
class Webhook extends Base
{
    const APP_SECRET = 'cedcommerce';

    /**
     * @var string
     */
    protected $_component = 'Webhook';

    /**
     * @param array $data
     * @return array
     */
    public function getAllWebhooks($data = []) {
        try{
            if($shop = $this->di->getRegistry()->getCurrentShop()) {
                if (!$this->isAuthenticated()) {
                    return [
                        'success' => false,
                        'message' => ComponentWebhook::AUTHENTICATION_ERROR_MESSAGE
                    ];
                }
                return $this->di->getObjectManager()->get(ComponentWebhook::class)->getAllWebhooks($data);
            } else {
                return [
                    'success' => false,
                    'message' => ComponentWebhook::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }
        } catch (\Exception $exception) {
            return [
                'success' => false,
                'message' => $exception->getMessage()
            ];
        }
    }

    /**
     * @param array $data
     * @return array
     */
    public function getWebhookByTopic($data = []) {
        try{
            if($shop = $this->di->getRegistry()->getCurrentShop()) {
                if (!$this->isAuthenticated()) {
                    return [
                        'success' => false,
                        'message' => ComponentWebhook::AUTHENTICATION_ERROR_MESSAGE
                    ];
                }
                return $this->di->getObjectManager()->get(ComponentWebhook::class)->getWebhookByTopic($data);
            } else {
                return [
                    'success' => false,
                    'message' => ComponentWebhook::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }
        } catch (\Exception $exception) {
            return [
                'success' => false,
                'message' => $exception->getMessage()
            ];
        }
    }

    /**
     * @param array $data
     * @return array
     */
    public function createWebhook($data = []) {
        try{
            if($shop = $this->di->getRegistry()->getCurrentShop()) {
                if (!$this->isAuthenticated()) {
                    return [
                        'success' => false,
                        'message' => ComponentWebhook::AUTHENTICATION_ERROR_MESSAGE
                    ];
                }
                return $this->di->getObjectManager()->get(ComponentWebhook::class)->createWebhook($data);
            } else {
                return [
                    'success' => false,
                    'message' => ComponentWebhook::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }
        } catch (\Exception $exception) {
            return [
                'success' => false,
                'message' => $exception->getMessage()
            ];
        }
    }

    /**
     * @param array $data
     * @return array
     */
    public function updateWebhook($data = []) {
        try{
            if($shop = $this->di->getRegistry()->getCurrentShop()) {
                if (!$this->isAuthenticated()) {
                    return [
                        'success' => false,
                        'message' => ComponentWebhook::AUTHENTICATION_ERROR_MESSAGE
                    ];
                }
                return $this->di->getObjectManager()->get(ComponentWebhook::class)->updateWebhook($data);
            } else {
                return [
                    'success' => false,
                    'message' => ComponentWebhook::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }
        } catch (\Exception $exception) {
            return [
                'success' => false,
                'message' => $exception->getMessage()
            ];
        }
    }

    /**
     * @param array $data
     * @return array
     */
    public function deleteWebhook($data = []) {
        try{
            if($shop = $this->di->getRegistry()->getCurrentShop()) {
                if (!$this->isAuthenticated()) {
                    return [
                        'success' => false,
                        'message' => ComponentWebhook::AUTHENTICATION_ERROR_MESSAGE
                    ];
                }
                return $this->di->getObjectManager()->get(ComponentWebhook::class)->deleteWebhook($data);
            } else {
                return [
                    'success' => false,
                    'message' => ComponentWebhook::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }
        } catch (\Exception $exception) {
            return [
                'success' => false,
                'message' => $exception->getMessage()
            ];
        }
    }

    /**
     * @param array $data
     * @return array
     */
    public function batchUpdateWebhooks($data = []) {
        try{
            if($shop = $this->di->getRegistry()->getCurrentShop()) {
                if (!$this->isAuthenticated()) {
                    return [
                        'success' => false,
                        'message' => ComponentWebhook::AUTHENTICATION_ERROR_MESSAGE
                    ];
                }
                return $this->di->getObjectManager()->get(ComponentWebhook::class)->batchUpdateWebhooks($data);
            } else {
                return [
                    'success' => false,
                    'message' => ComponentWebhook::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }
        } catch (\Exception $exception) {
            return [
                'success' => false,
                'message' => $exception->getMessage()
            ];
        }
    }

    /**
     * Will call register webhook method.
     *
     * @param array $data array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function register( $data = [] ) {
        try{
            if( ! $this->di->getRegistry()->getCurrentShop() ) {
                return [
                    'success' => false,
                    'message' => ComponentWebhook::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }

            return $this->di->getObjectManager()->get( ComponentWebhook::class )->createWebhook( $this->getFilteredParams( $data ) );
        } catch ( \Exception $exception ) {
            return [
                'success' => false,
                'message' => $exception->getMessage()
            ];
        }
    }

    /**
     * @param $data
     * @return array
     */
    public function unregister($data)
    {
        try{
            if($shop = $this->di->getRegistry()->getCurrentShop()) {
                if (!$this->isAuthenticated()) {
                    return [
                        'success' => false,
                        'message' => ComponentWebhook::AUTHENTICATION_ERROR_MESSAGE
                    ];
                }
                if(isset($data['topic'])) {
                    $status = $this->deleteSigleConfig($data);
                    if($status['success']) {
                        $webhookData = [
                            'topic'        => $data['topic']
                        ];
                        return $this->deleteWebhook($webhookData);
                    } else {
                        return $status;
                    }
                } else {
                    $status = $this->deleteAllConfig($data);

                    if($status['success']) {

                        if (isset($shop[ComponentWebhook::SHOP_WEBHOOKS_FIELD]) && count($shop[ComponentWebhook::SHOP_WEBHOOKS_FIELD])) {
                            $batchDeleteWebhooks['delete'] = [];
                            foreach ($shop[ComponentWebhook::SHOP_WEBHOOKS_FIELD] as $topic => $id) {
                                $batchDeleteWebhooks['delete'][] = ['topic' => $topic ];
                            }
                            return $this->batchUpdateWebhooks($batchDeleteWebhooks);
                        } else {
                            return $status;
                        }
                    } else {
                        return $status;
                    }
                }
            } else {
                return [
                    'success' => false,
                    'message' => ComponentWebhook::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }
        } catch (\Exception $exception) {
            return [
                'success' => false,
                'message' => $exception->getMessage()
            ];
        }
    }

    /**
     * @param $id
     * @param $data
     * @param $baseWebhookUrl
     * @return string
     */
    public function getWebhookAddress($id, $data, $baseWebhookUrl)
    {
        $queryParams = [
            'id'    => $id,
            'topic' => $data['topic']
        ];

        return $baseWebhookUrl.'?'. http_build_query($queryParams);
    }

    /**
     * @param $data
     * @return array
     */
    public function deleteSigleConfig($data)
    {
        try {
            $client = DynamoDbClient::factory(include BP.'/app/etc/dynamo.php');
            $shop = $this->di->getRegistry()->getCurrentShop();
            $appConfig = $this->di->getRegistry()->getAppConfig();
            $result = $client->scan([
                'FilterExpression' => "shop_id=:shop_id AND app_id=:app_id AND topic=:topic",
                "ExpressionAttributeValues" => [
                    ":shop_id" => [
                        'N' => (string)$shop['_id']
                    ],
                    ":app_id" => [
                        'S' => (string)$appConfig['_id']
                    ],
                    ":topic" => [
                        'S' => (string)$data['topic']
                    ]
                ],
                'TableName' => 'webhook_config', // REQUIRED
            ]);

            if($result->get('Count') > 0) {
                foreach($result->get('Items') as $item) {
                    $result = $client->deleteItem([
                        'Key' => [
                            'id' => [
                                'S' => (string)$item['id']['S'],
                            ]
                        ],
                        'TableName' => 'webhook_config', // REQUIRED
                    ]);
                }

                return [
                    'success' => true,
                    'message' => 'Unegistered Successfully',
                ];

            } else {
                return [
                    'success' => true,
                    'message' => 'Already Unregistered',
                    'code' => 'already_unregistered'
                ];
            }
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * @return array
     */
    public function deleteAllConfig()
    {
        try {
            $client = DynamoDbClient::factory(include BP.'/app/etc/dynamo.php');
            $shop = $this->di->getRegistry()->getCurrentShop();
            $appConfig = $this->di->getRegistry()->getAppConfig();
            $result = $client->scan([
                'FilterExpression' => "shop_id=:shop_id AND app_id=:app_id",
                "ExpressionAttributeValues" => [
                    ":shop_id" => [
                        'N' => (string)$shop['_id']
                    ],
                    ":app_id" => [
                        'S' => (string)$appConfig['_id']
                    ]
                ],
                'TableName' => 'webhook_config', // REQUIRED
            ]);
            if($result->get('Count') > 0) {
                foreach($result->get('Items') as $item) {
                    $result = $client->deleteItem([
                        'Key' => [
                            'id' => [
                                'S' => (string)$item['id']['S'],
                            ]
                        ],
                        'TableName' => 'webhook_config', // REQUIRED
                    ]);
                }

                return [
                    'success' => true,
                    'message' => 'Unegistered Successfully',
                ];

            } else {
                return [
                    'success' => true,
                    'message' => 'Already Unregistered',
                    'code' => 'already_unregistered'
                ];
            }
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * @param $data
     * @return array
     */
    public function validatePostData($data)
    {
        $required_indexes = ['queue_unique_id', 'queue_name', 'topic', 'type'];

        foreach ($required_indexes as $index) {
            if(strpos($index, '->') === false) {
                if(!isset($data[$index])) {
                    return [
                        'status' => false,
                        'error'  => "'{$index}' is required."
                    ];
                }
            }
            else {
                $expl = explode('->', $index);
                if(!isset($data[$expl[0]][$expl[1]])) {
                    return [
                        'status' => false,
                        'error'  => "'{$expl[0]}[{$expl[1]}]' is required."
                    ];
                }
            }
        }

        if(isset($data['queue_config']))
        {
            $required_indexes = ['queue_config->key', 'queue_config->secret', 'queue_config->region'];

            foreach ($required_indexes as $index) {
                if(strpos($index, '->') === false) {
                    if(!isset($data[$index])) {
                        return [
                            'status' => false,
                            'error'  => "'{$index}' is required."
                        ];
                    }
                }
                else {
                    $expl = explode('->', $index);
                    if(!isset($data[$expl[0]][$expl[1]])) {
                        return [
                            'status' => false,
                            'error'  => "'{$expl[0]}[{$expl[1]}]' is required."
                        ];
                    }
                }
            }
        }
        return ['status' => true];
    }

    /**
     * @param $data
     * @return array
     */
    public function saveConfig($data)
    {
        try {
            $shop = $this->di->getRegistry()->getCurrentShop();
            $appConfig = $this->di->getRegistry()->getAppConfig();
            $appCode = $this->getDi()->getConfig()->get('app_code');

            $client = DynamoDbClient::factory(include BP.'/app/etc/dynamo.php');
            $result = $client->putItem([
                'Item' => [ // REQUIRED
                    'id' => [
                        'S' => $appCode."_".(string)$appConfig['_id'],
                    ],
                    'app_id'=>[
                        'S'=>'cedcommerce_custom'
                    ],
                    'app_secret' => [
                        'S' => (string)self::APP_SECRET,
                    ],
                    'marketplace' => [
                        'S' => $appConfig['marketplace']
                    ]
                ],
                'TableName' => 'apps', // REQUIRED
            ]);
            if(isset($data['queue_config']))
            {
                $result = $client->putItem([
                    'Item' => [ // REQUIRED
                        'id' => [
                            'S' => (string)$data['queue_unique_id'],
                        ],
                        'queue_type' => [
                            'S' => (string)$data['type'],
                        ],
                        'app_secret' => [
                            'S' => (string)$data['queue_config']['secret'],
                        ],
                        'app_key' => [
                            'S' => (string)$data['queue_config']['key'],
                        ],
                        'queue_region' => [
                            'S' => (string)$data['queue_config']['region']
                        ],
                        'base_url' => [
                            'S' => (string)$this->getQueueBaseUrl($data['queue_url'])
                        ]
                    ],
                    'TableName' => 'queue_config', // REQUIRED
                ]);
            } else {
                $result = $client->getItem([
                    'Key' => [ // REQUIRED
                        'id' => [
                            'S' => (string)$data['queue_unique_id'],
                        ],

                    ],
                    'TableName' => 'queue_config', // REQUIRED
                ]);
                if(is_null($result->get('Item'))){
                    return [
                        'success' => false,
                        'message' => 'Queue does not exist. Pass existing queue id or pass queue id with queue_config',
                        'code' => 'queue_does_not_exist'
                    ];
                }
            }

            $result = $client->scan([
                'FilterExpression' => "shop_id=:shop_id AND app_id=:app_id AND topic=:topic",
                "ExpressionAttributeValues" => [
                    ":shop_id" => [
                        'N' => (string)$shop['_id']
                    ],
                    ":app_id" => [
                        'S' => (string)$appConfig['_id']
                    ],
                    ":topic" => [
                        'S' => (string)$data['topic']
                    ]
                ],
                'TableName' => 'webhook_config', // REQUIRED
            ]);
            $queueData = $data['queue_data']??[];
            if($result->get('Count') == 0) {
                $mongo = $this->di->getObjectManager()->create('\App\Core\Models\BaseMongo');
                $autoIncrementId = $mongo->getCounter('sqs_config_id');

                $result = $client->putItem([
                    'Item' => [
                        'id' => [
                            'S' => $appCode."_".(string)$autoIncrementId,
                        ],
                        'shop_id' => [
                            'N' => (string)$shop['_id'],
                        ],
                        'app_id' => [
                            'S' => $appCode."_".(string)$appConfig['_id'],
                        ],
                        'queue_reference_id' => [
                            'S' => (string)$data['queue_unique_id'],
                        ],
                        'topic' => [
                            'S' => (string)$data['topic']
                        ],
                        'queue_name' => [
                            'S' => (string)$data['queue_name']
                        ],
                        'queue_data' => [
                            'S' => (string)json_encode($queueData)
                        ]
                    ],
                    'TableName' => 'webhook_config', // REQUIRED
                ]);

                return [
                    'success' => true,
                    'message' => 'Registered Successfully',
                    'id'      => $appCode."_".$autoIncrementId
                ];

            } else {
                $items = $result->get('Items');

                $result = $client->putItem([
                    'Item' => [
                        'id' => [
                            'S' => (string)$items[0]['id']['S'],
                        ],
                        'shop_id' => [
                            'N' => (string)$shop['_id'],
                        ],
                        'app_id' => [
                            'S' => (string)$appConfig['_id'],
                        ],
                        'queue_reference_id' => [
                            'S' => (string)$data['queue_unique_id'],
                        ],
                        'topic' => [
                            'S' => (string)$data['topic']
                        ],
                        'queue_name' => [
                            'S' => (string)$data['queue_name']
                        ],
                        'queue_data' => [
                            'S' => (string)json_encode($queueData)
                        ]
                    ],
                    'TableName' => 'webhook_config', // REQUIRED
                ]);
                return [
                    'success' => true,
                    'message' => 'Already Registered',
                    'code' => 'already_registered',
                    'id'    => $items[0]['id']['S'],
                    'queue_data' => $queueData
                ];
            }
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage(),
                'shop' => $shop
            ];
        }
    }

    /**
     * @param $queue_url
     */
    public function getQueueBaseUrl($queue_url)
    {
        $explode = explode('/', $queue_url);
        unset($explode[count($explode)-1]);
        return implode('/', $explode);
    }
}