<?php

/**
 * CedCommerce
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the End User License Agreement (EULA)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://cedcommerce.com/license-agreement.txt
 *
 * @category    Ced
 * @package     Ced_Woocommercewebapi
 * @author      CedCommerce Core Team <connect@cedcommerce.com>
 * @copyright   Copyright CEDCOMMERCE (http://cedcommerce.com/)
 * @license     http://cedcommerce.com/license-agreement.txt
 */

namespace App\Woocommercewebapi\Api;

use App\Woocommercewebapi\Api\Base;
use App\Woocommercewebapi\Components\Product As ComponentProduct;

/**
 * Will handle product requests.
 *
 * @since 1.0.0
 */
class Product extends Base
{
    /**
     * @var string
     */
    protected $_component = 'Product';

    /**
     * Get all products from woocommerce.
     *
     * @param array $data array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function getAllProducts( $data = [] ) {
        try {
            if ( ! $this->di->getRegistry()->getCurrentShop() ) {
                return [
                    'success' => false,
                    'message' => ComponentProduct::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }

            return $this->di->getObjectManager()->get( ComponentProduct::class )->getAllProducts( $this->getFilteredParams( $data ) );
        } catch ( \Exception $e ) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Get product variations.
     *
     * @param array $data array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function getProductVariations($data = []) {
        try {
            if ( ! $this->di->getRegistry()->getCurrentShop() ) {
                return [
                    'success' => false,
                    'message' => ComponentProduct::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }

            return $this->di->getObjectManager()->get(ComponentProduct::class)->getProductVariations( $this->getFilteredParams( $data) );
        } catch ( \Exception $exception ) {
            return [
                'success' => false,
                'message' => $exception->getMessage()
            ];
        }
    }

    /**
     * Will get single product data.
     *
     * @param array $data array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function getProductById($data = []) {
        try{
            if ( ! $this->di->getRegistry()->getCurrentShop() ) {
                return [
                    'success' => false,
                    'message' => ComponentProduct::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }
            return $this->di->getObjectManager()->get(ComponentProduct::class)->getProductById($data);
        } catch (\Exception $exception) {
            return [
                'success' => false,
                'message' => $exception->getMessage()
            ];
        }
    }


    /**
     * @param array $data
     * @return array
     */
    public function getMetaAttributes($data = []) {
        try{
            if(! $this->di->getRegistry()->getCurrentShop()) {
                return [
                    'success' => false,
                    'message' => ComponentProduct::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
                // if (!$this->isAuthenticated()) {
                //     return [
                //         'success' => false,
                //         'message' => ComponentProduct::AUTHENTICATION_ERROR_MESSAGE
                //     ];
                // }
            } else {
                return $this->di->getObjectManager()->get(ComponentProduct::class)->getMetaAttributes($data);
                
            }
        } catch (\Exception $exception) {
            return [
                'success' => false,
                'message' => $exception->getMessage()
            ];
        }
    }

    /**
     * Will get product categories.
     *
     * @param array $data array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function getProductsCategories( $data = [] ) {
        try{
            if( ! $this->di->getRegistry()->getCurrentShop() ) {
                return [
                    'success' => false,
                    'message' => ComponentProduct::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }

            return $this->di->getObjectManager()->get(ComponentProduct::class)->getProductsCategories( $this->getFilteredParams( $data ) );
        } catch (\Exception $exception) {
            return [
                'success' => false,
                'message' => $exception->getMessage()
            ];
        }
    }

    /**
     * Will get product attributes.
     *
     * @param array $data array containing the request params
     * @since 1.0.0
     * @return array
     */
    public function getProductsAttributes( $data = [] ) {
        try {

            if ( ! $this->di->getRegistry()->getCurrentShop() ) {
                return [
                    'success' => false,
                    'message' => ComponentProduct::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }

            return $this->di->getObjectManager()->get(ComponentProduct::class)->getProductsAttributes( $this->getFilteredParams( $data ) );

        } catch ( \Exception $exception ) {
            return [
                'success' => false,
                'message' => $exception->getMessage()
            ];
        }
    }

     /**
     * Will get product attributes values.
     *
     * @param array $data array containing the request params
     * @since 1.0.0
     * @return array
     */
    public function getAttributeValues( $data = [] ) {
        try {

            if ( ! $this->di->getRegistry()->getCurrentShop() ) {
                return [
                    'success' => false,
                    'message' => ComponentProduct::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }

            return $this->di->getObjectManager()->get(ComponentProduct::class)->getAttributeValues( $this->getFilteredParams( $data ) );

        } catch ( \Exception $exception ) {
            return [
                'success' => false,
                'message' => $exception->getMessage()
            ];
        }
    }

    public function updateProduct( $data = [] ) {
        try {
            if ( ! $this->di->getRegistry()->getCurrentShop() ) {
                return [
                    'success' => false,
                    'message' => ComponentProduct::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }

            return $this->di->getObjectManager()->get( ComponentProduct::class )->updateProduct( $this->getFilteredParams( $data ) );
        } catch ( \Exception $e ) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    public function updateProductVariation( $data = [] ) {
        try {
            if ( ! $this->di->getRegistry()->getCurrentShop() ) {
                return [
                    'success' => false,
                    'message' => ComponentProduct::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }

            return $this->di->getObjectManager()->get( ComponentProduct::class )->updateProductVariation( $this->getFilteredParams( $data ) );
        } catch ( \Exception $e ) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Will get product setting.
     *
     * @param array $params array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function getProductSettings( $params = array() ) {
        try {
            if ( ! $this->di->getRegistry()->getCurrentShop() ) {
                return [
                    'success' => false,
                    'message' => ComponentProduct::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }

            return $this->di->getObjectManager()->get( ComponentProduct::class )->getProductSettings( $this->getFilteredParams( $params ) );
        } catch ( \Exception $e ) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Will get general setting.
     *
     * @param array $params array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function getGeneralSettings( $params = array() ) {
        try {
            if ( ! $this->di->getRegistry()->getCurrentShop() ) {
                return [
                    'success' => false,
                    'message' => ComponentProduct::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }

            return $this->di->getObjectManager()->get( ComponentProduct::class )->getGeneralSettings( $this->getFilteredParams( $params ) );
        } catch ( \Exception $e ) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Will upload/update products.
     *
     * @param array $params array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function upsertProducts($params = array()) {
        try {
            if ( ! $this->di->getRegistry()->getCurrentShop() ) {
                return [
                    'success' => false,
                    'message' => ComponentProduct::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }
            return $this->di->getObjectManager()->get(ComponentProduct::class)
                ->upsertProducts($this->getFilteredParams($params));
        } catch ( \Exception $e ) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Will upload/update variation products.
     *
     * @param array $params array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function upsertVariation($params = array()) {
        try {
            if ( ! $this->di->getRegistry()->getCurrentShop() ) {
                return [
                    'success' => false,
                    'message' => ComponentProduct::SHOP_NOT_FOUND_ERROR_MESSAGE
                ];
            }
            return $this->di->getObjectManager()->get(ComponentProduct::class)
                ->upsertVariation($this->getFilteredParams($params));
        } catch ( \Exception $e ) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }
}