<?php

namespace App\Webapi\Controllers;

use Phalcon\DI;
use Phalcon\Mvc\Application;
use App\Webapi\Components\Destination\Apiendpoint;

class RestController extends \App\Core\Controllers\BaseController
{

    public function v1Action()
    {
        $cache = $this->di->getCache();
        $startTime = date('m-d-Y h:i:s', time());
        $start_time = microtime(true);
        $this->di->getObjectManager()->get('\App\Webapi\Components\Helper')
            ->loadAreaConfigurations();

        $apiRoute = implode('/', $this->router->getParams());
        $httpMethod = $this->request->getMethod();

        $route = $this->di->getConfig()->get('restapi')->get('v1')
            ->get($httpMethod)->get('routes')->get($apiRoute);

        if ($route) {
            $method = $route->method;

            if (!$this->checkAcl($route)) {
                return $this->prepareResponse(['success' => false, 'code' => '403', 'msg' => 'Access forbidden.']);
            }

            #Stop process api requests if subUser is blocked.
            if (!$this->isSubUserActive()) {
                return $this->prepareResponse([
                    'success' => false,
                    'msg' => 'Cannot Process your request. Subuser is blocked!'
                ]);
            }

            if ($this->di->getRegistry()->getCurrentShop()) {
                $shop = $this->di->getRegistry()->getCurrentShop();
                $appConfig = $this->di->getRegistry()->getAppConfig();
                $app = $this->di->getObjectManager()->get('App\Apiconnect\Models\Apps\Shop')
                    ->getAppInShopById($shop['_id'], $appConfig['_id']);

                if ($this->di->getCache()->get('sub_app_status_' . $app['app_id'] . "_" . $app['sub_app_id'])) {
                    $sub_app_status = $this->di->getCache()
                        ->get('sub_app_status_' . $app['app_id'] . "_" . $app['sub_app_id']);
                    $sub_app_status = empty($sub_app_status) ? 'active' : $sub_app_status;
                } else {
                    $subApp = $this->di->getObjectManager()
                        ->get('App\Apiconnect\Models\Apps')->getSubApp($app['sub_app_id']);
                    $sub_app_status = empty($subApp['sub_app_status']) ? 'active' : $subApp['sub_app_status'];
                }

                #Stop process api call requests if sub-app is blocked.
                if ($sub_app_status !== 'active') {
                    return $this->prepareResponse([
                        'success' => false,
                        'msg' => 'Sorry for the inconvenience, Sub-app is inactive cannot process your request.'
                    ]);
                }

                #Stop process api call requests if apps-shop is blocked
                if (isset($app['active']) && !$app['active']) {
                    return $this->prepareResponse([
                        'success' => false,
                        'msg' => 'Sorry for the inconvenience, app is inactive cannot process your request.'
                    ]);
                }
            }

            if ($this->validateRequired($route)) {
                $requestData = $this->getRequestData();

                #request modifier start
                if ($route->get('request_modifier_class') && class_exists($route->request_modifier_class)) {
                    $requestData = $this->di->getObjectManager()
                                ->get($route->request_modifier_class)
                        ->$method($requestData);
                } elseif ($route->get('component')) {

                    $instance = new class extends \App\Apiconnect\Api\Request\Base {
                        protected $_component;
                    };
                    $this->di->setShared(
                        "apiComponentRequest",
                        function () use ($instance) {
                            return $instance;
                        }
                    );
                    $instance = $this->di->getApiComponentRequest();

                    $requestData = $instance->setComponent($route->component)
                        ->$method($requestData);
                }

                if ((isset($requestData['success']) && $requestData['success'] === false)) {
                    return $this->prepareResponse($requestData);
                }
                /** request modifier end **/
                if ($route->get('class') && class_exists($route->class)) {
                    $apiResponse = $this->di->getObjectManager()->get($route->class)
                        ->$method($requestData);
                } else {
                    $instance = new class extends \App\Apiconnect\Api\Base {
                        protected $_component;
                    };
                    $this->di->setShared(
                        "anonymousApiComponent",
                        function () use ($instance) {
                            return $instance;
                        }
                    );
                    $instance = $this->di->getAnonymousApiComponent();
                    $apiResponse = $instance->setComponent($route->component)
                        ->$method($requestData);
                }

                if (
                    !isset(
                    $this->di->getRegistry()->getHeaders()['Response-Modifier']
                ) || $this->di->getRegistry()->getHeaders()['Response-Modifier'] == 1
                ) {
                    if ($route->get('response_class') && class_exists($route->response_class)) {
                        $apiResponse = $this->di->getObjectManager()
                                    ->get($route->response_class)
                            ->$method($apiResponse);
                    } elseif ($route->get('component')) {
                        $instance = new class extends \App\Apiconnect\Api\Response\Base {
                            protected $_component;
                        };
                        $this->di->setShared(
                            "apiComponentResponse",
                            function () use ($instance) {
                                return $instance;
                            }
                        );
                        $instance = $this->di->getApiComponentResponse();
                        $apiResponse = $instance->setComponent($route->component)
                            ->$method($apiResponse);
                    }
                }

                if (
                    isset(
                    $apiResponse['code']
                ) && ($apiResponse['code'] == "unauthorized_access" || $apiResponse['code'] == "token_expired")
                ) {
                    $this->sendNotification($apiResponse['code']);
                }

                $apiResponse['start_time'] = $startTime;
                $apiResponse['end_time'] = date('m-d-Y h:i:s', time());
                $end_time = microtime(true);
                $execution_time = ($end_time - $start_time);
                $apiResponse['execution_time'] = $execution_time;

                #TODO: Handle case if response code starts from 500, then block api calls for that particular shop
                if (isset($apiResponse['code']) && $apiResponse['code'] >= 500) {
                    $response = $this->handleApiCallAttempts();
                    $response['code'] = $apiResponse['code'];
                    if (!$response['success']) {
                        return $this->prepareResponse($response);
                    }
                    return $this->prepareResponse($apiResponse);
                }
                // clear previous unsuccessful response attempts from cache
                if ($cache->has($shop['_id'] . "_" . $appConfig['app_code'] . "_temp_block_count", "attempts")) {
                    $cache->deleteMultiple(
                        [
                            $shop['_id'] . "_" . $appConfig['app_code'] . "_temp_block_count",
                            $shop['_id'] . "_" . $appConfig['app_code'] . "_temp_block_time"
                        ],
                        "attempts"
                    );
                }
                return $this->prepareResponse($apiResponse);
            } else {
                return $this->prepareResponse([
                    'success' => false,
                    'code' => '422',
                    'msg' => 'Missing required params.'
                ]);
            }
        } else {
            return $this->prepareResponse(['success' => false, 'code' => '404', 'msg' => 'Page not found']);
        }
    }

    public function checkAcl($route)
    {
        // if acl not already in cache, build it
        if (!$this->di->getCache()->has("acl", "setup")) {
            $this->di->getObjectManager()
                ->get('App\Core\Components\Setup')
                ->buildAclAction(false);
        }
        $acl = $this->di->getCache()->get("acl", "setup");
        $module = 'webapi';
        $controller = $route->resource;
        $action = strtolower($this->request->getMethod());

        if ($acl->isAllowed($this->di->getUser()->getRole()->getCode(), $module . '_' . $controller, $action)) {
            if ($this->di->getRegistry()->getIsSubUser()) {
                $subuser = $this->di->getSubUser();
                $childAcl = $this->di->getCache()->get('child_' . $subuser['_id'] . "_acl", "setup");
                if ($childAcl->isAllowed('child_' . $subuser['_id'], $module . '_' . $controller, $action)) {
                    return true;
                }
                return false;
            } else {
                return true;
            }
        } else {
            return false;
        }
    }

    public function validateRequired($route)
    {
        $route = $route->toArray();
        if (isset($route['required'])) {
            foreach ($route['required'] as $key) {
                if (!array_key_exists($key, $this->getRequestData())) {
                    return false;
                }
            }
        }
        return true;
    }

    public function getRequestData()
    {
        switch ($this->request->getMethod()) {
            case 'POST':
                $contentType = $this->request->getHeader('Content-Type');
                if (strpos($contentType, 'application/json') !== false) {
                    $data = $this->request->getJsonRawBody(true);
                } else {
                    $data = $this->request->getPost();
                }
                break;
            case 'PATCH':
            case 'PUT':
                $contentType = $this->request->getHeader('Content-Type');
                if (strpos($contentType, 'application/json') !== false) {
                    $data = $this->request->getJsonRawBody(true);
                } else {
                    $data = $this->request->getPut();
                }
                $data = array_merge($data, $this->request->get());
                break;
            case 'DELETE':
                $contentType = $this->request->getHeader('Content-Type');
                if (strpos($contentType, 'application/json') !== false) {
                    $data = $this->request->getJsonRawBody(true);
                } else {
                    $data = $this->request->getPost();
                }
                $data = array_merge($data, $this->request->get());
                break;
            default:
                $data = $this->request->get();
                break;
        }
        return $data;
    }

    public function sendNotification($eventCode)
    {
        $appConfig = $this->di->getRegistry()->getAppConfig();
        if (isset($appConfig['app_code'], $appConfig['marketplace'])) {
            $data = ['marketplace' => $appConfig['marketplace'], "type" => "app", "event_code" => $eventCode];
            $cifSubscriptionResponse = $this->di->getObjectManager()
                ->get("\App\Cedcommercewebapi\Api\EventSubscription")->getEventSubscription($data);
            if (isset($cifSubscriptionResponse['success']) && $cifSubscriptionResponse['success']) {
                $cifSubscription = null;
                foreach ($cifSubscriptionResponse['data'] as $subscription) {
                    if (
                        isset($subscription['queue_data']['app_code'])
                        && $subscription['queue_data']['app_code'] == $appConfig['app_code']
                    ) {
                        $cifSubscription = $subscription;
                        break;
                    }
                }
                $cifDestinationId = $cifSubscription['destination_id'] ?? null;
                $cifDestinationResponse = $this->di->getObjectManager()
                    ->get("\App\Cedcommercewebapi\Api\EventDestination")
                    ->getEventDestination(['event_handler_id' => $cifDestinationId]);
                if (isset($cifDestinationResponse['success']) && $cifDestinationResponse['success']) {
                    $cifDestination = $cifDestinationResponse['data'][0];
                    return $this->di->getObjectManager()
                        ->get("\App\Webapi\Components\Destination\\"
                            . ucfirst(str_replace("_", "", $cifDestination['event_handler'])))
                        ->sendNotificationToDestination($cifDestination, $cifSubscription);
                }
                return ['success' => false, 'message' => 'cifDestination not found.'];
            }
            return ['success' => false, 'message' => "cifSubscription not found."];
        }
        return ['success' => false, 'message' => "appConfig not found."];
    }

    public function handleApiCallAttempts()
    {
        $shop = $this->di->getRegistry()->getCurrentShop();
        $appConfig = $this->di->getRegistry()->getAppConfig();
        $apiCallLimit = $this->di->getConfig()->get("api_call_attempts");
        $cache = $this->di->getCache();
        $maxTempBlock = $apiCallLimit->get("max_temp_block_attempt") ?? 3;  // max_temp_block_attempt:3
        $maxPrmntBlock = $apiCallLimit->get("max_prmnt_block_attempt") ?? 6;  // max_temp_block_attempt:6
        $tempBlockDuration = $apiCallLimit->get("max_temp_block_duration") ?? 1800; // 30 mins
        $appCode = $appConfig['app_code'];
        $marketplace = $appConfig['app_code'];
        $shopId = $shop['_id'];

        if ($cache->has($shopId . "_" . $appCode . "_temp_block_count", "attempts")) {
            // Check if api calls are temporarily blocked
            if (
                $cache->get($shopId . "_" . $appCode . "_temp_block_count", "attempts") >
                $maxTempBlock && time() - $cache
                    ->get($shopId . "_" . $appCode . "_temp_block_time", "attempts") < $tempBlockDuration
            ) {
                // Check if user exceeded the $maxPrmntBlock limit
                if ($cache->get($shopId . "_" . $appCode . "_temp_block_count", "attempts") >= $maxPrmntBlock) {
                    // clear cache
                    if ($cache->has($shopId . "_" . $appCode . "_temp_block_count", "attempts")) {
                        $cache->deleteMultiple(
                            [
                                $shopId . "_" . $appCode . "_temp_block_count",
                                $shopId . "_" . $appCode . "_temp_block_time"
                            ],
                            "attempts"
                        );
                    }
                    // deactivate shop
                    $this->di->getObjectManager()->get('App\Apiconnect\Models\Apps\Shop')
                        ->changeAppShopStatus($shopId, $appCode, false);

                    $this->sendAppShopBlockedMail(['shop_id' => $shopId, 'app_code' => $appCode]);

                    $apiRoute = implode('/', $this->router->getParams());
                    $this->di->getLog()->logContent("API calls permanently blocked for Shop Id = " . $shopId . " , App Code = " . $appCode . " , due to this Api Call = " . $apiRoute, "info", $marketplace . DS . date("Y-m-d") . DS . "blocked_app_shop.log");

                    return [
                        'success' => false,
                        'message' => "Api calls permanently blocked for the current shop! Kindly contact to admin."
                    ];
                }

                $cache->set(
                    $shopId . "_" .
                    $appCode . "_temp_block_count",
                    $cache->get($shopId . "_" . $appCode . "_temp_block_count", "attempts") + 1,
                    "attempts"
                );
                return ['success' => false, 'message' => "Api calls temporarily blocked! Try after 30 mins."];

                // Allow api calls after time specified
            } elseif (
                $cache
                    ->get($shopId . "_" . $appCode . "_temp_block_count", "attempts") > $maxTempBlock && time() -
                $cache->get($shopId . "_" . $appCode . "_temp_block_time", "attempts") > $tempBlockDuration
            ) {
                if ($cache->has($shopId . "_" . $appCode . "_temp_block_count", "attempts")) {
                    $cache->deleteMultiple(
                        [
                            $shopId . "_" . $appCode . "_temp_block_count",
                            $shopId . "_" . $appCode . "_temp_block_time"
                        ],
                        "attempts"
                    );
                }
                $cache->set($shopId . "_" . $appCode . "_temp_block_count", 1, "attempts");
                return ['success' => true];
            }

            // Check if user exceeded the $maxTempBlock limit
            if ($cache->get($shopId . "_" . $appCode . "_temp_block_count", "attempts") >= $maxTempBlock) {
                $cache->set(
                    $shopId . "_" . $appCode . "_temp_block_count",
                    $cache->get($shopId . "_" . $appCode . "_temp_block_count", "attempts") + 1,
                    "attempts"
                );
                $cache->set($shopId . "_" . $appCode . "_temp_block_time", time(), "attempts");
                return ['success' => false, 'message' => "Api calls temporarily blocked! Try after 30 mins."];
            }

            if (
                $cache->get($shopId . "_" . $appCode . "_temp_block_count", "attempts") >= 1
                && $cache->get($shopId . "_" . $appCode . "_temp_block_count", "attempts") < $maxTempBlock
            ) {
                $cache->set(
                    $shopId . "_" . $appCode . "_temp_block_count",
                    $cache->get($shopId . "_" . $appCode . "_temp_block_count", "attempts") + 1,
                    "attempts"
                );
            }
            return ['success' => true];
        }

        $cache->set($shopId . "_" . $appCode . "_temp_block_count", 1, "attempts");
        return ['success' => true];
    }


    public function sendAppShopBlockedMail($data)
    {
        $criticalMail = $this->di->getConfig()->path('mailer.critical_mail_reciever', '');
        $criticalMailEndpoint = $this->di->getConfig()->path('mailer.critical_mail_endpoint', '');
        $bccs = $templateData = [];

        $apiRoute = implode('/', $this->router->getParams());
        $httpMethod = $this->request->getMethod();
        $route = $this->di->getConfig()->get('restapi')->get('v1')->get($httpMethod)->get('routes')->get($apiRoute);

        if (
            $this->di->getConfig()->get('mailer')
            && $this->di->getConfig()->get('mailer')->get('critical_mail_reciever_bcc')
        ) {
            $bccs = $this->di->getConfig()->get('mailer')->get('critical_mail_reciever_bcc');
        }

        if (isset($data['marketplace'])) {
            $templateData['marketplace'] = $data['marketplace'];
        }

        $templateData['templateName'] = 'appShopBlocked';
        $templatePath = $this->di->getObjectManager()->get('\App\Core\Models\User')->getTemplatePath($templateData);

        !empty($criticalMail) && $data['email'] = $criticalMail;
        !empty($bccs) && $data['bccs'] = $bccs;

        $data['path'] = $templatePath['path'];
        $data['subject'] = 'App Shop Blocked';
        $data['endpoint_route'] = $route;

        // Send Critical Mail Notification to the mentioned critical mail
        if (!empty($criticalMail)) {
            $this->di->getObjectManager()->get('App\Core\Components\SendMail')->send($data);
        }

        if (!empty($criticalMailEndpoint)) {
            $subscription['queue_data'] = $data;
            $destination['destination_data']['api_endpoint']['endpoint'] = $criticalMailEndpoint;
            $this->di->getObjectManager()->get(Apiendpoint::class)
                ->sendNotificationToDestination($destination, $subscription);
        }
    }


    /**
     * Check subuser status before process API call requests
     *
     * @return boolean
     */
    public function isSubUserActive()
    {
        $token = $this->di->getRegistry()->getDecodedToken();
        $subUserId = $token['subuser_id'] ?? $token['child_id'] ?? '';
        if (isset($subUserId) && !empty($subUserId)) {
            $subUserStatus = $this->di->getCache()->get($subUserId . "_sub_user_status");
            // if sub_user_status not found in cache then fetch status from DB
            if (!isset($subUserStatus) && empty($subUserStatus)) {
                $subUser = $this->di->getObjectManager()
                    ->get('\App\Core\Models\User\SubUser')->getSubUser(['id' => $subUserId]);
                if (isset($subUser['data'], $subUser['data']['status']) && !empty($subUser['data']['status'])) {
                    $subUserStatus = $subUser['data']['status'];
                } else {
                    return true;
                }
            }
            if (!empty($subUserStatus) && $subUserStatus === 'active') {
                return true;
            }
            return false;
        }
        return true;
    }
}
