<?php

/**
 * CedCommerce
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the End User License Agreement (EULA)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://cedcommerce.com/license-agreement.txt
 *
 * @category    Ced
 * @package     Ced_Shopifynxtgen
 * @author      CedCommerce Core Team <connect@cedcommerce.com>
 * @copyright   Copyright CEDCOMMERCE (http://cedcommerce.com/)
 * @license     http://cedcommerce.com/license-agreement.txt
 */

namespace App\Shopifywebapi\Components\Authenticate;

use App\Shopifywebapi\Components\Authenticate\Common;
use Exception;

class Sellerauth extends Common
{
    /**
     * Constructor.
     *
     * @param bool $private If this is a private or public app
     *
     * @return self
     */
    public function _construct()
    {
        $this->_callName = 'authURL';
        parent::_construct();
    }

    /*
     * Call if the app is yet to be published on Shopify app store
     *
     * @param Uri $uri
     *
     * @return string
     */
    public function fetchAuthenticationUrl($shopUrl)
    {
        if (!empty($this->_error)) return $this->_error;
        $this->_shop = $shopUrl;
        $redirectUrl = $this->getAuthUrl($this->_scope, $this->_redirectUrl, $this->_state);
        if (!$redirectUrl['success']) return $redirectUrl;
        else return [
            'success' => true,
            'authUrl' => $redirectUrl['authUrl']
        ];
    }

    /**
     * Gets the authentication URL for Shopify to allow the user to accept the app (for public apps).
     *
     * @param string|array $scopes      The API scopes as a comma seperated string or array
     * @param string       $redirectUri The valid redirect URI for after acceptance of the permissions.
     *                                  It must match the redirect_uri in your app settings.
     *
     * @return string Formatted URL
     */
    public function getAuthUrl($scopes, string $redirectUri, $state)
    {
        if ($this->_apiKey === null) {
            return [
                'success' => false,
                'message' => 'API key is missing'
            ];
        }

        if (is_array($scopes)) {
            $scopes = implode(',', $scopes);
        }
        return [
            'success' => true,
            'authUrl' => (string) $this->getBaseUri()
                ->withPath('/admin/oauth/authorize')
                ->withQuery(http_build_query([
                    'client_id'    => $this->_apiKey,
                    'scope'        => $scopes,
                    'redirect_uri' => $redirectUri,
                    'state' => $state
                ]))
        ];
    }

    /**
     * Verify the incoming request from Shopify and fetch Token
     *
     * @param string|array $scopes      The API scopes as a comma seperated string or array
     * @param string       $redirectUri The valid redirect URI for after acceptance of the permissions.
     *                                  It must match the redirect_uri in your app settings.
     *
     * @return string Formatted URL
     */
    public function verifyRequestandFetchToken($postData)
    {
        if ($this->verifyRequest($postData)) {
            $this->_shop = $postData['shop'];
            $this->_callName = 'fetchToken';
            $tokenInfo = $this->requestAccessToken($postData['code']);

            if (!$tokenInfo['success']) return $tokenInfo;
            return ['success' => true, 'token' => $tokenInfo['token']];
            // token : dce46c2bc3ce917a862685850542218f
        } else {
            return [
                'success' => false,
                'message' => 'Sorry , user cannot be validated !! Please try again. If problem persists, contact our 24*7 friendly support.'
            ];
        }
    }

    /**
     *
     */
    public function getUserInfo()
    {
        // pull info using user-id
        $params['shop'] = 'anshuman-ced.myshopify.com';
        $tokenInfo['token']  = 'df5e865982c83f1330ce1343797eded2';
    }


    /**
     * Is used to check whether all the credentials are correct or not
     * @param array $postData
     * @return array
     */
    public function verifyUserDetails(&$postData)
    {
        $helper = $this->di->getObjectManager()->get('App\Shopifywebapi\Components\Helper');
        if (!empty($postData['state'])) {
            $state = json_decode(base64_decode($postData['state']), true)
                ?: (json_decode($postData['state'], true) ?? []);
            $postData += $state;
            $postData['state'] = json_encode($state);
          
        }
        $shopUrl = $postData['shop'];
        $accessToken = $postData['accessToken'];
        $clientId = $postData['clientId'];
        $clientSecret = $postData['clientSecret'];
        $clientCredentialsResponse = $this->verifyClientCredentials($clientId, $clientSecret, $shopUrl, $accessToken);
        if ($clientCredentialsResponse['success']) {
            $configScopes = array_map('trim', explode(',', (string)$this->_scope)); //Scopes fetched from DB
            $tokenResponse = $this->getTokenScopes($shopUrl, $accessToken, $configScopes);
            $appDetails = $helper->getIdInfo($this->fetchShopifyAppId($shopUrl, $accessToken));
            $postData['shopify_app_id'] = $appDetails['id'];
            return $tokenResponse;
        } else {
            return $clientCredentialsResponse;
        }
    }

    /**
     * Checks whether the given client credentials are valid or not with respect to the given shop url
     * @param string $clientId
     * @param string $clientSecret
     * @param string $shopUrl
     * @return array []
     */
    public function verifyClientCredentials($clientId, $clientSecret, $shopUrl, $accessToken)
    {
        $apiVersion = $this->_restApiVersion;
        $url = 'https://' . $clientId . ':' . $accessToken . '@' . $shopUrl . '/admin/api/' . $apiVersion . '/products.json';

        // $url = 'https://'.$clientId.':'.$clientSecret.'@'.$shopUrl.'/admin/api/2022-04/products.json';
        $headers = [
            'Content-Type' => 'application/json'
        ];

        try {
            $response = $this->di->getObjectManager()->get('App\Core\Components\Guzzle')->call($url, $headers);
            if (isset($response['errors'])) {
                return [
                    'success' => false,
                    'message' => $response['errors'],
                ];
            }

            return [
                'success' => true,
                'response' => 'ClientId, AccessToken and shopUrl successfully verified'
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Incorrect Shopify store credentials!'
            ];
        }
    }

    /**
     * Checks whether the given token is correct or not with respect to shop url and token has all the required scopes
     * @param string $shopUrl
     * @param string $accessToken
     * @return array []
     */
    public function getTokenScopes($shopUrl, $accessToken, $configScopes)
    {
        try {
            $url = 'https://' . $shopUrl . '/admin/oauth/access_scopes.json';
            $headers = [
                'X-Shopify-Access-Token' => $accessToken,
            ];
            $tokenScopesResponse = $this->di->getObjectManager()->get('App\Core\Components\Guzzle')
                ->call($url, $headers);

            if (isset($tokenScopesResponse['errors'])) {
                return [
                    'success' => false,
                    'message' => $tokenScopesResponse['errors']
                ];
            }
            if (empty($tokenScopesResponse['access_scopes']) || !is_array($tokenScopesResponse['access_scopes'])) {
                return [
                    'success' => false,
                    'message' => 'Please provide access scopes or check the shop url entered is correct.'
                ];
            }
            $tokenScopesValidationResponse = $this->countTokenScopesAndValidate($tokenScopesResponse, $configScopes);

            if ($tokenScopesValidationResponse['success']) {
                return [
                    "success" => true,
                    'token' => $accessToken
                ];
            }

            return [
                'success' => false,
                'message' => isset($tokenScopesValidationResponse['message']) ? $tokenScopesValidationResponse['message'] : 'Missing Scope(s). Kindly check all the required scopes'
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => 'Incorrect Shopify store credentials!'
            ];
        }
    }


    /**
     * Compare the scopes configured in token and required scopes and validate all the required scopes are given
     * @param array $tokenScopes
     * @return array
     */
    public function countTokenScopesAndValidate($tokenScopes, $configScopes)
    {
        $necessaryScopes = [];
        $necessaryScopesCount = 0; // Number of scopes required by our app
        foreach ($configScopes as $value) {
            $necessaryScopes[$value] = 0; //Necessary scopes required by our app saving in key value pairs key is scope and value is by default 0 which means its not verified yet
            $necessaryScopesCount += 1;
        }
        $validatedScopesCount = 0; //Number of scopes validated 
        foreach ($tokenScopes['access_scopes'] as $key => $value) {
            $scope = $tokenScopes['access_scopes'][$key]['handle'];
            if (isset($necessaryScopes[$scope])) {
                $validatedScopesCount += 1;
                $necessaryScopes[$scope] = 1; //Modifying the default value of necessary scopes from 0 to 1 which means this scope is verified now
            }
        }

        if ($validatedScopesCount < $necessaryScopesCount) {
            $missingScopes = [];
            foreach ($necessaryScopes as $key => $value) {
                if ($value == 0) {
                    array_push($missingScopes, $key);
                }
            }
            return [
                'success' => false,
                'message' => 'Not enough scopes. Missing scope(s) :' . implode(", ", $missingScopes)
            ];
        }

        return [
            'success' => true,
            'response' => 'Token Scopes successfully verified.'
        ];
    }

    public function fetchShopifyAppId($shopUrl, $accessToken)
    {
        $apiVersion = $this->_gqlApiVersion;

        $query = '{
        app {
            id
            }
        }
        ';

        $url = 'https://' . $shopUrl . '/admin/api/' . $apiVersion . '/graphql.json';

        $headers = [
            'Content-Type' => 'application/graphql',
            'X-Shopify-Access-Token' => $accessToken
        ];
        $response = $this->di->getObjectManager()->get('App\Core\Components\Guzzle')
            ->call($url, $headers, $query, 'POST', 'body');
        $id = $response['data']['app']['id'];
        return $id;
    }
}
