<?php
/**
 * Copyright © 2016 Magento. All rights reserved.
 * See COPYING.txt for license details.
 */
namespace App\Rmq\Components;

use App\Rmq\Components\MessageQueue\ConnectionLostException;
use App\Rmq\Components\MessageQueue\EnvelopeInterface;
use App\Rmq\Components\MessageQueue\QueueInterface;
use PhpAmqpLib\Exception\AMQPProtocolConnectionException;
use PhpAmqpLib\Message\AMQPMessage;
use App\Rmq\Components\MessageQueue\EnvelopeFactory;
use App\Rmq\Components\App\DeploymentConfig;

/**
 * Class Queue
 */
class Rqueue implements QueueInterface
{

    /**
     * Durability for exchange and queue
     */
    const IS_DURABLE = true;

    /**
     * @var Config
     */
    private $amqpConfig;

    /**
     * @var string
     */
    private $queueName;

    /**
     * @var EnvelopeFactory
     */
    private $envelopeFactory;

    /**
     * @var LoggerInterface $logger
     */
    private $logger;
    /**
     * @var LoggerInterface $logger
     */
    private $di;

    /* on failure requeue process $maxFailCount times.*/
    private $maxFailCount = 5;


    public $trackProgress = false;
    /**
     * Initialize dependencies.
     *
     * @param Config $amqpConfig
     * @param EnvelopeFactory $envelopeFactory
     * @param string $queueName
     */
    public function __construct(
        Config $amqpConfig,
        EnvelopeFactory $envelopeFactory,
        $queueName,
        $di
    ) {
        $this->di = $di;
        $this->amqpConfig = $amqpConfig;
        $this->queueName = $queueName;
        $this->envelopeFactory = $envelopeFactory;
        $this->logger = $this->di->getLog();
    }

    /**
     * {@inheritdoc}
     */
    public function dequeue()
    {
        $envelope = null;
        $channel = $this->amqpConfig->getChannel();
        // @codingStandardsIgnoreStart
        /** @var AMQPMessage $message */
        try {
            $message = $channel->basic_get($this->queueName);
        } catch (AMQPProtocolConnectionException $e) {
            throw new ConnectionLostException(
                $e->getMessage(),
                $e->getCode(),
                $e
            );
        }

        if ($message !== null) {
            $properties = array_merge(
                $message->get_properties(),
                [
                    'topic_name' => $message->delivery_info['routing_key'],
                    'delivery_tag' => $message->delivery_info['delivery_tag'],
                ]
            );
            $envelope = $this->envelopeFactory->create(['body' => $message->body, 'properties' => $properties]);
        }

        // @codingStandardsIgnoreEnd
        return $envelope;
    }

    /**
     * {@inheritdoc}
     */
    public function acknowledge(EnvelopeInterface $envelope)
    {
        $properties = $envelope->getProperties();
        $channel = $this->amqpConfig->getChannel();
        // @codingStandardsIgnoreStart
        try {
            $channel->basic_ack($properties['delivery_tag']);
        } catch (AMQPProtocolConnectionException $e) {
            throw new ConnectionLostException(
                $e->getMessage(),
                $e->getCode(),
                $e
            );
        }
        // @codingStandardsIgnoreEnd
    }

    /**
     * {@inheritdoc}
     */
    public function subscribe()
    {
        $skipQueues  = [
            'process_maintenance'=>'',
          //  'process_failed'=>'',
        ];
        $maxFailCount = $this->maxFailCount;
        $callback = function ($msg) use ($maxFailCount, $skipQueues) {
            try{
                //$hash = spl_object_hash($this);
                $file = "queues/{$this->queueName}-{$msg->delivery_info['consumer_tag']}.log";

                $this->logger->logContent($msg->body, \Phalcon\Logger::DEBUG, $file);
                $msgArray = json_decode($msg->body, true);
                $msgArray['consumer_tag'] = "{$this->queueName}-{$msg->delivery_info['consumer_tag']}";
                $start_time = microtime(true);
                $this->logger->logContent("Started queue message processing ".PHP_EOL, \Phalcon\Logger::DEBUG, "queue-processing/{$this->queueName}-{$msg->delivery_info['consumer_tag']}.log");
                $end_time = false;
                $processResult = $this->processQueueMsg($this->queueName, $msgArray);
                $end_time = microtime(true);
                $execution_time = ($end_time - $start_time);
                $this->logger->logContent("End queue message processing".PHP_EOL."Total time taken :".$execution_time.PHP_EOL, \Phalcon\Logger::DEBUG, "queue-processing/{$this->queueName}-{$msg->delivery_info['consumer_tag']}.log");

                //$this->logger->close( "queue-processing/{$this->queueName}-{$msg->delivery_info['consumer_tag']}.log");

                if ($processResult) {
                    $msg->delivery_info['channel']->basic_ack($msg->delivery_info['delivery_tag']); /*sending acknowledgement that task is done*/
                    if($processResult === true){
                        if(isset($msgArray['message_reference_id'])){
                            $model = $this->di->getObjectManager()->create('\App\Core\Models\BaseMongo');
                            $model->setSource('queue_messages');
                            $model->getCollection()->deleteOne(['_id' => $msgArray['message_reference_id']]);

                        }
                        if ($this->trackProgress && !isset($skipQueues[$msgArray['queue_name']]) && isset($msgArray['message_id']) && $msgArray['message_id']) {
                            $message = \App\Rmq\Models\Message::findFirst($msgArray['message_id']);
                            $progress = $message->progress+$msgArray['own_weight'];
                            if ($progress<100) {
                                $status = 1;
                            } else {
                                $status = 2;
                            }
                            if($message)
                                $message->save(['progress' => $progress,
                                    'status' => $status
                                ]);
                            if ($msgArray['is_child']) {
                                $message = \App\Rmq\Models\Message::findFirst($msgArray['parent_id']);
                                $progress = $message->progress+$msgArray['weight_for_parent'];
                                if($message)
                                    $message->save(['progress' => $progress
                                    ]);
                            }
                        }

                    }

                    $this->logger->logContent('DONE', \Phalcon\Logger::DEBUG, $file);
                } else {
                    /*process failed */
                    $messageBody = $msgArray;
                    if (isset($messageBody['failed_count'])) {
                        if ($messageBody['failed_count']<=$maxFailCount) {
                            $this->logger->logContent('Failed Rqueue again', \Phalcon\Logger::DEBUG, $file);
                            /*Requeue*/
                            $messageBody['failed_count']++;
                            $this->push(json_encode($messageBody));
                        } else {
                            $this->queueFailedMessage($messageBody, $skipQueues, $file);
                        }
                    } else {
                        if ($maxFailCount>0) {
                            $messageBody['failed_count'] = 1;
                            $this->logger->logContent('Failed Rqueue again', \Phalcon\Logger::DEBUG, $file);
                            $this->push(json_encode($messageBody));
                        } else {
                            $this->queueFailedMessage($messageBody, $skipQueues, $file);
                        }
                    }

                    $msg->delivery_info['channel']->basic_ack($msg->delivery_info['delivery_tag']);
                }
                //$this->logger->close($file);

            }
            catch(\PhpAmqpLib\Exception\AMQPRuntimeException $e){
                if(!$end_time){
                    $execution_time = ($end_time - $start_time);
                    $this->logger->logContent("AMQPRuntimeException arised".PHP_EOL."Total time taken :".$execution_time.PHP_EOL, \Phalcon\Logger::DEBUG, "queue-processing/{$this->queueName}-{$msg->delivery_info['consumer_tag']}.log");

                }
                $this->logger->logContent(
                    'AMQPRuntimeException Queue Name : '.$this->queueName.':'.$e->getMessage().PHP_EOL.$e->getTraceAsString(),
                    \Phalcon\Logger::CRITICAL,
                    "workers-error.log"
                );
                exit;
            }
            catch(\Exception $e){
                if(!$end_time){
                    $execution_time = ($end_time - $start_time);
                    $this->logger->logContent("Exception arised".PHP_EOL."Total time taken :".$execution_time.PHP_EOL, \Phalcon\Logger::DEBUG, "queue-processing/{$this->queueName}-{$msg->delivery_info['consumer_tag']}.log");

                }
                $this->logger->logContent(
                    'Exception Queue Name : '.$this->queueName.':'.$e->getMessage().PHP_EOL.$e->getTraceAsString(),
                    \Phalcon\Logger::CRITICAL,
                    "workers-error.log"
                );
                exit;
            }

        };

        $channel = $this->amqpConfig->getChannel();

        $channel->basic_qos(null, 1, null);/*prefetch_count = 1 this parameter tells to rabbit mq that dont send message to consumer till get acknwoledgement of previous task*/

        // @codingStandardsIgnoreStart
        $channel->basic_consume($this->queueName, '', false, false, false, false, $callback);
        // @codingStandardsIgnoreEnd
        while (count($channel->callbacks)) {
            $channel->wait();
        }
    }

    /* function for adding message to failed queue if it fails for more than Max Limit */

    public function queueFailedMessage($messageBody, $skipQueues, $file)
    {
        $messageBody['failed_count'] = 0;
        if (!isset($messageBody['failed_times'])) {
            $messageBody['failed_times'] = 1;
        } else {
            $messageBody['failed_times']++;
        }
       
        $data = ['message'=>$messageBody,'queue_name'=>$this->di->getConfig()->get('app_code').'_process_failed','type'=>'class','method'=>'requeueFailedData','class_name'=>'Default1'];

        
        if ($messageBody['failed_times']<=3) {
            $this->createQueue($this->di->getConfig()->get('app_code').'_process_failed', json_encode($data));
            if ($this->trackProgress && !isset($skipQueues[$messageBody['queue_name']]) && isset($messageBody['message_id'])) {
                $message = \App\Rmq\Models\Message::findFirst($messageBody['message_id']);
                if ($message) {
                    $message->save([
                                    'status' => 3
                        ]);
                }
            }
            $this->logger->logContent('Failed Reached up to max limit.Queue this process in process_failed', \Phalcon\Logger::DEBUG, $file);
        } else {
            if ($this->trackProgress &&  !isset($skipQueues[$messageBody['queue_name']]) && isset($messageBody['message_id'])) {
                $message = \App\Rmq\Models\Message::findFirst($messageBody['message_id']);
                if ($message) {
                    $message->save([
                                    'status' => 5
                        ]);
                }
            }
            $data = ['message'=>$messageBody,'queue_name'=>$this->di->getConfig()->get('app_code').'_process_failed_permanently','type'=>'class','method'=>'processPermanentFailedData','class_name'=>'Default1'];
            $this->createQueue($this->di->getConfig()->get('app_code').'_process_failed_permanently', json_encode($data));
            $this->logger->logContent('Process failed permanentaly', \Phalcon\Logger::DEBUG, $file);
        }
    }

    public function createQueue($queueName, $data)
    {

        $deploymentConfig = new DeploymentConfig();
        $config = new Config($deploymentConfig);
        $envelopeFactory = new EnvelopeFactory();
        $rqueue = new Rqueue($config, $envelopeFactory, $queueName, $this->di);

        $rqueue->pushToQueue($queueName,$data);

    }

    public function canProcessThisRequest(&$msg){

        $currentTime = time();
        if(isset($msg['throttle_group']) && isset($this->di->getConfig()->throttle[$msg['throttle_group']])){
            $config = $this->di->getConfig()->throttle[$msg['throttle_group']];
            $throttleSeconds = $config['seconds'];
            $throttleLimit = $config['limit'];
        }
        else{
            $throttleSeconds = 1;
            $throttleLimit = 1;
        }
        $data = [
                'message_id'=>0,
                'token_id'=>$msg['message_id'],
                'method'=>'prepareToken',
                'queue_name'=>$this->di->getConfig()->get('app_code').'_prepare_token',
                'type'=>'class',
                'class_name'=>'Default1',
                'data_throttle_key' => $msg['throttle_key'],
                'throttle_seconds' => $throttleSeconds,
                'throttle_limit' => $throttleLimit
                
        ];
        $this->createQueue($data['queue_name'], json_encode($data));
        $token = $this->getToken($msg['message_id']);
        $startTime = $currentTime;
        while($token===false){
            if(time()-$startTime> 2){
                return false;
            }
            usleep(500000); /*half second*/
            $token = $this->getToken($msg['message_id']);
        }

        return $token;


    }

    public function getToken($messageId){

        $file = BP.DS.'var'.DS.'locks'.DS.$messageId;
        if(file_exists($file)){
            $result = file_get_contents($file);
            unlink($file);
            return $result;
        }
        else{
            return false;
        }

    }
    public function processQueueMsg($queueName, $msg)
    {
        if(isset($msg['run_after']) && $msg['run_after'] > time()){
            $this->createQueue($msg['queue_name'], json_encode($msg));
            return 1;
        }
        $baseUrl = 'https://apps.cedcommerce.com/integration/';
        $msgArray = $msg;
        if(isset($msgArray['throttle_key'])){
            if(!$this->canProcessThisRequest($msgArray)){
                $this->createQueue($msg['queue_name'], json_encode($msg));
                return 1;
            }
        }
        if(isset($msgArray['user_id'])){
            $user = \App\Core\Models\User::findFirst($msgArray['user_id']);
            $this->di->setUser($user);
            $decodedToken = [
                'role'=>'admin',
                'user_id'=>$msgArray['user_id'],
            ];
            $this->di->getRegistry()->setDecodedToken($decodedToken);
        }

        if (isset($msgArray['handler'])) {
             $helper = $this->di->getObjectManager()->get($msgArray['handler']);
             $result = $helper->processMsg($msgArray);

            if ($this->trackProgress && !$result) {
                $message = \App\Rmq\Models\Message::findFirst($msgArray['message_id']);
                if($message){
                    $message->progress = 0;
                    $message->save();
                }

            }
        } else {
            $helper = $this->di->getObjectManager()->get('\App\Rmq\Components\Helper');
            $result = $helper->processMsg($msgArray);

            if ($this->trackProgress && !$result) {
                $message = \App\Rmq\Models\Message::findFirst($msgArray['message_id']);
                if($message){
                    $message->progress = 0;
                    $message->save();
                }

            }
        }
        if (isset($msg['response_handler'])) {
            $responseHandler = $msg['response_handler'];
            $method = $msg['response_method'];
            $responseHandler = $this->di->getObjectManager()->get($responseHandler);
            $responseHandler->$method($msg, $result);
        }

        foreach ($this->di->getConfig()->databases as $key => $database) {
            if ($database['adapter'] != 'Mongo') {
                $this->di->get($key)->connect();
            }
        }
        return $result;
    }

    public function requeueFailedData($msg)
    {
        $data = json_decode($msg, true);
        
        if (isset($data['queue_name'])) {
            $this->createQueue($data['queue_name'], json_encode($data['message']));
        }
        return true;
    }

    public function requeueMaintenanceData($msg)
    {
        $data = json_decode($msg, true);
        
        if (isset($data['queue_name'])) {
            $this->createQueue($data['queue_name'], json_encode($data['message']));
        }
        return true;
    }
    public function callCurl($url, $data, $json = true, $returnTransfer = true)
    {

        $data_string = json_encode($data);
        // Initialize cURL
        $ch = curl_init();
        // Set URL on which you want to post the Form and/or data
        curl_setopt($ch, CURLOPT_URL, $url);
        // Data+Files to be posted
        if ($json) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $data_string);
            curl_setopt($ch, CURLOPT_HTTPHEADER, array(
                'Content-Type: application/json',
                'Content-Length: ' . strlen($data_string)));
        } else {
            curl_setopt($ch, CURLOPT_POST, 1);
            // Data+Files to be posted

            curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
        }
        // Pass TRUE or 1 if you want to wait for and catch the response against the request made
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, $returnTransfer);
        curl_setopt($ch, CURLOPT_HEADER, false);


        // For Debug mode; shows up any error encountered during the operation
        curl_setopt($ch, CURLOPT_VERBOSE, 1);

        
        // Execute the request
        $response = curl_exec($ch);
        
        $httpcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        if (curl_errno($ch)) {
            return ['responseCode'=>$httpcode,'errorMsg'=>curl_error($ch)];
        }
        
        curl_close($ch);
        if ($httpcode!=200) {
           // preg_match_all('/<div class=\"header(.*?)\">(.*?)<\/div>/s',$response,$output,PREG_SET_ORDER);
            //preg_match_all('/<div class="header">(.*?)</div>/s', $response, $output);
            
            return ['responseCode'=>$httpcode,'errorMsg'=>$response];
            //var_dump($output);
        } else {
            return ['responseCode'=>$httpcode,'response'=>$response];
        }
        // Just for debug: to see response
    }

    /**
     * (@inheritdoc)
     */
    public function reject(EnvelopeInterface $envelope, $requeue = true, $rejectionMessage = null)
    {
        $properties = $envelope->getProperties();

        $channel = $this->amqpConfig->getChannel();
        // @codingStandardsIgnoreStart
        $channel->basic_reject($properties['delivery_tag'], $requeue);
        // @codingStandardsIgnoreEnd
        if ($rejectionMessage !== null) {
            // @todo log critical message $rejectionMessage
        }
    }


    /**
     * Declare Amqp Queue
     *
     * @param string $queueName
     * @return void
     */
    private function declareQueue($queueName)
    {
        $this->amqpConfig->getChannel()->queue_declare($queueName, false, self::IS_DURABLE, false, false);
    }

    /**
     * (@inheritdoc)
     */
    public function push($messsage, $properties = ['delivery_mode' => 2])
    {


        $this->declareQueue($this->queueName);

        $envelope = $this->envelopeFactory->create(['body' => $messsage, 'properties' => $properties]);
        $messageProperties = $envelope->getProperties();
        $msg = new AMQPMessage(
            $envelope->getBody(),
            [
                'delivery_mode' => $messageProperties['delivery_mode']
            ]
        );
        $this->amqpConfig->getChannel()->basic_publish($msg, '', $this->queueName);
    }


    /**
     * push message to queue
     */

    public function pushToQueue($queueName,$messsage,$properties = ['delivery_mode' => 2]){
        $this->declareQueue($queueName);

        $envelope = $this->envelopeFactory->create(['body' => $messsage, 'properties' => $properties]);
        $messageProperties = $envelope->getProperties();
        $msg = new AMQPMessage(
            $envelope->getBody(),
            [
                'delivery_mode' => $messageProperties['delivery_mode']
            ]
        );
        $this->amqpConfig->getChannel()->basic_publish($msg, '', $queueName);
    }
}