<?php

/**
 * CedCommerce
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the End User License Agreement (EULA)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://cedcommerce.com/license-agreement.txt
 *
 * @category    Ced
 * @package     Ced_Michaelswebapi
 * @author      CedCommerce Core Team <connect@cedcommerce.com>
 * @copyright   Copyright CEDCOMMERCE (http://cedcommerce.com/)
 * @license     http://cedcommerce.com/license-agreement.txt
 */

namespace App\Michaelswebapi\Components;

use App\Michaelswebapi\Components\Core\Common as CoreCommon;
use PhpOffice\PhpSpreadsheet\Cell\Coordinate;
use PhpOffice\PhpSpreadsheet\IOFactory;

/**
 * Class Listing
 * @package App\Michaelswebapi\Components
 */
class Listing extends CoreCommon
{
    private $headerColumnNames = [];

    private $taxonomyHeaderColumnNames = [];

    const TAXONOMY_ATTRIBUTE_LABEL_VALUE_SEPARATOR_FOR_INDEX = ',';

    /**
     * @param $data
     * @return mixed
     */
    public function getQueryListing($data)
    {
        return $this->rest(
            'GET',
            self::GET_QUERY_LISTING_END_POINT,
            $data,
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function getAllListing($data)
    {
        return $this->rest(
            'GET',
            self::GET_ALL_LISTINGS,
            $data,
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function getListingBySkuNumber($data)
    {
        return $this->rest(
            'GET',
            str_replace('{skuNumber}', $data['skuNumber'], self::GET_LISTING_BY_SKU_NUMBER_END_POINT),
            [],
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function getInventoriesByPrimarySkuNumbers($data)
    {
        return $this->rest(
            'GET',
            str_replace(
                '{primarySkuNumber}',
                $data['primarySkuNumber'],
                self::GET_SKU_INVENTORIES_BY_PRIMARY_SKU_NUMBER_END_POINT
            ),
            [],
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function getPriceByPrimarySkuNumbers($data)
    {
        return $this->rest(
            'GET',
            str_replace(
                '{primarySkuNumber}',
                $data['primarySkuNumber'],
                self::GET_SKU_PRICE_BY_PRIMARY_SKU_NUMBER_END_POINT
            ),
            [],
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function createListing($data)
    {
        // $this->di->getLog()->logContent('json' . json_encode($data,JSON_PRETTY_PRINT),'info','checkingWEBAPI');

        if (!isset($data['priceAndInventory'])) {
            $data['priceAndInventory'] = [
                'price' => null,
                'quantity' => null,
            ];
        }

        return $this->rest(
            'POST',
            self::CREATE_LISTING_END_POINT,
            $data,
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function updatePriceBySellerSkuNumbers($data)
    {
        return $this->rest(
            'POST',
            self::UPDATE_SKU_PRICE_BY_SELLER_SKU_NUMBERS_END_POINT,
            $data['data'],
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function getListingInBatchBySkuNumbers($data)
    {
        return $this->rest(
            'POST',
            self::GET_LISTING_IN_BATCH_BY_SKU_NUMBERS_END_POINT,
            $data['sku_numbers'],
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function getListingInBatchBySellerSkuNumbers($data)
    {
        return $this->rest(
            'POST',
            self::GET_LISTING_IN_BATCH_BY_SELLER_SKU_NUMBERS_END_POINT,
            $data['seller_sku_numbers'],
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function getInventoryInBatchBySkuNumbers($data)
    {
        return $this->rest(
            'POST',
            self::GET_INVENTORIES_IN_BATCH_BY_SKU_NUMBERS_END_POINT,
            $data,
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function getSkuPriceInBatchBySkuNumbers($data)
    {
        return $this->rest(
            'POST',
            self::GET_SKU_PRICE_IN_BATCH_BY_SKU_NUMBERS_END_POINT,
            $data,
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function updateInventoryBySellerSkuNumber($data)
    {
        return $this->rest(
            'POST',
            self::UPDATE_INVENTORY_BY_SELLER_SKU_NUMBERS_END_POINT,
            $data['data'],
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function updateListing($data)
    {
        return $this->rest(
            'PUT',
            str_replace('{primarySkuNumber}', $data['skuNumber'], self::UPDATE_LISTING_END_POINT),
            $data,
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function updateListingBySellerSkuNumber($data)
    {
        return $this->rest(
            'PUT',
            str_replace(
                '{sellerSkuNumber}',
                $data['sellerSkuNumber'],
                self::UPDATE_A_LISTING_BY_SELLER_SKU_NUMBER_END_POINT
            ),
            $data,
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function activateListing($data)
    {
        return $this->rest(
            'POST',
            self::ACTIVATE_LISTING_END_POINT,
            $data,
            []
        );
    }

    public function cancelUploadTasks($data)
    {

        return $this->rest(
            'POST',
            self::CANCEL_BULK_TASK_END_POINT,
            $data,
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function deactivateListing($data)
    {
        return $this->rest(
            'POST',
            self::DEACTIVATE_LISTING_END_POINT,
            $data,
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function activateDeactivateSkus($data)
    {

        if (!is_array($data['data'])) {
            return [
                'success' => false,
                'message' => self::DATA_SHOULD_BE_AN_ARRAY_ERROR_MESSAGE,
            ];
        }
        return $this->rest(
            'POST',
            self::ACTIVATE_DEACTIVATE_SKUS_END_POINT,
            $data['data'],
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function updateInventoryBySkuNumbers($data)
    {
        return $this->rest(
            'POST',
            self::UPDATE_INVENTORY_BY_SKU_NUMBERS_END_POINT,
            $data['inventory'],
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function updatePriceBySkuNumbers($data)
    {
        return $this->rest(
            'POST',
            self::UPDATE_PRICE_BY_SKU_NUMBERS_END_POINT,
            $data['price'],
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function getUploadListingTasks($data)
    {
        return $this->rest(
            'GET',
            self::GET_UPLOAD_LISTING_TASKS_END_POINT,
            $data,
            []
        );
    }

    /**
     * @param $data
     * @return array
     */
    public function getInventoryInBatchBySellerSkuNumber($data)
    {
        if (!is_array($data['sellerSkuNumbers'])) {
            return [
                'success' => false,
                'message' => self::SELLER_SKU_NUMBERS_SHOULD_BE_AN_ARRAY_ERROR_MESSAGE,
            ];
        }
        return $this->rest(
            'POST',
            self::GET_INVENTORY_IN_BATCH_BY_SELLER_SKU_NUMBERS_END_POINT,
            $data,
            []
        );
    }

    /**
     * @param $data
     * @return array
     */
    public function getPriceInBatchBySellerSkuNumber($data)
    {
        if (!is_array($data['sellerSkuNumbers'])) {
            return [
                'success' => false,
                'message' => self::SELLER_SKU_NUMBERS_SHOULD_BE_AN_ARRAY_ERROR_MESSAGE,
            ];
        }
        return $this->rest(
            'POST',
            self::GET_PRICE_IN_BATCH_BY_SELLER_SKU_NUMBERS_END_POINT,
            $data,
            []
        );
    }

    /**
     * @param $data
     * @return mixed
     */
    public function exportExcel($data)
    {

        $response = $this->rest(
            'EXPORT_EXCEL',
            self::EXPORT_EXCEL_END_POINT,
            $data,
            []
        );

        $statusCode = $response->getStatusCode();
        $responseData = [];
        if ($statusCode == 200) {

            /* Write response in excel file start */
            $content = $response->getBody()->getContents();
            $excelExportVarDirectory = BP . DS . 'var' . DS . self::MICHAELS_EXPORT_DIRECTORY_NAME;

            $writeExcelName = self::WRITE_EXCEL_PREFIX . $data['shop_id'] . '.xlsx';

            if (!file_exists($excelExportVarDirectory) && !is_dir($excelExportVarDirectory)) {
                @mkdir($excelExportVarDirectory, 0777, true);
            }
            $filePath = $excelExportVarDirectory . '/' . $writeExcelName;

            $fp = @fopen($filePath, 'a');
            @fwrite($fp, $content);
            @fclose($fp);
            // return ['success'=>false,'message'=>'here '.$filePath];

            /* Write response in excel file end */

            /* Prepare Custom response by read excel file start */
            $defaultRequestColumns = [
                'seller_sku' => 'seller_sku',
                'sku_number' => 'sku_number',
                'item_name' => 'item_name',
                'price' => 'price',
                'quantity' => 'quantity',
                'global_trade_item_number' => 'global_trade_item_number',
                'status' => 'status',
                'sku_type' => 'sku_type',
                // 'variant_value' => 'variant_value',
            ];
            $requestColumns = $defaultRequestColumns;
            if (isset($data['requestAttribute']) && is_array($data['requestAttribute']) && count($data['requestAttribute'])) {
                $requestColumns = array_merge($requestColumns, $data['requestAttribute']);
            }
            //$objReader = IOFactory::createReader('Xlsx');
            $objReader = $this->di->get(\PhpOffice\PhpSpreadsheet\Reader\Xlsx::class);
            $objReader->setIncludeCharts(true);
            $objPHPExcel = $objReader->load($filePath);
            $activeSheet = $objPHPExcel->getSheetByName(self::ACTIVE_WORKSHEET_NAME);
            $highestColumnIndex = Coordinate::columnIndexFromString($activeSheet->getHighestDataColumn());
            $taxonomyColumnStartIndex = Coordinate::columnIndexFromString(self::EXPORT_EXCEL_TAXONOMY_ATTRIBUTE_COLUMN_STARTS_FROM);
            $this->setHeaderColumnNames($taxonomyColumnStartIndex - 1, $activeSheet);
            $this->setTaxonomyHeaderColumnNames($taxonomyColumnStartIndex, $highestColumnIndex, $activeSheet);

            $highestRowCount = $activeSheet->getHighestDataRow();
            $rowStartFrom = self::EXCEL_HEADER_ROW + 1;
            for ($rowIndex = $rowStartFrom; $rowIndex <= $highestRowCount; $rowIndex++) {
                $rowData = [];
                foreach ($requestColumns as $key => $column) {
                    if ($key == 'taxonomyAttribute') {
                        if (is_array($column) && count($column)) {
                            foreach ($column as $attributeCode => $attributeValue) {
                                $index = $attributeCode .
                                    self::TAXONOMY_ATTRIBUTE_LABEL_VALUE_SEPARATOR_FOR_INDEX .
                                    trim($attributeValue);
                                if (isset($this->taxonomyHeaderColumnNames[$index])) {
                                    $rowData[$attributeCode] = $activeSheet->getCell($this->taxonomyHeaderColumnNames[$index] . $rowIndex)->getValue();
                                }
                            }
                        }
                    } elseif (is_string($column)) {
                        if ($column == 'tag') {
                            $rowData[$column] = $this->getTags($activeSheet, $rowIndex);
                        } elseif ($column == 'image_url') {
                            $rowData[$column] = $this->getImageUrls($activeSheet, $rowIndex);
                        } elseif ($column == 'variant_value') {
                            $rowData[$column] = $this->getVariantValues($activeSheet, $rowIndex);
                        } else {
                            // $this->di->getLog()->logContent('column - ' . $column . 'data == > ' . json_encode($$this->headerColumnNames[$column],JSON_PRETTY_PRINT),'info','checkingWEBAPI');
                            if (isset($this->headerColumnNames[$column])) {
                                $rowData[$column] = $activeSheet->getCell($this->headerColumnNames[$column] . $rowIndex)->getValue();
                            }
                        }
                    }
                }
                if (count($rowData)) {
                    $responseData[] = $rowData;
                }
            }
            $this->deleteFile($filePath);
            /* Prepare Custom response by read excel file end */
            return [
                'success' => true,
                'message' => '',
                'data' => $responseData,
            ];
        } else {
            $responseData = json_decode($response->getBody()->getContents(), true);
            $responseData['success'] = false;
        }

        return $responseData;
    }

    /**
     * @param $filePath
     */
    private function deleteFile($filePath)
    {
        if (file_exists($filePath)) {
            @unlink($filePath);
        }
    }

    /**
     * @param $data
     * @return mixed
     */
    public function createUpdateListingByExcelTemplate($data)
    {
        try {
            if (!isset($data['file_path'])) {
                return [
                    'success' => false,
                    'message' => '\'file_path\' parameter not present.',
                ];
            }
            $filePath = $data['file_path'];
            // if (!file_exists($filePath)) {
            //     return [
            //         'success' => false,
            //         'message' => 'File not found.',
            //         'file_path' => $filePath,
            //     ];
            // }
            $requestData = [];
            $requestData[] = [
                'name' => 'uploadFile',
                'contents' => file_get_contents($data['file_path']),
                'filename' => basename($data['file_path']),
            ];
            $response = $this->rest(
                'MEDIA',
                self::CREATE_UPDATE_LISTING_BY_EXCEL_TEMPLATE_END_POINT,
                $requestData,
                []
            );
            //$this->deleteFile($filePath);
            return $response;
        } catch (\Exception $exception) {
            return [
                'code' => 400,
                'success' => false,
                'message' => $exception->getMessage(),
            ];
        }
        return [
            'code' => 400,
            'success' => false,
            'message' => '',
        ];
    }

    /**
     * @param $data
     * @return mixed
     */
    public function asynchronousCreateUpdateListingByExcelTemplate($data)
    {
        try {
            if (!isset($data['file_path'])) {
                return [
                    'success' => false,
                    'message' => '\'file_path\' parameter not present.',
                ];
            }
            $filePath = $data['file_path'];
            if (!file_exists($filePath)) {
                return [
                    'success' => false,
                    'message' => 'File not found.',
                ];
            }
            $requestData = [];
            $requestData[] = [
                'name' => 'uploadFile',
                'contents' => file_get_contents($data['file_path']),
                'filename' => basename($data['file_path']),
            ];
            $response = $this->rest(
                'MEDIA',
                self::ASYNCHRONOUS_CREATE_UPDATE_LISTING_BY_EXCEL_TEMPLATE_END_POINT,
                $requestData,
                []
            );
            //$this->deleteFile($filePath);
            return $response;
        } catch (\Exception $exception) {
            return [
                'code' => 400,
                'success' => false,
                'message' => $exception->getMessage(),
            ];
        }
        return [
            'code' => 400,
            'success' => false,
            'message' => '',
        ];
    }

    /**
     * @param $data
     * @return mixed
     */
    public function createMediaUrl($data)
    {
        try {
            if (!isset($data['file_path'])) {
                return [
                    'success' => false,
                    'message' => '\'file_path\' parameter not present.',
                ];
            }
            $filePath = $data['file_path'];
            if (!file_exists($filePath)) {
                return [
                    'success' => false,
                    'message' => 'File not found.',
                ];
            }
            $requestData = [];
            $requestData[] = [
                'name' => 'uploadFile',
                'contents' => file_get_contents($data['file_path']),
                'filename' => basename($data['file_path']),
            ];
            $response = $this->rest(
                'MEDIA',
                self::CREATE_MEDIA_URL_END_POINT,
                $requestData,
                []
            );
            return $response;
        } catch (\Exception $exception) {
            return [
                'code' => 400,
                'success' => false,
                'message' => $exception->getMessage(),
            ];
        }
        return [
            'code' => 400,
            'success' => false,
            'message' => '',
        ];
    }

    /**
     * @param $filePath
     * @param $columnName
     * @param $rowNumberSaperator
     * @param $errorMessage
     * @param $errorMessageSaperator
     * @return array
     * @throws \PhpOffice\PhpSpreadsheet\Exception
     * @throws \PhpOffice\PhpSpreadsheet\Reader\Exception
     */
    private function getMessageByCellValue(
        $filePath,
        $columnName,
        $rowNumberSaperator,
        $errorMessage,
        $errorMessageSaperator
    ) {
        $message = [];
        $errorMessage = strstr($errorMessage, $errorMessageSaperator, false);
        $errorMessage = trim(str_replace($errorMessageSaperator, '', $errorMessage));
        $rowNumber = (int) strstr($errorMessage, $rowNumberSaperator, true);
        $objReader = IOFactory::createReader('Xlsx');
        $objReader->setIncludeCharts(true);
        $objPHPExcel = $objReader->load($filePath);
        $activeSheet = $objPHPExcel->getSheetByName(self::ACTIVE_WORKSHEET_NAME);
        if (isset($this->headerColumnNames[$columnName])) {
            $sellerSku = $activeSheet->getCell($this->headerColumnNames[$columnName] . $rowNumber)->getValue();
            $errorMessage = strstr($errorMessage, $rowNumber . $rowNumberSaperator, false);
            $errorMessage = trim(str_replace($rowNumber . $rowNumberSaperator, '', $errorMessage));
            $message = [
                $sellerSku => $errorMessage,
            ];
        }
        return $message;
    }

    /**
     * @param $highestColumnIndex
     * @param $activeSheet
     */
    private function setHeaderColumnNames($highestColumnIndex, $activeSheet)
    {
        $this->headerColumnNames = [];
        for ($col = 1; $col <= $highestColumnIndex; $col++) {
            $columnName = Coordinate::stringFromColumnIndex($col);
            $cellValue = $activeSheet->getCell($columnName . self::EXCEL_HEADER_ROW)->getValue();
            if (!empty($cellValue)) {
                $this->headerColumnNames[$cellValue] = $columnName;
            }
        }
    }

    /**
     * @param $taxonomyColumnStartIndex
     * @param $highestColumnIndex
     * @param $activeSheet
     */
    private function setTaxonomyHeaderColumnNames($taxonomyColumnStartIndex, $highestColumnIndex, $activeSheet)
    {
        $this->taxonomyHeaderColumnNames = [];
        $excelHeaderPreviousRow = self::EXCEL_HEADER_ROW - 1;
        for ($col = $taxonomyColumnStartIndex; $col <= $highestColumnIndex; $col++) {
            $columnName = Coordinate::stringFromColumnIndex($col);
            $attributeCodeCellValue = $activeSheet->getCell($columnName . self::EXCEL_HEADER_ROW)->getValue();
            $attributeLabelCellValue = $activeSheet->getCell($columnName . $excelHeaderPreviousRow)->getValue();
            if (!empty($attributeCodeCellValue) && !empty($attributeLabelCellValue)) {
                $index = $attributeCodeCellValue .
                    self::TAXONOMY_ATTRIBUTE_LABEL_VALUE_SEPARATOR_FOR_INDEX .
                    trim($attributeLabelCellValue);
                $this->taxonomyHeaderColumnNames[$index] = $columnName;
            }
        }
    }

    /**
     * @param $activeSheet
     * @param $rowIndex
     * @return array
     */
    private function getTags($activeSheet, $rowIndex)
    {
        $tags = [];
        for ($tagCount = 1; $tagCount <= self::MAX_TAG_COUNT; $tagCount++) {
            if (isset($this->headerColumnNames[self::TAG_PREFIX_FOR_EXCEL . $tagCount])) {
                $tagValue = $activeSheet->getCell($this->headerColumnNames[self::TAG_PREFIX_FOR_EXCEL . $tagCount] . $rowIndex)->getValue();
                if (!empty($tagValue)) {
                    $tags[] = $tagValue;
                }
            }
        }
        return $tags;
    }

    /**
     * @param $activeSheet
     * @param $rowIndex
     * @return array
     */
    private function getImageUrls($activeSheet, $rowIndex)
    {
        $imageUrls = [];
        for ($imageCount = 1; $imageCount <= self::MAX_IMAGE_COUNT; $imageCount++) {
            if (isset($this->headerColumnNames[self::IMAGE_URL_PREFIX_FOR_EXCEL . $imageCount])) {
                $imageValue = $activeSheet->getCell($this->headerColumnNames[self::IMAGE_URL_PREFIX_FOR_EXCEL . $imageCount] . $rowIndex)->getValue();
                if (!empty($imageValue)) {
                    $imageUrls[] = $imageValue;
                }
            }
        }
        return $imageUrls;
    }

    /**
     * @param $activeSheet
     * @param $rowIndex
     * @return array
     */
    private function getVariantValues($activeSheet, $rowIndex)
    {
        $imageUrls = [];
        for ($imageCount = 1; $imageCount <= self::MAX_VARIANT_VALUE_COUNT; $imageCount++) {
            if (isset($this->headerColumnNames[self::VARIANT_PREFIX_FOR_EXCEL . $imageCount])) {
                $imageValue = $activeSheet->getCell($this->headerColumnNames[self::VARIANT_PREFIX_FOR_EXCEL . $imageCount] . $rowIndex)->getValue();
                if (!empty($imageValue)) {
                    $imageUrls[] = $imageValue;
                }
            }
        }
        return $imageUrls;
    }
    public function deleteListingBySkuNumber($data)
    {
        $data = ['skuNumbers' => $data['skuNumbers']];
        return $this->rest(
            'DELETE/FORM',
            self::DELETE_LISTING_BY_SKU_ENDPOINT,
            $data,
            []
        );
    }
}
