<?php

/**
 * Copyright © Cedcommerce, Inc. All rights reserved.
 * See LICENCE.txt for license details.
 */

namespace App\Connector\Service;

use App\Connector\Contracts\Sales\Order\ShipInterface;
use App\Connector\Service\AbstractShipment;

/**
 * Interface CurrencyInterface
 * @services
 */
class Shipment extends AbstractShipment implements ShipInterface
{
    /**
     * ship function
     * For preparing Target Shipment
     * @param array $data
     * @return array
     */
    public function ship(array $data): array
    {
        $this->mongo = $this->di->getObjectManager()->create('\App\Core\Models\BaseMongo');
        $orderCollection = $this->mongo->getCollectionForTable('order_container');
        $arrayParams = ["typeMap" => ['root' => 'array', 'document' => 'array']];
        $params = [
            'user_id' => $data['user_id'],
            '$or' => [['object_type' => 'source_order'], ['object_type' => 'target_order']],
            'marketplace_reference_id' => (string) $data['marketplace_reference_id']
        ];
        $orderData = $orderCollection->findOne($params, $arrayParams);
        if (!empty($orderData)) {
            $this->objectType = $orderData['object_type'];
            $checkParams = [
                'user_id' => $data['user_id'],
                'object_type' => ['$regex' => 'shipment'],
                'marketplace_reference_id' => (string)$data['marketplace_reference_id'],
                'reference_id' => ['$exists' => true]
            ];
            $shipmentCheck = $orderCollection->findOne($checkParams, $arrayParams);
            if (!empty($shipmentCheck)) {
                $errorParams = [
                    'user_id' => $data['user_id'],
                    'object_type' => $this->objectType,
                    'marketplace_reference_id' => (string)$data['marketplace_reference_id'],
                ];
                $orderCollection->updateOne(
                    $errorParams,
                    [
                        '$addToSet' => ['shipment_error' => 'Shipment tried from wrong source']
                    ]
                );
                return ['success' => false, 'message' => 'Shipment tried from wrong source'];
            }
            if ($this->objectType == "source_order") {
                $sourceShipment = $this->prepareSourceShipment($data);
                if (isset($sourceShipment['success']) && $sourceShipment['success']) {
                    if (isset($sourceShipment['data']) && !empty($sourceShipment['data'])) {
                        $targetShipment = $this->prepareTargetShipment($data, $sourceShipment['data']);
                        return $targetShipment;
                    }
                }
                return $sourceShipment;
            } else {
                $targetShipment = $this->prepareTargetShipment($data);
                if (isset($targetShipment['success']) && $targetShipment['success']) {
                    if (isset($targetShipment['data']) && !empty($targetShipment['data'])) {
                        $sourceShipment = $this->prepareSourceShipment($data, $targetShipment['data']);
                        return $sourceShipment;
                    }
                }
                return $targetShipment;
            }
        }
        return ['success' => false, 'message' => 'OrderData Not Found'];
    }

    /**
     * prepareTargetShipment function
     * For preparing Target Shipment
     * @param [array] $data
     * @param [array] $sourceShipment
     * @return array
     */
    public function prepareTargetShipment($data, $sourceShipment = [])
    {
        $this->mongo = $this->di->getObjectManager()->create('\App\Core\Models\BaseMongo');
        $orderCollection = $this->mongo->getCollectionForTable('order_container');
        if ($this->objectType == "source_order") {
            foreach ($sourceShipment['targets'] as $target) {
                if (($target['marketplace'] == $data['target_marketplace'])
                    && $target['shop_id'] == $data['target_shop_id']
                ) {
                    $targetOrderId = $target['order_id'];
                }
            }
        } else {
            $targetOrderId = $data['marketplace_reference_id'];
        }
        if (!empty($targetOrderId)) {
            $targetParams = [
                'user_id' => $data['user_id'],
                'object_type' => 'target_order',
                'marketplace_reference_id' => (string) $targetOrderId
            ];
            $arrayParams = ["typeMap" => ['root' => 'array', 'document' => 'array']];
            $targetData = $orderCollection->findOne($targetParams, $arrayParams);
            $items = $errorSku = [];
            if (!empty($targetData)) {
                $toBeUpdated = false;
                $shipmentParams = [
                    'user_id' => $data['user_id'],
                    'object_type' => 'target_shipment',
                    'marketplace' => $targetData['marketplace'],
                    'marketplace_shop_id' => $targetData['marketplace_shop_id'],
                    'marketplace_shipment_id' => (string) $data['marketplace_shipment_id']
                ];
                $shipmentData = $orderCollection->findOne($shipmentParams, $arrayParams);
                if ($targetData['object_type'] == $this->objectType) {
                    if (!empty($shipmentData)) {
                        $toBeUpdated = true;
                    }
                    $getItems = $this->prepareItems($data, $targetData, $toBeUpdated);
                    $prepareData = [
                        'object_type' => 'target_shipment',
                        'marketplace' => $data['target_marketplace'],
                        'shop_id' => $data['target_shop_id'],
                        'targets_marketplace' => $data['source_marketplace'],
                        'targets_status' => "In Progress",
                        'targets_shop_id' => $data['source_shop_id'],
                        'targets_order_id' => $getItems['orderId'],
                        'marketplace_reference_id' => $targetData['marketplace_reference_id'],
                        'cif_order_id' => $targetData['cif_order_id'],
                        'items' => $getItems['items'],
                        'totalQty' => $getItems['totalQuantity'],
                        'shippedQty' => $getItems['quantityShipped'],
                        'data' => $data,
                    ];

                    $targetShipment = $this->prepareShipmentData($prepareData);
                    if ($toBeUpdated) {
                        $targetShipmentCopy = $targetShipment;
                        unset($targetShipmentCopy['items']);
                        unset($targetShipmentCopy['created_at']);
                        $orderCollection->updateOne($shipmentParams, ['$set' => $targetShipmentCopy]);
                    } else {
                        $saveShipment = $orderCollection->insertOne($targetShipment);
                        $id = $saveShipment->getInsertedId();
                        $itemIds = [];
                        $items = $getItems['items'];
                        $parentStatus = $getItems['totalQuantity'] == $getItems['savedShippedQuantity'] ? "Shipped" : "Partially Shipped";
                        foreach ($items as $key => $value) {
                            $shippedQty = $getItems['orderItems'][$value['marketplace_item_id']]['shipped_qty'];
                            $itemStatus = $shippedQty == $value['qty'] ? 'Shipped' : 'Partially Shipped';
                            $items[$key]['shipment_id'] = (string) $id;
                            $saveShipment = $orderCollection->insertOne($items[$key]);
                            array_push($itemIds, (string) $saveShipment->getInsertedId());
                            $orderCollection->updateOne(
                                [
                                    'user_id' => $data['user_id'],
                                    'object_type' => 'target_order_items',
                                    'marketplace_item_id' => (string) $value['marketplace_item_id']
                                ],
                                ['$set' => ['item_status' => $itemStatus, 'shipped_qty' => $shippedQty]]
                            );
                        }
                        $orderCollection->updateOne(
                            ['_id' => $id],
                            ['$set' => ['items' => $items, 'shipment_id' => (string) $id, "target_shipment_id" => $itemIds]]
                        );
                        $sourceParams = [
                            'user_id' => $data['user_id'],
                            'object_type' => 'source_order',
                            'marketplace_reference_id' => (string)$getItems['orderId'],
                            "shop_id" => (string)$data['source_shop_id'],
                            "targets.marketplace" => (string)$data['target_marketplace']
                        ];
                        $targetParams = [
                            'user_id' => $data['user_id'],
                            'object_type' => 'target_order',
                            'marketplace_reference_id' => (string) $data['marketplace_reference_id']
                        ];
                        $prepareDocData = [
                            'sourceParams' => $sourceParams,
                            'targetParams' => $targetParams,
                            'orderItems' => $getItems['orderItems'],
                            'status' => $targetData['status'],
                            'parentStatus' => $parentStatus
                        ];
                        $this->updateDocs($prepareDocData);
                        $targetShipment['shipment_id'] = (string) $id;
                    }
                    return ['success' => true, 'data' => $targetShipment];
                } else {
                    if (!empty($shipmentData)) {
                        $tracking = $shipmentData['tracking'];
                        if (
                            $tracking['company'] == $data['tracking']['company']
                            && $tracking['number'] == $data['tracking']['number']
                            && ($targetData['status'] == "Shipped" || $targetData['status'] == "Partially Shipped")
                        ) {
                            return ['success' => false, 'message' => 'Same Fulfillment tried to re-attempt'];
                        } else {
                            $toBeUpdated = true;
                        }
                    }
                    $mongo = $this->di->getObjectManager()->create('\App\Core\Models\BaseMongo');
                    $orderCollection = $mongo->getCollectionForTable('order_container');
                    $productByQueryObj = $this->di->getObjectManager()->create(\App\Connector\Models\Product\Marketplace::class);
                    $usedItemCode = [];
                    if ($this->checkSku($targetData['items'])) {
                        foreach ($data['items'] as $sourceItem) {
                            $matchedItem = [];
                            $message = "Product Not Found SKU: ";
                            foreach ($targetData['items'] as $targetItem) {
                                if (
                                    $targetItem['sku'] == $sourceItem['sku']
                                ) {
                                    if (!isset($targetItem['marketplace_shipment_id'])) {
                                        if (!in_array($targetItem['marketplace_item_id'], $usedItemCode)) {
                                            $matchedItem = $targetItem;
                                            $usedItemCode[] = $targetItem['marketplace_item_id'];
                                            break;
                                        }
                                    } else {
                                        if ($data['marketplace_shipment_id'] == $targetItem['marketplace_shipment_id']) {
                                            if (!in_array($targetItem['marketplace_item_id'], $usedItemCode)) {
                                                $matchedItem = $targetItem;
                                                $usedItemCode[] = $targetItem['marketplace_item_id'];
                                                break;
                                            }
                                        }
                                    }
                                } elseif (isset($sourceItem['product_identifier']) && !empty($sourceItem['product_identifier'])) {
                                    $query = [
                                        'user_id' => $targetData['user_id'],
                                        'source_marketplace' => $targetData['marketplace'],
                                        'shop_id' => (string)$targetData['marketplace_shop_id'],
                                        'source_product_id' => (string)$sourceItem['product_identifier']
                                    ];
                                    $arrayParams = ["typeMap" => ['root' => 'array', 'document' => 'array']];
                                    $targetProduct = $productByQueryObj->getProductByQuery($query, $arrayParams);
                                    if (isset($targetProduct) && !empty($targetProduct)) {
                                        if (!isset($targetProduct[0]['sku'])) {
                                            $query = [
                                                'user_id' => $data['user_id'],
                                                'target_marketplace' => $data['source_marketplace'],
                                                'shop_id' => (string)$data['source_shop_id'],
                                                'source_product_id' => (string)$sourceItem['product_identifier']
                                            ];
                                            $sourceProduct = $productByQueryObj->getProductByQuery($query, $arrayParams);
                                            if (isset($sourceProduct) && !empty($sourceProduct)) {
                                                $targetSku = $sourceProduct[0]['sku'] ?? '';
                                            }
                                        } else {
                                            $targetSku = $targetProduct[0]['sku'] ?? '';
                                        }
                                        if ($targetItem['sku'] == $targetSku) {
                                            if (!isset($targetItem['marketplace_shipment_id'])) {
                                                if (!in_array($targetItem['marketplace_item_id'], $usedItemCode)) {
                                                    $matchedItem = $targetItem;
                                                    $usedItemCode[] = $targetItem['marketplace_item_id'];
                                                    break;
                                                }
                                            } else {
                                                if ($data['marketplace_shipment_id'] == $targetItem['marketplace_shipment_id']) {
                                                    if (!in_array($targetItem['marketplace_item_id'], $usedItemCode)) {
                                                        $matchedItem = $targetItem;
                                                        $usedItemCode[] = $targetItem['marketplace_item_id'];
                                                        break;
                                                    }
                                                }
                                            }
                                        } else {
                                            $message = "Invalid Source Mapped SKU: ";
                                        }
                                    }
                                } else {
                                    $product = $this->getProductSku(
                                        $sourceShipment['shop_id'],
                                        $targetData['marketplace_shop_id'],
                                        $sourceItem['sku']
                                    );
                                    if ($product['success']) {
                                        $productData = $product['data'];
                                        if ($productData['sku'] == $targetItem['sku']) {
                                            if (!isset($targetItem['marketplace_shipment_id'])) {
                                                if (!in_array($targetItem['marketplace_item_id'], $usedItemCode)) {
                                                    $matchedItem = $targetItem;
                                                    $usedItemCode[] = $targetItem['marketplace_item_id'];
                                                    break;
                                                }
                                            } else {
                                                if ($data['marketplace_shipment_id'] == $targetItem['marketplace_shipment_id']) {
                                                    if (!in_array($targetItem['marketplace_item_id'], $usedItemCode)) {
                                                        $matchedItem = $targetItem;
                                                        $usedItemCode[] = $targetItem['marketplace_item_id'];
                                                        break;
                                                    }
                                                }
                                            }
                                        } else {
                                            $message = "Invalid Source Mapped SKU: ";
                                        }
                                    }
                                }
                            }

                            if (!empty($matchedItem)) {
                                $temp = [
                                    "type" => $matchedItem['type'],
                                    "title" => $matchedItem['title'],
                                    "sku" => $matchedItem['sku'],
                                    "qty" => $matchedItem['qty'],
                                    "weight" => $sourceItem['weight'],
                                    "shipped_qty" => $sourceItem['shipped_qty'],
                                    "object_type" => "target_shipment_items",
                                    "user_id" => $data['user_id'],
                                    "cif_order_id" => $targetData['cif_order_id'],
                                    "marketplace_item_id" => $matchedItem['marketplace_item_id'] ?? "",
                                    'item_status' => $sourceItem['shipped_qty'] == $matchedItem['qty']
                                        ? "Shipped" : "Partially Shipped",
                                ];
                                array_push($items, $temp);
                                $filter[] =
                                    [
                                        'updateOne' => [
                                            [
                                                'user_id' => $data['user_id'],
                                                'object_type' => 'target_order',
                                                'marketplace_reference_id' => $targetData['marketplace_reference_id'],
                                                'items.marketplace_item_id' => $matchedItem['marketplace_item_id']
                                            ],
                                            [
                                                '$set' => ['items.$.marketplace_shipment_id' => (string)$data['marketplace_shipment_id']]
                                            ],
                                        ],
                                    ];
                            } else {
                                $errorSku[] = $sourceItem['sku'];
                            }
                        }
                        if (!empty($filter)) {
                            $orderCollection->bulkWrite($filter);
                        }
                    } else {
                        foreach ($data['items'] as $sourceItem) {
                            $matchedItem = [];
                            $message = "Product Not Found SKU: ";
                            foreach ($targetData['items'] as $targetItem) {
                                if (
                                    $targetItem['sku'] == $sourceItem['sku']
                                ) {
                                    $matchedItem = $targetItem;
                                    break;
                                } elseif (isset($sourceItem['product_identifier']) && !empty($sourceItem['product_identifier'])) {
                                    $query = [
                                        'user_id' => $targetData['user_id'],
                                        'source_marketplace' => $targetData['marketplace'],
                                        'shop_id' => (string)$targetData['marketplace_shop_id'],
                                        'source_product_id' => (string)$sourceItem['product_identifier']
                                    ];
                                    $arrayParams = ["typeMap" => ['root' => 'array', 'document' => 'array']];
                                    $targetProduct = $productByQueryObj->getProductByQuery($query, $arrayParams);
                                    if (isset($targetProduct) && !empty($targetProduct)) {
                                        if (!isset($targetProduct[0]['sku'])) {
                                            $query = [
                                                'user_id' => $data['user_id'],
                                                'target_marketplace' => $data['source_marketplace'],
                                                'shop_id' => (string)$data['source_shop_id'],
                                                'source_product_id' => (string)$sourceItem['product_identifier']
                                            ];
                                            $sourceProduct = $productByQueryObj->getProductByQuery($query, $arrayParams);
                                            if (isset($sourceProduct) && !empty($sourceProduct)) {
                                                $targetSku = $sourceProduct[0]['sku'] ?? '';
                                            }
                                        } else {
                                            $targetSku = $targetProduct[0]['sku'] ?? '';
                                        }
                                        if ($targetItem['sku'] == $targetSku) {
                                            $matchedItem = $targetItem;
                                            break;
                                        } else {
                                            $message = "Invalid Source Mapped SKU: ";
                                        }
                                    }
                                } else {
                                    $product = $this->getProductSku(
                                        $sourceShipment['shop_id'],
                                        $targetData['marketplace_shop_id'],
                                        $sourceItem['sku']
                                    );
                                    if ($product['success']) {
                                        $productData = $product['data'];
                                        if ($productData['sku'] == $targetItem['sku']) {
                                            $matchedItem = $targetItem;
                                            break;
                                        } else {
                                            $message = "Invalid Source Mapped SKU: ";
                                        }
                                    }
                                }
                            }

                            if (!empty($matchedItem)) {
                                $temp = [
                                    "type" => $matchedItem['type'],
                                    "title" => $matchedItem['title'],
                                    "sku" => $matchedItem['sku'],
                                    "qty" => $matchedItem['qty'],
                                    "weight" => $sourceItem['weight'],
                                    "shipped_qty" => $sourceItem['shipped_qty'],
                                    "object_type" => "target_shipment_items",
                                    "user_id" => $data['user_id'],
                                    "cif_order_id" => $targetData['cif_order_id'],
                                    "marketplace_item_id" => $matchedItem['marketplace_item_id'] ?? "",
                                    'item_status' => $sourceItem['shipped_qty'] == $matchedItem['qty']
                                        ? "Shipped" : "Partially Shipped",
                                ];
                                array_push($items, $temp);
                            } else {
                                $errorSku[] = $sourceItem['sku'];
                            }
                        }
                    }
                    if (!empty($errorSku)) {
                        $response = $this->checkInvalidSku($errorSku, $targetParams, $message);
                        return $response;
                    }
                    if (isset($targetData['shipment_error'])) {
                        $orderCollection->updateOne($targetParams, ['$set' => ["shipping_status" => "In Progress"]]);
                    }
                    $prepareData = [
                        'object_type' => 'target_shipment',
                        'marketplace' => $data['target_marketplace'],
                        'shop_id' => $data['target_shop_id'],
                        'targets_marketplace' => $data['source_marketplace'],
                        'targets_shop_id' => $data['source_shop_id'],
                        'targets_status' => $sourceShipment['status'],
                        'targets_order_id' => $data['marketplace_reference_id'],
                        'marketplace_reference_id' => $targetData['marketplace_reference_id'],
                        'cif_order_id' => $targetData['cif_order_id'],
                        'data' => $data,
                        'items' => $items,
                    ];
                    $targetShipment = $this->prepareShipmentData($prepareData);

                    if ($toBeUpdated) {
                        $targetShipmentCopy = $targetShipment;
                        unset($targetShipmentCopy['items']);
                        unset($targetShipmentCopy['created_at']);
                        $orderCollection->updateOne($shipmentParams, ['$set' => $targetShipmentCopy]);
                    } else {
                        $itemIds = [];
                        $saveTargetShipment = $orderCollection->insertOne($targetShipment);
                        $id = $saveTargetShipment->getInsertedId();

                        foreach ($items as $key => $targetShipmentItem) {
                            $items[$key]['shipment_id'] = (string) $id;
                            $saveShipment = $orderCollection->insertOne($items[$key]);
                            array_push($itemIds, (string) $saveShipment->getInsertedId());
                        }
                        $orderCollection->updateOne(
                            ['_id' => $id],
                            ['$set' => ['items' => $items, 'shipment_id' => (string) $id, "target_shipment_id" => $itemIds]]
                        );
                        $targetShipment['items'] = $items;
                        $getSourceShipment = $orderCollection->findOne(['object_type' => "source_shipment", 'user_id' => $targetData['user_id'], 'marketplace_shipment_id' => (string)$data['marketplace_shipment_id']], $arrayParams);
                        $this->addKey("target_shipment", $targetData['marketplace_reference_id'], $getSourceShipment);
                    }
                    return ['success' => true, 'data' => $targetShipment];
                }
            }
        }
        return ['success' => false, 'message' => 'Unable to Prepare Shipment'];
    }

    /**
     * prepareSourceShipment function
     * For preparing Source Shipment
     * @param [array] $data
     * @param [array] $targetShipment
     * @return array
     */
    public function prepareSourceShipment($data, $targetShipment = [])
    {
        $sourceOrderId = '';
        if ($this->objectType != "source_order") {
            foreach ($targetShipment['targets'] as $source) {
                if (($source['marketplace'] == $data['source_marketplace'])
                    && $source['shop_id'] == $data['source_shop_id']
                ) {
                    $sourceOrderId = $source['order_id'];
                }
            }
        } else {
            $sourceOrderId = $data['marketplace_reference_id'];
        }
        $items = $errorSku = [];
        if (!empty($sourceOrderId)) {
            $orderCollection = $this->mongo->getCollectionForTable('order_container');
            $sourceParams = [
                'user_id' => $data['user_id'], 'object_type' => 'source_order',
                'marketplace_reference_id' => $sourceOrderId
            ];
            $arrayParams = ["typeMap" => ['root' => 'array', 'document' => 'array']];
            $sourceData = $orderCollection->findOne($sourceParams, $arrayParams);
            if (!empty($sourceData)) {
                $toBeUpdated = false;
                $shipmentParams = [
                    'user_id' => $data['user_id'],
                    'object_type' => 'source_shipment',
                    'marketplace' => $sourceData['marketplace'],
                    'marketplace_shop_id' => $sourceData['marketplace_shop_id'],
                    'marketplace_shipment_id' => (string) $data['marketplace_shipment_id']
                ];
                $shipmentData = $orderCollection->findOne($shipmentParams, $arrayParams);
                if ($sourceData['object_type'] == $this->objectType) {
                    if (!empty($shipmentData)) {
                        $toBeUpdated = true;
                    }
                    $targetOrderId = $sourceData['targets'][0]['order_id'];
                    $getItems = $this->prepareItems($data, $sourceData, $toBeUpdated);
                    $prepareData = [
                        'object_type' => 'source_shipment',
                        'marketplace' => $data['source_marketplace'],
                        'shop_id' => $data['source_shop_id'],
                        'targets_marketplace' => $data['target_marketplace'],
                        'targets_status' => 'In Progress',
                        'targets_shop_id' => $data['target_shop_id'],
                        'targets_order_id' => $getItems['orderId'],
                        'marketplace_reference_id' => $sourceData['marketplace_reference_id'],
                        'cif_order_id' => $sourceData['cif_order_id'],
                        'items' => $getItems['items'],
                        'totalQty' => $getItems['totalQuantity'],
                        'shippedQty' => $getItems['quantityShipped'],
                        'data' => $data,

                    ];
                    $sourceShipment = $this->prepareShipmentData($prepareData);
                    if ($toBeUpdated) {
                        $sourceShipmentCopy = $sourceShipment;
                        unset($sourceShipmentCopy['items']);
                        unset($sourceShipmentCopy['created_at']);
                        $orderCollection->updateOne($shipmentParams, ['$set' => $sourceShipmentCopy]);
                    } else {
                        $saveShipment = $orderCollection->insertOne($sourceShipment);
                        $id = $saveShipment->getInsertedId();
                        $itemIds = [];
                        $items = $getItems['items'];
                        $parentStatus = $getItems['totalQuantity'] == $getItems['savedShippedQuantity'] ? "Shipped" : "Partially Shipped";
                        foreach ($items as $key => $value) {
                            $shippedQty = $getItems['orderItems'][$value['marketplace_item_id']]['shipped_qty'];
                            $itemStatus = $shippedQty == $value['qty'] ? 'Shipped' : 'Partially Shipped';
                            $items[$key]['shipment_id'] = (string) $id;
                            $saveShipment = $orderCollection->insertOne($items[$key]);
                            array_push($itemIds, (string) $saveShipment->getInsertedId());
                            $orderCollection->updateOne(
                                [
                                    'user_id' => $data['user_id'],
                                    'object_type' => 'source_order_items',
                                    'marketplace_item_id' => (string) $value['marketplace_item_id']
                                ],
                                ['$set' => ['item_status' => $itemStatus, 'shipped_qty' => $shippedQty]]
                            );
                        }
                        $orderCollection->updateOne(
                            ['_id' => $id],
                            ['$set' => ['items' => $items, 'shipment_id' => (string) $id, "source_shipment_id" => $itemIds]]
                        );

                        $sourceParams = [
                            'user_id' => $data['user_id'],
                            'object_type' => 'source_order',
                            'marketplace_reference_id' => (string) $data['marketplace_reference_id']
                        ];
                        $targetParams = [
                            'user_id' => $data['user_id'],
                            'object_type' => 'target_order',
                            'marketplace_reference_id' => (string)$targetOrderId,
                            "shop_id" => (string)$data['target_shop_id'],
                            "targets.marketplace" => (string)$data['source_marketplace']
                        ];
                        $prepareDocData = [
                            'sourceParams' => $targetParams,
                            'targetParams' => $sourceParams,
                            'orderItems' => $getItems['orderItems'],
                            'status' => $sourceData['status'],
                            'parentStatus' => $parentStatus
                        ];
                        $this->updateDocs($prepareDocData);
                        $sourceShipment['shipment_id'] = (string) $id;
                    }
                    return ['success' => true, 'data' => $sourceShipment];
                }
                if (!empty($shipmentData)) {
                    $tracking = $shipmentData['tracking'];
                    if (
                        $tracking['company'] == $data['tracking']['company']
                        && $tracking['number'] == $data['tracking']['number']
                        && ($sourceData['status'] == "Shipped" || $sourceData['status'] == "Partially Shipped")
                    ) {
                        return ['success' => false, 'message' => 'Same Fulfillment tried to re-attempt'];
                    } else {
                        $toBeUpdated = true;
                    }
                }
                $productByQueryObj = $this->di->getObjectManager()->create(\App\Connector\Models\Product\Marketplace::class);
                $message = "Product Not Found SKU: ";
                $usedItemCode = [];
                if ($this->checkSku($sourceData['items'])) {
                    foreach ($data['items'] as $targetItem) {
                        $matchedItem = [];
                        foreach ($sourceData['items'] as $sourceItem) {
                            if (
                                $sourceItem['sku'] == $targetItem['sku']
                            ) {
                                if (!isset($sourceItem['marketplace_shipment_id'])) {
                                    if (!in_array($sourceItem['marketplace_item_id'], $usedItemCode)) {
                                        $matchedItem = $sourceItem;
                                        $usedItemCode[] = $sourceItem['marketplace_item_id'];
                                        break;
                                    }
                                } else {
                                    if ($data['marketplace_shipment_id'] == $sourceItem['marketplace_shipment_id']) {
                                        if (!in_array($sourceItem['marketplace_item_id'], $usedItemCode)) {
                                            $matchedItem = $sourceItem;
                                            $usedItemCode[] = $sourceItem['marketplace_item_id'];
                                            break;
                                        }
                                    }
                                }
                            } elseif (isset($targetItem['product_identifier']) && !empty($targetItem['product_identifier'])) {
                                $query = [
                                    'user_id' => $sourceData['user_id'],
                                    'target_marketplace' => $sourceData['marketplace'],
                                    'shop_id' => (string)$sourceData['marketplace_shop_id'],
                                    'source_product_id' => (string)$targetItem['product_identifier']
                                ];
                                $arrayParams = ["typeMap" => ['root' => 'array', 'document' => 'array']];
                                $sourceProduct = $productByQueryObj->getProductByQuery($query, $arrayParams);
                                if (isset($sourceProduct) && !empty($sourceProduct)) {
                                    if (!isset($sourceProduct[0]['sku'])) {
                                        $query = [
                                            'user_id' => $data['user_id'],
                                            'source_marketplace' => $data['target_marketplace'],
                                            'shop_id' => (string)$data['target_shop_id'],
                                            'source_product_id' => (string)$targetItem['product_identifier']
                                        ];
                                        $targetProduct = $productByQueryObj->getProductByQuery($query, $arrayParams);
                                        if (isset($targetProduct) && !empty($targetProduct)) {
                                            $sourceSku = $targetProduct[0]['sku'] ?? '';
                                        }
                                    } else {
                                        $sourceSku = $sourceProduct[0]['sku'] ?? '';
                                    }
                                    if ($sourceItem['sku'] == $sourceSku) {
                                        if (!isset($sourceItem['marketplace_shipment_id'])) {
                                            if (!in_array($sourceItem['marketplace_item_id'], $usedItemCode)) {
                                                $matchedItem = $sourceItem;
                                                $usedItemCode[] = $sourceItem['marketplace_item_id'];
                                                break;
                                            }
                                        } else {
                                            if ($data['marketplace_shipment_id'] == $sourceItem['marketplace_shipment_id']) {
                                                if (!in_array($sourceItem['marketplace_item_id'], $usedItemCode)) {
                                                    $matchedItem = $sourceItem;
                                                    $usedItemCode[] = $sourceItem['marketplace_item_id'];
                                                    break;
                                                }
                                            }
                                        }
                                    } else {
                                        $message = "Invalid Target Mapped SKU: ";
                                    }
                                }
                            } else {
                                $product = $this->getProductSku(
                                    $targetShipment['shop_id'],
                                    $sourceData['marketplace_shop_id'],
                                    $targetItem['sku']
                                );
                                if ($product['success']) {
                                    $productData = $product['data'];
                                    if ($productData['sku'] == $sourceItem['sku']) {
                                        if (!isset($sourceItem['marketplace_shipment_id'])) {
                                            if (!in_array($sourceItem['marketplace_item_id'], $usedItemCode)) {
                                                $matchedItem = $sourceItem;
                                                $usedItemCode[] = $sourceItem['marketplace_item_id'];
                                                break;
                                            }
                                        } else {
                                            if ($data['marketplace_shipment_id'] == $sourceItem['marketplace_shipment_id']) {
                                                if (!in_array($sourceItem['marketplace_item_id'], $usedItemCode)) {
                                                    $matchedItem = $sourceItem;
                                                    $usedItemCode[] = $sourceItem['marketplace_item_id'];
                                                    break;
                                                }
                                            }
                                        }
                                    } else {
                                        $message = "Invalid Target Mapped SKU: ";
                                    }
                                }
                            }
                        }
                        if (!empty($matchedItem)) {
                            $temp = [
                                "type" => $matchedItem['type'],
                                "title" => $matchedItem['title'],
                                "sku" => $matchedItem['sku'],
                                "qty" => $matchedItem['qty'],
                                "weight" => $targetItem['weight'],
                                "shipped_qty" => $targetItem['shipped_qty'],
                                "object_type" => "source_shipment_items",
                                "user_id" => $data['user_id'],
                                "cif_order_id" => $sourceData['cif_order_id'],
                                "marketplace_item_id" => $matchedItem['marketplace_item_id'] ?? "",
                                'item_status' => $targetItem['shipped_qty'] == $matchedItem['qty']
                                    ? "Shipped" : "Partially Shipped",
                            ];
                            array_push($items, $temp);
                            $filter[] =
                                [
                                    'updateOne' => [
                                        [
                                            'user_id' => $data['user_id'],
                                            'object_type' => 'source_order',
                                            'marketplace_reference_id' => $sourceData['marketplace_reference_id'],
                                            'items.marketplace_item_id' => $matchedItem['marketplace_item_id']
                                        ],
                                        [
                                            '$set' => ['items.$.marketplace_shipment_id' => (string)$data['marketplace_shipment_id']]
                                        ],
                                    ],
                                ];
                        } else {
                            $errorSku[] = $targetItem['sku'];
                        }
                    }
                    if (!empty($filter)) {
                        $orderCollection->bulkWrite($filter);
                    }
                } else {
                    foreach ($data['items'] as $targetItem) {
                        $matchedItem = [];
                        foreach ($sourceData['items'] as $sourceItem) {
                            if (
                                $sourceItem['sku'] == $targetItem['sku']
                            ) {
                                $matchedItem = $sourceItem;
                                break;
                            } elseif (isset($targetItem['product_identifier']) && !empty($targetItem['product_identifier'])) {
                                $query = [
                                    'user_id' => $sourceData['user_id'],
                                    'target_marketplace' => $sourceData['marketplace'],
                                    'shop_id' => (string)$sourceData['marketplace_shop_id'],
                                    'source_product_id' => (string)$targetItem['product_identifier']
                                ];
                                $arrayParams = ["typeMap" => ['root' => 'array', 'document' => 'array']];
                                $sourceProduct = $productByQueryObj->getProductByQuery($query, $arrayParams);
                                if (isset($sourceProduct) && !empty($sourceProduct)) {
                                    if (!isset($sourceProduct[0]['sku'])) {
                                        $query = [
                                            'user_id' => $data['user_id'],
                                            'source_marketplace' => $data['target_marketplace'],
                                            'shop_id' => (string)$data['target_shop_id'],
                                            'source_product_id' => (string)$targetItem['product_identifier']
                                        ];
                                        $targetProduct = $productByQueryObj->getProductByQuery($query, $arrayParams);
                                        if (isset($targetProduct) && !empty($targetProduct)) {
                                            $sourceSku = $targetProduct[0]['sku'] ?? '';
                                        }
                                    } else {
                                        $sourceSku = $sourceProduct[0]['sku'] ?? '';
                                    }
                                    if ($sourceItem['sku'] == $sourceSku) {
                                        $matchedItem = $sourceItem;
                                        break;
                                    } else {
                                        $message = "Invalid Target Mapped SKU: ";
                                    }
                                }
                            } else {
                                $product = $this->getProductSku(
                                    $targetShipment['shop_id'],
                                    $sourceData['marketplace_shop_id'],
                                    $targetItem['sku']
                                );
                                if ($product['success']) {
                                    $productData = $product['data'];
                                    if ($productData['sku'] == $sourceItem['sku']) {
                                        $matchedItem = $sourceItem;
                                        break;
                                    } else {
                                        $message = "Invalid Target Mapped SKU: ";
                                    }
                                }
                            }
                        }

                        if (!empty($matchedItem)) {
                            $temp = [
                                "type" => $matchedItem['type'],
                                "title" => $matchedItem['title'],
                                "sku" => $matchedItem['sku'],
                                "qty" => $matchedItem['qty'],
                                "weight" => $targetItem['weight'],
                                "shipped_qty" => $targetItem['shipped_qty'],
                                "object_type" => "source_shipment_items",
                                "user_id" => $data['user_id'],
                                "cif_order_id" => $sourceData['cif_order_id'],
                                "marketplace_item_id" => $matchedItem['marketplace_item_id'] ?? "",
                                'item_status' => $targetItem['shipped_qty'] == $matchedItem['qty']
                                    ? "Shipped" : "Partially Shipped",
                            ];
                            array_push($items, $temp);
                        } else {
                            $errorSku[] = $targetItem['sku'];
                        }
                    }
                }
                if (!empty($errorSku)) {

                    $response = $this->checkInvalidSku($errorSku, $sourceParams, $message);
                    return $response;
                }
                if (isset($sourceData['shipment_error'])) {
                    $orderCollection->updateOne($sourceParams, ['$set' => ["shipping_status" => "In Progress"]]);
                }
                $prepareData = [
                    'object_type' => 'source_shipment',
                    'marketplace' => $data['source_marketplace'],
                    'shop_id' => $data['source_shop_id'],
                    'targets_marketplace' => $data['target_marketplace'],
                    'targets_status' => $targetShipment['status'],
                    'targets_shop_id' => $data['target_shop_id'],
                    'targets_order_id' => $targetShipment['marketplace_reference_id'],
                    'marketplace_reference_id' => $sourceData['marketplace_reference_id'],
                    'cif_order_id' => $sourceData['cif_order_id'],
                    'data' => $data,
                    'items' => $items
                ];
                $sourceShipment = $this->prepareShipmentData($prepareData);
                if ($toBeUpdated) {
                    $sourceShipmentCopy = $sourceShipment;
                    unset($sourceShipmentCopy['items']);
                    unset($sourceShipmentCopy['created_at']);
                    $orderCollection->updateOne($shipmentParams, ['$set' => $sourceShipmentCopy]);
                } else {
                    $itemIds = [];
                    $saveSourceShipment = $orderCollection->insertOne($sourceShipment);
                    $id = $saveSourceShipment->getInsertedId();

                    foreach ($items as $key => $sourceShipmentItem) {
                        $items[$key]['shipment_id'] = (string) $id;
                        $saveShipment = $orderCollection->insertOne($items[$key]);
                        array_push($itemIds, (string) $saveShipment->getInsertedId());
                    }
                    $orderCollection->updateOne(
                        ['_id' => $id],
                        ['$set' => ['items' => $items, 'shipment_id' => (string) $id, "source_shipment_id" => $itemIds]]
                    );
                    $sourceShipment['items'] = $items;
                    $getTargetShipment = $orderCollection->findOne(['object_type' => "target_shipment", 'user_id' => $targetShipment['user_id'], 'marketplace_shipment_id' => (string)$data['marketplace_shipment_id']], $arrayParams);
                    $this->addKey("source_shipment", $sourceData['marketplace_reference_id'], $getTargetShipment);
                }
                return ['success' => true, 'data' => $sourceShipment];
            }
        }
        return ['success' => false, 'message' => 'Unable to Prepare Shipment'];
    }

    /**
     * setSourceStatus function
     * For Syncing Source or Target Status
     * @param [string] $orderId
     * @param [string] $marketplace
     * @param boolean $success
     * @param string $reason
     * @param string $userId
     * @param boolean $sendMail
     * @return array
     */
    public function setSourceStatus($orderId, $marketplace, $success = false, $reason = 'Unknown', $userId = '', $sendMail = false)
    {
        if ((!empty($orderId)) && (!empty($marketplace))) {
            $mongo = $this->di->getObjectManager()->create('\App\Core\Models\BaseMongo');
            $orderCollection = $mongo->getCollectionForTable('order_container');
            $shipmentMailCollection = $mongo->getCollectionForTable('failed_shipment');
            $arrayParams = ["typeMap" => ['root' => 'array', 'document' => 'array']];
            $object = $this->di->getObjectManager()->get("App\\Connector\\Components\\Order\\OrderStatus");
            if (empty($userId)) {
                $userId = $this->di->getUser()->id;
            }
            $params = [
                'user_id' => $userId,
                '$or' => [['object_type' => 'source_order'], ['object_type' => 'target_order']],
                'marketplace_reference_id' => $orderId
            ];
            $orderData = $orderCollection->findOne($params, $arrayParams);
            $objectType = $orderData['object_type'];
            $shippingStatus = '';
            if (!empty($orderData)) {
                $userId = empty($userId) ? $orderData['user_id'] : $userId;
                if ($success) {
                    if ($objectType == "source_order") {
                        $fetchParams = [
                            'user_id' => $userId,
                            'object_type' => 'target_order',
                            'targets.order_id' => $orderId
                        ];
                        $sourceProductMarketplace = "source_marketplace";
                        $targetProductMarketplace = "target_marketplace";
                    } else {
                        $fetchParams = [
                            'user_id' => $userId,
                            'object_type' => 'source_order',
                            'targets.order_id' => $orderId
                        ];
                        $sourceProductMarketplace = "target_marketplace";
                        $targetProductMarketplace = "source_marketplace";
                    }
                    $getData = $orderCollection->findOne($fetchParams, $arrayParams);
                    if (!empty($getData)) {
                        $status = $getData['shipping_status'];
                    }
                    if (isset($orderData['shipment_error'])) {
                        if ($orderData['shipping_status'] != 'Unshipped' && $orderData['shipping_status'] != 'Failed') {
                            $filter[] =
                                [
                                    'updateOne' => [
                                        [
                                            'user_id' => $userId,
                                            'object_type' => $objectType,
                                            'marketplace_reference_id' => $orderId,
                                        ],
                                        ['$unset' => ["shipment_error" => true]],
                                    ],
                                ];
                        } else {
                            return [
                                "success" => false,
                                "message" => "Shipment Sync Failed",
                            ];
                        }
                    }
                    $response = $object->validateStatus($orderData['status'], $status);
                    if ($response) {
                        $filter[] =
                            [
                                'updateOne' => [
                                    [
                                        'user_id' => $userId,
                                        'object_type' => $objectType,
                                        'marketplace_reference_id' => $orderId,
                                    ],
                                    ['$set' => ["status" => $status, 'shipping_status' => $status]],
                                ],
                            ];
                    } else {
                        $filter[] = [
                            'updateOne' => [
                                $params,
                                ['$set' => ['shipping_status' => $status]],
                            ],
                        ];
                    }

                    $productByQueryObj = $this->di->getObjectManager()
                        ->create(\App\Connector\Models\Product\Marketplace::class);
                    $usedItemCode = [];
                    foreach ($getData['items'] as $getItem) {
                        $matchedItem = false;
                        if (isset($getItem['shipped_qty']) && !empty($getItem['shipped_qty'])) {
                            $shippedQty = $getItem['shipped_qty'];
                            $itemStatus = $getItem['shipped_qty'] == $getItem['qty'] ? 'Shipped' : 'Partially Shipped';
                            foreach ($orderData['items'] as $orderItem) {

                                if (
                                    $orderItem['sku'] == $getItem['sku']
                                ) {
                                    if (!in_array($orderItem['marketplace_item_id'], $usedItemCode)) {
                                        $matchedItem = true;
                                        $usedItemCode[] = $orderItem['marketplace_item_id'];
                                        break;
                                    }
                                } elseif (
                                    isset($getItem['product_identifier'])
                                    && !empty($getItem['product_identifier'])
                                ) {
                                    $query = [
                                        'user_id' => $orderData['user_id'],
                                        $targetProductMarketplace => $orderData['marketplace'],
                                        'shop_id' => (string)$orderData['marketplace_shop_id'],
                                        'source_product_id' => (string)$getItem['product_identifier']
                                    ];
                                    $arrayParams = ["typeMap" => ['root' => 'array', 'document' => 'array']];
                                    $sourceProduct = $productByQueryObj->getProductByQuery($query, $arrayParams);
                                    if (isset($sourceProduct) && !empty($sourceProduct)) {
                                        if (!isset($sourceProduct[0]['sku'])) {
                                            $query = [
                                                'user_id' => $getData['user_id'],
                                                $sourceProductMarketplace => $getData['marketplace'],
                                                'shop_id' => (string)$getData['marketplace_shop_id'],
                                                'source_product_id' => (string)$getItem['product_identifier']
                                            ];
                                            $targetProduct = $productByQueryObj
                                                ->getProductByQuery($query, $arrayParams);
                                            if (isset($targetProduct) && !empty($targetProduct)) {
                                                $sourceSku = $targetProduct[0]['sku'] ?? '';
                                            }
                                        } else {
                                            $sourceSku = $sourceProduct[0]['sku'] ?? '';
                                        }
                                        if ($orderItem['sku'] == $sourceSku) {
                                            if (!in_array($orderItem['marketplace_item_id'], $usedItemCode)) {
                                                $matchedItem = true;
                                                $usedItemCode[] = $orderItem['marketplace_item_id'];
                                                break;
                                            }
                                        }
                                    }
                                } else {
                                    $product = $this->getProductSku(
                                        $getData['shop_id'],
                                        $orderData['marketplace_shop_id'],
                                        $getItem['sku']
                                    );
                                    if ($product['success']) {
                                        $productData = $product['data'];
                                        if ($productData['sku'] == $orderItem['sku']) {
                                            if (!in_array($orderItem['marketplace_item_id'], $usedItemCode)) {
                                                $matchedItem = true;
                                                $usedItemCode[] = $orderItem['marketplace_item_id'];
                                                break;
                                            }
                                        }
                                    }
                                }
                            }
                            if ($matchedItem) {
                                $filter[] =
                                    [
                                        'updateOne' => [
                                            [
                                                'user_id' => $userId,
                                                'object_type' => $objectType,
                                                'cif_order_id' => $orderItem['cif_order_id'],
                                                'items.marketplace_item_id' => (string)$orderItem['marketplace_item_id'],
                                            ],
                                            ['$set' => ["items.$.shipped_qty" => $shippedQty]],
                                        ],
                                    ];
                                $filter[] =
                                    [
                                        'updateOne' => [
                                            [
                                                'user_id' => $userId,
                                                'object_type' => $objectType . "_items",
                                                'cif_order_id' => $orderItem['cif_order_id'],
                                                'marketplace_item_id' => (string)$orderItem['marketplace_item_id'],
                                            ],
                                            ['$set' => ["shipped_qty" => $shippedQty, "item_status" => $itemStatus]],
                                        ],
                                    ];
                            }
                        }
                    }
                    foreach ($orderData['targets'] as $sourceTarget) {
                        if ($sourceTarget['marketplace'] == $marketplace) {
                            $filter[] =
                                [
                                    'updateOne' => [
                                        [
                                            'user_id' => $userId,
                                            'object_type' => $getData['object_type'],
                                            'marketplace_reference_id' => $getData['marketplace_reference_id'],
                                            "shop_id" => $getData['shop_id'],
                                            "targets.marketplace" => $orderData['marketplace'],
                                        ],
                                        ['$set' => ["targets.$.status" => $status]],
                                    ],
                                ];

                            if (!empty($filter)) {
                                $orderCollection->bulkWrite($filter);
                            }
                            return [
                                "success" => true,
                                "message" => "Status Changed Successfully!!",
                                'shipping_status' => $status,
                            ];
                        }
                    }
                } else {
                    if ($sendMail && !isset($orderData['shipment_mail'])) {
                        $prepareData = [
                            'source_order_id' => $orderData['marketplace_reference_id'],
                            'source_marketplace' => $orderData['marketplace'],
                            'target_order_id' => $orderData['targets'][0]['order_id'],
                            'target_marketplace' => $orderData['targets'][0]['marketplace'],
                            'user_id' => $userId,
                            'reason' => $reason,
                        ];
                        $shipmentMailCollection->insertOne($prepareData);
                        $setValue['shipment_mail'] = true;
                    }
                    $shippingStatus = "Unshipped";
                    $setValue['shipping_status'] = $shippingStatus;
                    $orderCollection->updateOne(
                        $params,
                        [
                            '$set' => $setValue,
                            '$addToSet' => ['shipment_error' => $reason]
                        ]
                    );
                    return [
                        "success" => false,
                        "message" => "Shipment Unsuccessful",
                        'shipping_status' => $shippingStatus
                    ];
                }
            }
        }
    }

    /**
     * prepareShipmentData function
     * Preparing Source or Target Shipment Data
     * @param [array] $prepareData
     * @return array
     */
    public function prepareShipmentData($prepareData)
    {
        $status = "In Progress";
        if (isset($prepareData['totalQty']) && isset($prepareData['shippedQty'])) {
            $status = $prepareData['totalQty'] == $prepareData['shippedQty']
                ? "Shipped" : "Partially Shipped";
        }
        return [
            "schema_version" => "2.0",
            "object_type" => $prepareData['object_type'],
            "cif_order_id" => $prepareData['cif_order_id'],
            "shop_id" => $prepareData['shop_id'],
            "marketplace" => $prepareData['marketplace'],
            "marketplace_shop_id" => $prepareData['shop_id'],
            "marketplace_reference_id" => $prepareData['marketplace_reference_id'],
            "marketplace_shipment_id" => (string) $prepareData['data']['marketplace_shipment_id'],
            "user_id" => $prepareData['data']['user_id'],
            "status" => $status,
            "targets" => [
                [
                    "marketplace" => $prepareData['targets_marketplace'],
                    "order_id" => $prepareData['targets_order_id'],
                    "status" => $prepareData['targets_status'],
                    "shop_id" => $prepareData['targets_shop_id'],

                ],
            ],
            "items" => $prepareData['items'],
            "shipping_status" => $prepareData['data']['shipment_status']
                ?? $prepareData['data']['shipping_status'] ?? $status,
            "shipping_method" => [
                "code" => $prepareData['data']['shipping_method']['code'] ?? $prepareData['data']['tracking']['company'],
                "title" => $prepareData['data']['shipping_method']['title'] ?? $prepareData['data']['tracking']['company'],
                "carrier" => $prepareData['data']['shipping_method']['carrier'] ?? $prepareData['data']['tracking']['company'],
            ],
            "tracking" => [
                "company" => $prepareData['data']['shipping_method']['title'] ?? $prepareData['data']['tracking']['company'],
                "number" => $prepareData['data']['tracking']['number'] ?? '',
                "name" => $prepareData['data']['tracking']['name'] ?? '',
                "url" => $prepareData['data']['tracking']['url'] ?? '',
            ],
            "created_at" => date('c'),
            "updated_at" => date('c'),
            "source_created_at" => $prepareData['data']['shipment_created_at'],
            "source_updated_at" => $prepareData['data']['shipment_updated_at'],
            "attributes" => $prepareData['data']['attributes'] ?? '',

        ];
    }

    /**
     * getProductSku function
     * For mapping different SKU's
     * @param [string] $sourceShopId
     * @param [string] $targetShopId
     * @param [string] $sku
     * @return array
     */
    public function getProductSku($sourceShopId, $targetShopId, $sku)
    {
        $productBySkuObj = $this->di->getObjectManager()->create(\App\Connector\Models\Product\Marketplace::class);
        $skudata = [
            "sku" => $sku,
            "source_shop_id" => $sourceShopId,
            "target_shop_id" => $targetShopId,
        ];

        $product = $productBySkuObj->getProductBySku($skudata);
        if (empty($product)) {
            return [
                'success' => false,
                'message' => 'Product not found',
            ];
        } else {
            return [
                'success' => true,
                'data' => $product,
            ];
        }
    }

    /**
     * prepareItems function
     * For preparing Source or Target Shipment Items
     * @param [array] $data
     * @param [array] $sourceOrTargetData
     * @param [string] $toBeUpdated
     * @return array
     */
    public function prepareItems($data, $sourceOrTargetData, $toBeUpdated)
    {
        $objectType = 'source_shipment_items';
        $marketplace = 'target_marketplace';
        if ($this->objectType != "source_order") {
            $objectType = 'target_shipment_items';
            $marketplace = 'source_marketplace';
        }
        $items = [];
        $totalQuantity = 0;
        $quantityShipped = 0;
        $orderItems = [];
        $savedShippedQuantity = 0;
        foreach ($sourceOrTargetData['items'] as $key => $value) {
            $orderItems[(string) $value['marketplace_item_id']] = $value;
            $totalQuantity = $totalQuantity + $value['qty'] ?? 0;
        }
        foreach ($data['items'] as $key => $value) {
            $temp = [
                "type" => $value['type'],
                "title" => $value['title'],
                "sku" => $value['sku'],
                "qty" => $orderItems[$value['marketplace_item_id']]['qty'] ?? "",
                "weight" => $value['weight'],
                "shipped_qty" => $value['shipped_qty'],
                "object_type" => $objectType,
                "user_id" => $data['user_id'],
                "cif_order_id" => $sourceOrTargetData['cif_order_id'],
                "product_identifier" => $value['product_identifier'],
                "marketplace_item_id" => $value['marketplace_item_id'],
                'item_status' => $orderItems[$value['marketplace_item_id']]['qty'] == $value['shipped_qty']
                    ? "Shipped" : "Partially Shipped",
            ];

            array_push($items, $temp);
            $quantityShipped = $quantityShipped + $value['shipped_qty'] ?? 0;
            if (isset($orderItems[$value['marketplace_item_id']]) && !$toBeUpdated) {
                $tempQty = isset($orderItems[$value['marketplace_item_id']]['shipped_qty'])
                    ? (int) $orderItems[$value['marketplace_item_id']]['shipped_qty'] : 0;
                $orderItems[$value['marketplace_item_id']]['shipped_qty'] = $tempQty + (int) $temp['shipped_qty'];
            }
        }
        foreach ($orderItems as $key => $value) {
            if (isset($value['shipped_qty'])) {
                $savedShippedQuantity = $savedShippedQuantity + $value['shipped_qty'];
            }
        }

        $orderId = '';
        foreach ($sourceOrTargetData['targets'] as $value) {
            if ($value['marketplace'] == $data[$marketplace]) {
                $orderId = $value['order_id'];
            }
        }

        $prepareData = [
            'data' => $data, 'orderData' => $sourceOrTargetData,
            'totalQuantity' => $totalQuantity, 'quantityShipped' => $quantityShipped,
            'savedShippedQuantity' => $savedShippedQuantity, 'orderItems' => $orderItems, 'items' => $items, 'orderId' => $orderId
        ];

        return $prepareData;
    }

    /**
     * updateDocs function
     * For updating Source and Target Docs
     * @param [array] $data
     * @return array
     */
    public function updateDocs($data)
    {
        $orderCollection = $this->mongo->getCollectionForTable('order_container');
        $object = $this->di->getObjectManager()->get("App\\Connector\\Components\\Order\\OrderStatus");
        if (!isset($data['status'])) {
            $filter[] =
                [
                    'updateOne' => [
                        $data['targetParams'],
                        ['$set' => [
                            'items' => array_values($data['orderItems']),
                            'status' => $data['parentStatus'],
                            'shipping_status' => $data['parentStatus']
                        ]],
                    ]
                ];
            $filter[] =
                [
                    'updateOne' => [
                        $data['sourceParams'],
                        ['$set' => [
                            'targets.$.status' => $data['parentStatus'],
                            'shipping_status' => 'In Progress'
                        ]],
                    ]
                ];
        } else {
            $response = $object->validateStatus($data['status'], $data['parentStatus']);
            if ($response) {
                $filter[] =
                    [
                        'updateOne' => [
                            $data['targetParams'],
                            ['$set' => [
                                'items' => array_values($data['orderItems']),
                                'status' => $data['parentStatus'], 'shipping_status' => $data['parentStatus']
                            ]],
                        ]
                    ];
                $filter[] =
                    [
                        'updateOne' => [
                            $data['sourceParams'],
                            ['$set' => [
                                "targets.$.status" => $data['parentStatus'],
                                'shipping_status' => 'In Progress'
                            ]],
                        ]
                    ];
            } else {
                $filter[] =
                    [
                        'updateOne' => [
                            $data['targetParams'],
                            ['$set' => [
                                'items' => array_values($data['orderItems']),
                                'shipping_status' => $data['parentStatus']
                            ]],
                        ]
                    ];
                $filter[] =
                    [
                        'updateOne' => [
                            $data['sourceParams'],
                            ['$set' => ['shipping_status' => 'In Progress']],
                        ]
                    ];
            }
        }
        if (!empty($filter)) {
            $orderCollection->bulkWrite($filter);
        }
    }

    /**
     * checkInvalidSku function
     * For checking Invalid Mapped Sku Error
     * @param [array] $errorSku
     * @param [array] $params
     * @param [string] $message
     * @return array
     */
    public function checkInvalidSku($errorSku, $params, $message)
    {
        $orderCollection = $this->mongo->getCollectionForTable('order_container');
        foreach ($errorSku as $sku) {
            $filter[] =
                [
                    'updateOne' => [
                        $params,
                        ['$set' => ['shipping_status' => "Failed"], '$addToSet' => [
                            'shipment_error' => $message . $sku,
                        ]],
                    ],
                ];
        }
        if (!empty($filter)) {
            $orderCollection->bulkWrite($filter);
        }
        return ['success' => false, "message" => "Invalid Mapped SKU's"];
    }

    /**
     * addKey function
     * For Creating Relation between Source and Target Shipment Doc
     * Handling DeadLock Case
     * @param [string] $objectType
     * @param [string] $orderId
     * @param [array] $data
     * @return array
     */
    public function addKey($objectType, $orderId, $data)
    {
        $orderCollection = $this->mongo->getCollectionForTable('order_container');
        $id = (string)$data['_id'];
        $params = ['object_type' => $objectType, 'user_id' => $data['user_id'], 'marketplace_reference_id' => $orderId, 'marketplace_shipment_id' => $data['marketplace_shipment_id']];
        $orderCollection->updateOne(
            $params,
            ['$set' => [
                "reference_id" => $id,
            ]]
        );

        return ['success' => true, 'message' => 'Doc Linked'];
    }

    /**
     * checkSku function
     * To check Same SKU in all line items
     * @param [array] $items
     * @return bool
     */
    public function checkSku($items)
    {
        $getSku = array_column($items, "sku");
        $count = count($getSku);
        $finalSku = array_unique($getSku);
        $uniqueCount = count($finalSku);
        if ($count != $uniqueCount) {
            return true;
        }
        return false;
    }

    /**
     * isShippable function
     * Abstract Function
     *
     * @return bool
     */
    public function isShippable()
    {
        return true;
    }

    /**
     * mold function
     * Abstract Function
     *
     * @return array
     */
    public function mold(array $data): array
    {
        return [];
    }
}