<?php

namespace App\Cedcommercewebapi\Api;

use Aws\DynamoDb\DynamoDbClient;

class EventSubscription extends Base
{
    protected $_component = 'EventSubscription';


    /**
     * Function checks whether all the required functions
     * to register a webhook is defined in marketplace module or not
     * @return array
     */
    public function init()
    {
        $ced_app_shop =  $this->di->getRegistry()->getCurrentCedAppShop();
        $appModel = $this->di->getObjectManager()->create('App\Apiconnect\Models\Apps');
        $appConfig = $appModel->getSubConfig($ced_app_shop['sAppId']);
        $targetMarketplace = $this->di->getConfig()->get('marketplace-modules')->get($appConfig['marketplace']) ?: ucfirst($appConfig['marketplace']);
        $baseClass = '\App\\' .  $targetMarketplace . '\\Api\Base';

        if (class_exists($baseClass)) {
            if (!method_exists($baseClass, 'getDestinationUniqueKeys')) return  ['success' => false, 'message' => 'Class found but getDestinationUniqueKeys method not found !'];
            if (!method_exists($baseClass, 'getSubscriptionUniqueKeys')) return  ['success' => false, 'message' => 'Class found but getSubscriptionUniqueKeys method not found !'];
        } else return  ['success' => false, 'message' => 'Base Class not found!'];

        return ['success' => true];
    }

    /**
     * @OA\Server(url="https://docs-app.cifapps.com/")
     * @OA\Get(path="/event/subscription",
     * tags={"EventSubscription"},
     * summary="Get all event subscriptions based on marketplace and type",
     * @OA\Parameter(
     *    name="marketplace", *  
     *    in="query",
     *    required=true, *    
     *    description="The marketplace passed to get in query string goes here",
     *    @OA\Schema(
     *       type="string"
     *    ),
     *    name="app_code", *  
     *    in="query",
     *    required=true, *    
     *    description="The app_code passed to get in query string goes here",
     *    @OA\Schema(
     *       type="string"
     *    ),
     *    name="type", *  
     *    in="query",
     *    required=true, *    
     *    description="The type passed to get in query string goes here",
     *    @OA\Schema(
     *       type="string"
     *    ),
     * ),
     * @OA\Response(response="200", description="Success"),
     * @OA\Response(response="404", description="Not found")
     */
    public function getEventSubscription($data)
    {
        try {
            $options = [
                "typeMap" => ['root' => 'array', 'document' => 'array']
            ];
            $model = $this->getDi()->getObjectManager()->create('\App\Core\Models\BaseMongo');
            $collection = $model->getCollection('cif_subscription');
            $filter = [];
            if (isset($data['subscription_id'])) {
                $filter = ['_id' => $data['subscription_id']];
            } elseif (isset($data['marketplace']) && isset($data['type'])) {
                $filter = ["marketplace" => $data['marketplace'], 'type' => $data['type']];
            } elseif (isset($data['shop_id'])) {
                $filter = ["remote_shop_id" => $data['shop_id']];
            } else {
                return ['success' => false, 'data' => "market place and type missing  or  subscription id missing or shop_id missing"];
            }
            $response = $collection->find($filter, $options)->toArray();
            return ['success' => true, 'data' => $response];
        } catch (\Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }


    /**
     * @OA\Post(
     *     path="/event/Subscription",
     *     summary="Create Event Subscription",
     *     description="Create Event Subscription",
     *     operationId="create Subscription",
     *     tags={"event/subscription"},
     *    * @OA\Parameter(
     *    name="shop_id", *  
     *    in="query",
     *    required=true, *    
     *    description="The remote_shop_id passed to get in query string goes here",
     *    @OA\Schema(
     *       type="string"
     *    ),
     *    name="app_code", *  
     *    in="query",
     *    required=true, *    
     *    description="The app_code passed to get in query string goes here",
     * ),
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/json",
     *             @OA\Schema( * 
     *                 @OA\Property(
     *                     property="event_code",
     *                     type="string"                          
     *                 ),
     *                 @OA\Property(
     *                     property="event_handler_id",
     *                     type="string"                         
     *                 ),
     *                 @OA\Property(
     *                     property="marketplace_data",
     *                     type="string"                         
     *                 ),
     *                 @OA\Property(
     *                     property="queue_data",
     *                     type="object" ,
     *                     @OA\Property(
     *                     property="type",
     *                     type="string"                         
     *                     ),
     *                     @OA\Property(
     *                     property="class_name",
     *                     type="string",     *                                
     *                     ),  
     *                       @OA\Property(
     *                     property="method",
     *                     type="string",     *                                
     *                     ), 
     *                       @OA\Property(
     *                     property="user_id",
     *                     type="string",     *                                
     *                     ), 
     *                      @OA\Property(
     *                     property="shop_id",
     *                     type="string",     *                                
     *                     ), 
     *                     @OA\Property(
     *                     property="action",
     *                     type="string",     *                                
     *                     ), 
     *                      @OA\Property(
     *                     property="queue_name",
     *                     type="string",     *                                
     *                     ), 
     *                      @OA\Property(
     *                     property="marketplace",
     *                     type="string",     *                                
     *                     ),                   
     *                 ),
     *                     
     *                 example={
     *                   "data": {
     *                       "event_code": "products/update",
     *                       "event_handler_id": "9",
     *                      "marketplace_data": [],
     *                       "queue_data": {
     *                       "type": "full_class",
     *                       "class_name": "AppConnectorModelsSourceModel",
     *                       "method": "triggerWebhooks",
     *                       "user_id": "62a856a0aa3c83636a4d6f74",
     *                       "shop_id": "3",
     *                       "action": "product_update",
     *                       "queue_name": "product_update",
     *                       "marketplace": "shopify"
     *                       }
     *                   }
     *                   }
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *         response=200,
     *         description="OK",
     *         @OA\JsonContent(
     *             @OA\Schema(
     *             schema="Result",
     *             title="Sample schema ",
     * 	           @OA\Property(
     *                  property="data",
     *                  type="string"
     *              ),
     *             @OA\Property(
     *                  property="success",
     *                  type="string"
     *              ),
     *             @OA\Property(
     *                  property="destination_id",
     *                  type="string"
     *              ),
     *             @OA\Property(
     *                  property="ip",
     *                  type="string"
     *              ),
     *              @OA\Property(
     *                  property="start_time",
     *                  type="string"
     *              ),
     *             @OA\Property(
     *                  property="end_time,
     *                  type="string"
     *              ),
     *             @OA\Property(
     *                  property="execution_time",
     *                  type="string"
     *              ),
     *             @OA\Examples(example="result", value={"success": true}, summary="An result object."),
     *             @OA\Examples(example="bool", value=false, summary="A boolean value."),
     *         )
     *     )
     * )
     */
    public function createEventSubscription($data)
    {
        try {
            $options = [
                "typeMap" => ['root' => 'array', 'document' => 'array']
            ];
            $model = $this->getDi()->getObjectManager()->create('\App\Core\Models\BaseMongo');
            $collection = $model->getCollection('cif_subscription');
            $event_subscription_obj = $this->di->getObjectManager()->get('\App\Cedcommercewebapi\Components\EventSubscription');
            $ced_app_shop =  $this->di->getRegistry()->getCurrentCedAppShop();
            $shop = $this->di->getRegistry()->getCurrentShop();
            $data = $data['data'];
            if (isset($data['event_code'], $data['event_handler_id'])) {
                $event_destination_collection = $model->getCollection('cif_destination');
                $event_destination = $event_destination_collection->findOne(['_id' => $data['event_handler_id']], $options);
                if (!empty($event_destination)) {
                    if ($event_destination['type'] != 'app') {
                        $res = $this->init();
                        if (!$res['success']) return $res;
                    }

                    #TODO: creating queue 
                    if (isset($event_destination['destination_data']) && !empty($event_destination['destination_data']) && isset($event_destination['destination_data']['type']) && $event_destination['destination_data']['type'] == 'sqs') {
                        $sqsData = $event_destination['destination_data']['sqs'];
                        $sqs = $this->di->getObjectManager()->get('App\Core\Components\Sqs');
                        $sqsClient = $sqs->getClient($sqsData['region'], $sqsData['key'], $sqsData['secret']);
                        if (isset($data['queue_data']['queue_name']))
                        $queueResponse = $sqs->createQueue($data['queue_data']['queue_name'], $sqsClient);
                        if (!$queueResponse['success']) return ['success' => false, 'message' => 'Trouble creating queue !'];

                    }

                    $isDuplicate = $this->checkSubscriptionDuplicacy($event_destination, $shop, $ced_app_shop, $data);
                    if (isset($isDuplicate['duplicate']) && $isDuplicate['duplicate']) return ['success' => true, 'config_save_result' => ['id' => $isDuplicate['subscription']['_id']], 'data' => "Event subscription already added for this event_code"];
                    $marketplace_destination_id = null;
                    // if ($event_destination['type'] != 'app'){
                        $marketplace_destination_res = $event_subscription_obj->createDestinationOnMarketplace($data);
                        $this->di->getLog()->logContent('logrrrrrrr = ' . print_r($marketplace_destination_res, true), 'info', 'apptype.log');
                        if(isset($marketplace_destination_res['success']) && !$marketplace_destination_res['success']) return $marketplace_destination_res;
                    // }
                    if (isset($marketplace_destination_res['success']) && $marketplace_destination_res['success']) {
                        $data['type'] = $event_destination['type'];
                        $marketplace_destination_id = $marketplace_destination_res['destination_id'];
                        $marketplace_subscription_res = $event_subscription_obj->createSubscriptionOnMarketplace($data, $marketplace_destination_id);
                        if(isset($marketplace_subscription_res['success']) && !$marketplace_subscription_res['success']) return $marketplace_subscription_res;  
                    }
                    $insertData = [
                        '_id' => $model->getCounter('cif_subscription_id'),
                        'marketplace' => $appConfig['marketplace'] ?? $data['queue_data']['marketplace'],
                        'remote_shop_id' => $shop['_id'] ?? null,
                        'subscription_id' => $marketplace_subscription_res['subscription_id'] ? (string)$marketplace_subscription_res['subscription_id'] : null,
                        'event_code' => $data['event_code'],
                        'destination_id' => $data['event_handler_id'],
                        'type' => $event_destination['type'],
                        'marketplace_data' => $data['marketplace_data'] ?? [],
                        'queue_data' => $data['queue_data'] ?? [],
                    ];
                    if ($event_destination['type'] == 'user_app') $insertData['app_code'] = $data['queue_data']['app_code'];
                    $response = $collection->insertOne($insertData);
                    if ($response->getInsertedCount()) {
                        $data['marketplace_destination_id'] = $marketplace_destination_id;
                        $data['marketplace_subscription_id'] = $insertData['subscription_id'];
                        $data['destination_id'] = $insertData['destination_id'];
                        $data['subscription_id'] = $insertData['_id'];
                        $data['event_code'] = $insertData['event_code'];
                        $data['marketplace'] = $insertData['marketplace'];
                        $data['type'] = $insertData['type'];
                        $event_subscription_obj->insertSubscriptionIntoDynamoDb($data);
                        return ['success' => true, 'data' => "Event subscription added successfully", 'config_save_result' => ['id' => $response->getInsertedId()]];
                    }
                } else {
                    return ['success' => false, 'data' => "Wrong event_handler_id"];
                }
            } else {
                return ['success' => false, 'data' => "event_code and event_handler_id required params"];
            }
        } catch (\Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    public function updateEventSubscription($data)
    {
        try {
            $options = [
                "typeMap" => ['root' => 'array', 'document' => 'array']
            ];
            $model = $this->getDi()->getObjectManager()->create('\App\Core\Models\BaseMongo');
            $collection = $model->getCollection('cif_subscription');
            $event_subscription_obj = $this->di->getObjectManager()->get('\App\Cedcommercewebapi\Components\EventSubscription');
            if (isset($data['subscription_id'])) {
                $dataToUpdate = [];
                if (isset($data['data']['queue_data'])) {
                    $dataToUpdate['queue_data'] = $data['data']['queue_data'];
                }
                $cifSubscription = $collection->findOne(['_id' => $data['subscription_id']], $options);
                $event_destination_collection = $model->getCollection('cif_destination');
                $event_destination = $event_destination_collection->findOne(['_id' => $cifSubscription['destination_id']], $options);

                if ($event_destination['type'] != 'app') {
                    $marketplace_destination_res = $event_subscription_obj->createDestinationOnMarketplace($cifSubscription);
                    $marketplace_destination_id = $marketplace_destination_res['destination_id'];
                    if (isset($marketplace_destination_res['success']) && $marketplace_destination_res['success']) {
                        $marketplace_subscription_res = $event_subscription_obj->createSubscriptionOnMarketplace($cifSubscription, $marketplace_destination_id);
                        if (isset($marketplace_subscription_res['activated']) && $marketplace_subscription_res['activated']) {
                            $dataToUpdate['subscription_id'] = (string)$marketplace_subscription_res['subscription_id'];
                        }
                    } else {
                        return ['success' => false, 'message' => 'Trouble in updating subscription. Error in marketplace_destination_res'];
                    }
                }
                if (!empty($dataToUpdate)) {
                    $updateEventSubscription = $collection->updateOne(
                        ['_id' => (string)$data['subscription_id']],
                        ['$set' => $dataToUpdate]
                    );
                    if ($updateEventSubscription->getModifiedCount() == 1) {
                        $event_subscription_obj = $this->di->getObjectManager()->get('\App\Cedcommercewebapi\Components\EventSubscription');
                        $event_subscription_obj->updateSubscriptionInDynamodb($data['subscription_id']);
                        return ['success' => true, 'data' => "Event subscription updated successfully"];
                    } else {
                        return ['success' => false, 'data' => "Event subscription not updated"];
                    }
                } else {
                    return ['success' => false, 'data' => "Nothing to update in subscription !!"];
                }
            } else {
                return ['success' => false, 'data' => "subscription_id required"];
            }
        } catch (\Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }



    /**
     * @OA\Server(url="https://docs-app.cifapps.com/")
     * @OA\Delete(path="/event/subscription",
     * tags={"EventSubscription"},
     * summary="Delete event destination",
     * @OA\Parameter(
     *    name="app_code", *  
     *    in="query",
     *    required=true, *    
     *    description="The app_code passed to get in query string goes here",
     *    @OA\Schema(
     *       type="string"
     *    ),
     *    name="subscription_id", *  
     *    in="query",
     *    required=true, *    
     *    description="The event subscription id passed to get in query string goes here",
     *    @OA\Schema(
     *       type="string"
     *    ),
     *   name="shop_id", *  
     *    in="query",
     *    required=true, *    
     *    description="The remote shop id passed to get in query string goes here",
     *    @OA\Schema(
     *       type="string"
     *    ),
     * ),
     * @OA\Response(response="200", message=Event destination id deleted successfully."),
     * @OA\Response(response="404", description="Not found")
     */
    public function deleteEventSubscription($data)
    {
        try {
            $shop = $this->di->getRegistry()->getCurrentShop();
            $options = ["typeMap" => ['root' => 'array', 'document' => 'array']];
            $model = $this->getDi()->getObjectManager()->create('\App\Core\Models\BaseMongo');
            $cif_collection = $model->getCollection('cif_subscription');
            $event_subscription_obj = $this->di->getObjectManager()->get('\App\Cedcommercewebapi\Components\EventSubscription');

            if (isset($data['subscription_id'])) {
                $cif_subscription = $cif_collection->findOne(['_id' => $data['subscription_id']], $options);
                if (!empty($cif_subscription)) {
                    $marketkplace_collection = $model->getCollection('marketplace_subscription');
                    $marketplace_subscription = $marketkplace_collection->findOne(['marketplace_subscription_id' => $cif_subscription['subscription_id']], $options);
                    $subscriptionData = [
                        'marketplace_destination_id' => $marketplace_subscription['marketplace_destination_id'] ?? null,
                        'marketplace_subscription_id' => $marketplace_subscription['marketplace_subscription_id'] ?? null,
                        'subscription_id' => $cif_subscription['_id'],
                        'event_code' => $cif_subscription['event_code'],
                        'type' => $cif_subscription['type'],
                        'marketplace' => $cif_subscription['marketplace'],
                    ];
                    if ($cif_subscription['type'] == 'user_app') $subscriptionData['app_code'] = $cif_subscription['app_code'];

                    if (isset($cif_subscription['subscription_id']) && $cif_subscription['subscription_id'] != null) {
                        if (isset($data['app_uninstalled']) && $data['app_uninstalled']) {
                            $marketkplace_collection = $model->getCollection('marketplace_subscription');
                            $marketkplace_collection->deleteOne(
                                ['marketplace_subscription_id' => $cif_subscription['subscription_id']]
                            );
                        } else {
                            $params = [
                                'shop_id' => $shop['_id'],
                                'subscription_id' => $cif_subscription['subscription_id']
                            ];
                            if ($cif_subscription['type'] == 'user_app') $params['app_code'] = $cif_subscription['app_code'];
                            $isAlreadyInUse =  $this->isMarketplaceSubscriptionAlreadyInUse($cif_subscription['subscription_id']);

                            if (!$isAlreadyInUse)
                                $marketplaceSubscription = $this->marketplacesCall('delete/subscription', $params, 'DELETE');
                        }
                        if ((isset($marketplaceSubscription['success']) && $marketplaceSubscription['success']) || (isset($data['app_uninstalled']) && $data['app_uninstalled']) || $isAlreadyInUse) {
                            $deleteCifSubscription = $cif_collection->deleteOne(
                                ['_id' => $cif_subscription['_id']]
                            );
                            if ($deleteCifSubscription->getDeletedCount() > 0) {
                                $event_subscription_obj->deleteSubscriptionFromDynamodb($subscriptionData);
                                return ['success' => true, 'message' => "Event Subscription deleted successfully"];
                            } else {
                                return ['success' => true, 'message' => "Event Subscription not deleted"];
                            }
                        } else {
                            return ['success' => false, 'message' => "Something went wrong in deleting marketplace subscription"];
                        }
                    } else {
                        $deleteCifSubscription = $cif_collection->deleteOne(
                            ['_id' => $cif_subscription['_id']]
                        );
                        if ($deleteCifSubscription->getDeletedCount() > 0) {
                            $event_subscription_obj->deleteSubscriptionFromDynamodb($subscriptionData);
                            return ['success' => true, 'message' => "Event Subscription deleted successfully"];
                        } else {
                            return ['success' => true, 'message' => "Event Subscription not deleted"];
                        }
                    }
                } else {
                    return ['success' => false, 'data' => "Wrong subscription_id"];
                }
            } else {
                return ['success' => false, 'data' => "subscription_id required params"];
            }
        } catch (\Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    public function checkSubscriptionDuplicacy($event_destination, $shop, $ced_app_shop, $data)
    {
        $options = [
            "typeMap" => ['root' => 'array', 'document' => 'array']
        ];
        $model = $this->getDi()->getObjectManager()->create('\App\Core\Models\BaseMongo');
        $collection = $model->getCollection('cif_subscription');
        if ($event_destination['type'] != 'app') {
            $appModel = $this->di->getObjectManager()->create('App\Apiconnect\Models\Apps');
            $appConfig = $appModel->getSubConfig($ced_app_shop['sAppId']);
        }
        $filter = [
            'marketplace' => $appConfig['marketplace'] ?? $data['queue_data']['marketplace'],
            'remote_shop_id' => $shop['_id'] ?? null,
            'event_code' => $data['event_code'],
            'destination_id' => $data['event_handler_id'],
            'type' => $event_destination['type'],
            'marketplace_data' => $data['marketplace_data'] ?? [],
            'queue_data' => $data['queue_data'] ?? [],
        ];
        if ($event_destination['type'] == 'user_app') {
            $filter['app_code'] = $data['queue_data']['app_code'];
        }
        $subscription = $collection->findOne($filter, $options);
        return empty($subscription) ? ['duplicate' => false] : ['duplicate' => true, 'subscription' => $subscription];
    }

    public function isMarketplaceSubscriptionAlreadyInUse($subscriptionId)
    {
        $model = $this->getDi()->getObjectManager()->create('\App\Core\Models\BaseMongo');
        $cif_collection = $model->getCollection('cif_subscription');
        $options = ['typeMap'   => ['root' => 'array', 'document' => 'array']];
        $aggregation = [['$match' => ['subscription_id' => $subscriptionId]], ['$count' => 'count']];
        $subscriptionCount = $cif_collection->aggregate(
            $aggregation,
            $options
        )->toArray();

        if (isset($subscriptionCount, $subscriptionCount[0]['count']) && $subscriptionCount[0]['count'] > 1) return true;
        return false;
    }

    public function deleteSubscriptionInBulk($data)
    {
        $shop = $this->di->getRegistry()->getCurrentShop();
        $model = $this->getDi()->getObjectManager()->create('\App\Core\Models\BaseMongo');
        $cifCollection = $model->getCollection('cif_subscription');
        $marketplaceCollection = $model->getCollection('marketplace_subscription');
        if (!empty($data['subscription_ids']) && !empty($shop)) {
            $subscriptionIds = $data['subscription_ids'];
            $response = $this->getEventSubscription(['shop_id' => $shop['_id']]);
            $ced_app_shop =  $this->di->getRegistry()->getCurrentCedAppShop();
            $appModel = $this->di->getObjectManager()->create('App\Apiconnect\Models\Apps');
            $appConfig = $appModel->getSubConfig($ced_app_shop['sAppId']);
            $targetMarketplace = $this->di->getConfig()->get('marketplace-modules')->get($appConfig['marketplace']) ?: ucfirst($appConfig['marketplace']);
            $baseClass = '\App\\' .  $targetMarketplace . '\\Api\Notification';
             // Remove marketplace subscriptions in bulk from marketplace
            if (method_exists($baseClass, 'deleteSubscriptionInBulk')) {
                $this->di->getObjectManager()->get($baseClass)->deleteSubscriptionInBulk($response['data']);
            }
            if (isset($response['success']) && $response['success']) {
                $subscriptions = $response['data'];
                $marketplaceSubscriptionIds = [];
                $subscriptionCifSKs = [];
                $subscriptionMarketplaceSKs = [];
                foreach($subscriptions as $subscription) {
                    if (in_array($subscription['_id'],$subscriptionIds)) {
                        if ($subscription['subscription_id'] != null)
                            array_push($marketplaceSubscriptionIds,$subscription['subscription_id']);
                        if ($subscription['type'] != 'app') {
                            $uniqueKey = $this->di->getObjectManager()->get('\App\Cedcommercewebapi\Components\EventSubscription')->getSubscriptionSortKey($subscription);
                            if ($subscription['type'] == "user_app") $uniqueKey = $uniqueKey . "_" . $subscription['app_code'];
                        } else {
                            $uniqueKey = $subscription['marketplace'] . "_" . $subscription['event_code'];
                        }
                        array_push($subscriptionCifSKs,$uniqueKey."_" . $subscription['_id']);
                        array_push($subscriptionMarketplaceSKs,$uniqueKey."_marketplace");
                    }
                }
                $cifCollection->deleteMany(["_id" => [ '$in' => $subscriptionIds]]);
                if (!empty($marketplaceSubscriptionIds))
                    $marketplaceCollection->deleteMany(["marketplace_subscription_id" => [ '$in' => $marketplaceSubscriptionIds]]);
                $this->di->getObjectManager()->get('\App\Cedcommercewebapi\Components\EventSubscription')->bulkDeleteSubscriptionFromDynamodb($subscriptionCifSKs,$subscriptionMarketplaceSKs);
                return ['success' => true, 'message' => "Subscription deleted successfully."];
            } else {
                return ['success' => false, 'message' => "Trouble in deleting subscriptions, No subscription found for the shop."];
            }
        } else {
            return ['success' => false, 'message' => 'subscription_ids array missing/empty or shop not set.'];
        }
    }
}
