<?php
namespace App\Arisewebapi\Components\Product;
use App\Arisewebapi\Components\Core\Common;
use IopRequest;

/**
 * Arise products handlers core logics.
 *
 * @since 1.0.0
 */
class Product extends Common {

    /**
     * Get arise products.
     *
     * @param array $params array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function getAriseProducts( $params ) {
        $params = array_merge(
            array(
                'status' => 'all',
                'offset' => '0',
                'limit'  => '1',
            ),
            $params
        );
        $request = new IopRequest( '/products/get', 'GET' );
        $request->addApiParam( 'filter', $params['status'] );
        $request->addApiParam( 'offset', $params['offset'] );
        $request->addApiParam( 'limit', $params['limit'] );
        if ( ! empty( $params['sku_list'] ) ) {
            $request->addApiParam( 'sku_seller_list', json_encode( $params['sku_list'] ) );
        }

        return [
            'success' => true,
            'data'    => $this->execute( $request ),
        ];
    }

    /**
     * Will get Item data from arise.
     *
     * @param array $params array containing the request param.
     * @since 1.0.0
     * @return array
     */
    public function getItem( $params = array() ) {
        if ( empty( $params['item_id'] ) || empty( $params['sku'] ) ) {
            return [
                'success' => false,
                'msg'     => 'Oops! Required params are missing'
            ];
        }
        $request = new IopRequest( '/product/item/get', 'GET' );
        $request->addApiParam( 'item_id', $params['item_id'] );
        $request->addApiParam( 'seller_sku', $params['sku'] );
        return [
            'success' => true,
            'data'    => $this->execute( $request )
        ];
    }

    /**
     * Will upload/update product at arise.
     *
     * @param array $params array containing the request params.
     * @param string $action
     * @since 1.0.0
     * @return array
     */
    public function uploadProductArise( $params, $action = 'insert' ) {
        if ( empty( $params['product_data'] ) ) {
            return [
                'success' => false,
                'msg'     => 'Product data is required to upload the product'
            ];
        }

        if ( 'update' === $action ) {
            $toPerform = '/product/update';
        } else {
            $toPerform = '/product/create';
        }

        $request = new IopRequest( $toPerform );
        $request->addApiParam( 'payload', json_encode( $params['product_data'] ) );
        return [
            'success' => true,
            'data'    => $this->execute( $request ),
        ];
    }

    /**
     * Will create feed on Arise ( step 1 ).
     *
     * @param array $params array containing the request params.
     * @return void
     * @since 1.0.0
     */
    public function createFeedDocumentArise( $params ) {
        $request = new IopRequest( '/feed/createFeedDocument' );
        $request->addApiParam( 'content_type', 'application/json' );

        return [
            'success' => true,
            'data'    => $this->execute( $request )
        ];
    }

    /**
     * Will upload feed data on arise at the url recieved from above steps ( step 2 ).
     *
     * @param array $params array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function putFeedDataArise( $params ) {
        if ( empty( $params['url'] ) ) {
            return [
                'success' => false,
                'msg'     => 'Feed URL is required.'
            ];
        }

        if ( empty( $params['product_data'] ) ) {
            return [
                'success' => false,
                'msg'     => 'Product Data is required.'
            ];
        }
        $dir = BP . DS . 'public' . DS . 'assets' . DS . 'arise-bulk-product' . DS . 'document' . DS;
        if ( ! is_dir( $dir ) ) {
            mkdir( $dir, 0755, true );
        }
        $fileName   = $dir . time() . '.json';
        $fileObject = fopen( $fileName, 'w' );
        fwrite( $fileObject, json_encode( $params['product_data'] ) );
        fclose( $fileObject );
        return [
            'success'   => true,
            'data'      => $this->executeRequest( $params['url'], file_get_contents( $fileName ), 'PUT' )->body ?? false,
            'file_name' => $fileName
        ];
    }

    /**
     * Will delete bulk upload document.
     *
     * @param array $params array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function deleteBulkUploadDocument( $params ) {
        try {
            if ( empty( $params['file_path'] ) ) {
                return [
                    'success' => false,
                    'msg'     => 'File Path is required to delete document.'
                ];
            }
            $dir = BP . DS . 'public' . DS . 'assets' . DS . 'arise-bulk-product' . DS . 'document' . DS;
            if ( false === strpos( $params['file_path'], $dir ) ) {
                return [
                    'success' => false,
                    'msg'     => 'Oops! Invalid file path, can not delete.'
                ];
            }
            $res = @unlink( $params['file_path'] );
            if ( ! $res ) {
                return [
                    'success' => false,
                    'msg'     => 'Unexpected error occured while deleting file.'
                ];
            }
            return [
                'success' => true,
                'msg'     => 'File deleted successfully.'
            ];
        } catch ( \Exception $e ) {
            return [
                'success' => false,
                'msg'     => $e->getMessage()
            ];
        }
    }

    /**
     * Will create feed by specifying the type on feed ID received above ( step 3 ).
     *
     * @param array $params array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function createFeedArise( $params ) {
        if ( empty( $params['feed_document_id'] ) ) {
            return [
                'success' => false,
                'msg'     => 'Feed Document ID is required.'
            ];
        }

        $params = array_merge(
            array(
                'feed_type' => 'PRODUCT_LISTING'
            ),
            $params
        );
        $request = new IopRequest( '/feed/createFeed' );
        $request->addApiParam( 'feed_type', $params['feed_type'] );
        $request->addApiParam( 'feed_document_id', $params['feed_document_id'] );

        return [
            'success' => true,
            'data'    => $this->execute( $request )
        ];
    }

    /**
     * Will get Feed status from Arise ( setp 4 ).
     *
     * @param array $params array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function getFeedArise( $params ) {
        if ( empty( $params['feed_id'] ) ) {
            return [
                'success' => false,
                'msg'     => 'Feed ID is required to get feed status'
            ];
        }

        $request = new IopRequest( '/feed/getFeed', 'GET' );
        $request->addApiParam( 'feed_id', $params['feed_id'] );

        return [
            'success' => true,
            'data'    => $this->execute( $request )
        ];
    }

    /**
     * Get feed document response ( step 5 ).
     *
     * @param array $params array containing the feed document response.
     * @since 1.0.0
     * @return array
     */
    public function getFeedDocumentArise( $params ) {
        if ( empty( $params['feed_document_id'] ) ) {
            return [
                'success' => false,
                'msg'     => 'Feed document id is required to get document'
            ];
        }

        $request = new IopRequest( '/feed/getFeedDocument', 'GET' );
        $request->addApiParam( 'feed_document_id', $params['feed_document_id'] );

        return [
            'success' => true,
            'data'    => $this->execute( $request )
        ];
    }

    /**
     * Get feed result document data ( step 6 ).
     *
     * @param array $params array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function getFeedDocumentDataArise( $params ) {
        if ( empty( $params['url'] ) ) {
            return [
                'success' => false,
                'msg'     => 'Feed response result url is required'
            ];
        }

        return [
            'success' => true,
            'data'    => $this->executeRequest( $params['url'] )->body
        ];
    }

    /**
     * Will delete products from Arise.
     *
     * @param array $params array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function deleteProductArise( $params ) {
        if ( empty( $params['skus'] ) ) {
            return [
                'success' => false,
                'msg'     => 'Please send skus to delete'
            ];
        }
        $request = new IopRequest( '/product/remove' );
        $request->addApiParam( 'seller_sku_list', json_encode( $params['skus'] ) );
    
        return [
            'success' => true,
            'data'    => $this->execute( $request )
        ];
    }

    /**
     * Will update Inventory and/or Price at Arise.
     *
     * @param array $params array containing the params.
     * @since 1.0.0
     * @return array
     */
    public function updateInventoryPriceArise( $params ) {
        if ( empty( $params['product_data'] ) ) {
            return [
                'success' => false,
                'msg'     => 'Product data is required to update quantity and/or price'
            ];
        }
        $request = new IopRequest('/product/price_quantity/update');
        $request->addApiParam( 'payload', json_encode( $params['product_data'] ) );

        return [
            'success' => true,
            'data'    => $this->execute( $request )
        ];
    }

    /**
     * Will deactivate product on remote.
     *
     * @param array $params array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function deactivateProduct( $params = array() ) {
        $request  = new IopRequest( '/product/deactivate' );
        $subDatas = '';
        if ( ! empty( $params['item_ids'] ) && is_array( $params['item_ids'] ) ) {
            $itemIds = $params['item_ids'];
            foreach ( $itemIds as $itemId ) {
                $subDatas .= "<ItemId>{$itemId}</ItemId>"; 
            }
        }
        if ( ! empty( $params['skus'] ) && is_array( $params['skus'] ) ) {
            $skus = $params['skus'];
            $subDatas .= '<Skus>';
            foreach ( $skus as $sku ) {
                $subDatas .= "<SellerSku>{$sku}</SellerSku>";
            }
            $subDatas .= '</Skus>';
        }
        $payload = "<Request>
                        <Product>
                            {$subDatas}
                        </Product>
                    </Request>";
        $request->addApiParam( 'payload', $payload );
        return [
            'success' => true,
            'data'    => $this->execute( $request )
        ];
    }

    /**
     * Will activate product on remote.
     *
     * @param array $params array containing the request params.
     * @since 1.0.0
     * @return array
     */
    public function activateProduct( $params = array() ) {
        if ( empty( $params['skus'] ) || ! is_array( $params['skus'] ) || empty( $params['item_id'] ) ) {
            return [
                'success' => false,
                'msg'     => 'SKUs and ItemID are required to activate products.'
            ];
        }
        $request  = new IopRequest( '/product/update' );
        $subDatas = [];
        if ( ! empty( $params['skus'] ) && is_array( $params['skus'] ) ) {
            $skus      = $params['skus'];
            foreach ( $skus as $sku ) {
                $subDatas[] = [
                    'SellerSku' => $sku,
                    'Status'    => 'active'
                ];
            }
        }
        $payload = [
            'Request' => [
                'Product' => [
                    'ItemId'     => $params['item_id'],
                    'Attributes' => '{}',
                    'Skus'       => [
                        'Sku' => $subDatas
                    ]
                ]
            ]
        ];
        $request->addApiParam( 'payload', json_encode( $payload ) );
        return [
            'success' => true,
            'data'    => $this->execute( $request )
        ];
    }
}